/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.core;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.solr.common.SolrException;
import org.apache.solr.common.cloud.ZkMaintenanceUtils;
import org.apache.solr.common.util.Utils;
import org.apache.solr.core.ConfigSetService;
import org.apache.solr.core.CoreContainer;
import org.apache.solr.core.CoreDescriptor;
import org.apache.solr.core.SolrConfig;
import org.apache.solr.core.SolrResourceLoader;
import org.apache.solr.util.FileTypeMagicUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemConfigSetService
extends ConfigSetService {
    private static final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    public static final String METADATA_FILE = ".metadata.json";
    private final Path configSetBase;

    public FileSystemConfigSetService(CoreContainer cc) {
        super(cc.getResourceLoader(), cc.getConfig().hasSchemaCache());
        this.configSetBase = cc.getConfig().getConfigSetBaseDirectory();
    }

    protected FileSystemConfigSetService(Path configSetBase) {
        super(null, false);
        this.configSetBase = configSetBase;
    }

    @Override
    public SolrResourceLoader createCoreResourceLoader(CoreDescriptor cd) {
        Path instanceDir = this.locateInstanceDir(cd);
        SolrResourceLoader solrResourceLoader = new SolrResourceLoader(instanceDir, this.parentLoader.getClassLoader());
        return solrResourceLoader;
    }

    @Override
    public String configSetName(CoreDescriptor cd) {
        return (cd.getConfigSet() == null ? "instancedir " : "configset ") + this.locateInstanceDir(cd);
    }

    @Override
    public boolean checkConfigExists(String configName) throws IOException {
        return Files.exists(this.getConfigDir(configName), new LinkOption[0]);
    }

    @Override
    public void deleteConfig(String configName) throws IOException {
        this.deleteDir(this.getConfigDir(configName));
    }

    @Override
    public void deleteFilesFromConfig(String configName, List<String> filesToDelete) throws IOException {
        Path configDir = this.getConfigDir(configName);
        Objects.requireNonNull(filesToDelete);
        for (String fileName : filesToDelete) {
            Path file = configDir.resolve(this.normalizePathToOsSeparator(fileName));
            if (!Files.exists(file, new LinkOption[0])) continue;
            if (Files.isDirectory(file, new LinkOption[0])) {
                this.deleteDir(file);
                continue;
            }
            Files.delete(file);
        }
    }

    @Override
    public void copyConfig(String fromConfig, String toConfig) throws IOException {
        Path source = this.getConfigDir(fromConfig);
        Path dest = this.getConfigDir(toConfig);
        this.copyRecursively(source, dest);
    }

    private void deleteDir(Path dir) throws IOException {
        try {
            Files.walkFileTree(dir, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path path, BasicFileAttributes attrs) throws IOException {
                    Files.delete(path);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException ioException) throws IOException {
                    Files.delete(dir);
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        catch (NoSuchFileException noSuchFileException) {
            // empty catch block
        }
    }

    @Override
    protected void uploadConfig(String configName, Path source) throws IOException {
        Path dest = this.getConfigDir(configName);
        this.copyRecursively(source, dest);
    }

    @Override
    public void uploadFileToConfig(String configName, String fileName, byte[] data, boolean overwriteOnExists) throws IOException {
        if (ZkMaintenanceUtils.isFileForbiddenInConfigSets((String)fileName)) {
            log.warn("Not including uploading file to config, as it is a forbidden type: {}", (Object)fileName);
        } else if (!FileTypeMagicUtil.isFileForbiddenInConfigset(data)) {
            Path filePath = this.getConfigDir(configName).resolve(this.normalizePathToOsSeparator(fileName));
            if (!Files.exists(filePath, new LinkOption[0]) || overwriteOnExists) {
                Files.write(filePath, data, new OpenOption[0]);
            }
        } else {
            String mimeType = FileTypeMagicUtil.INSTANCE.guessMimeType(data);
            log.warn("Not including uploading file {}, as it matched the MAGIC signature of a forbidden mime type {}", (Object)fileName, (Object)mimeType);
        }
    }

    @Override
    public void setConfigMetadata(String configName, Map<String, Object> data) throws IOException {
        Path configDir = this.getConfigDir(configName);
        if (!Files.exists(configDir, new LinkOption[0])) {
            Files.createDirectory(configDir, new FileAttribute[0]);
        }
        Path metadataPath = configDir.resolve(METADATA_FILE);
        Files.write(metadataPath, Utils.toJSON(data), new OpenOption[0]);
    }

    @Override
    public Map<String, Object> getConfigMetadata(String configName) throws IOException {
        Path metadataPath = this.getConfigDir(configName).resolve(METADATA_FILE);
        byte[] data = null;
        try {
            data = Files.readAllBytes(metadataPath);
        }
        catch (NoSuchFileException e) {
            return Collections.emptyMap();
        }
        Map metadata = (Map)Utils.fromJSON((byte[])data);
        return metadata;
    }

    @Override
    public void downloadConfig(String configName, Path dest) throws IOException {
        Path source = this.getConfigDir(configName);
        this.copyRecursively(source, dest);
    }

    private void copyRecursively(final Path source, final Path target) throws IOException {
        try {
            Files.walkFileTree(source, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    Files.createDirectories(target.resolve(source.relativize(dir).toString()), new FileAttribute[0]);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    if (ZkMaintenanceUtils.isFileForbiddenInConfigSets((String)file.getFileName().toString())) {
                        log.warn("Not including uploading file to config, as it is a forbidden type: {}", (Object)file.getFileName());
                    } else if (!FileTypeMagicUtil.isFileForbiddenInConfigset(file)) {
                        Files.copy(file, target.resolve(source.relativize(file).toString()), StandardCopyOption.REPLACE_EXISTING);
                    } else {
                        String mimeType = FileTypeMagicUtil.INSTANCE.guessMimeType(file);
                        log.warn("Not copying file {}, as it matched the MAGIC signature of a forbidden mime type {}", (Object)file.getFileName(), (Object)mimeType);
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        catch (NoSuchFileException noSuchFileException) {
            // empty catch block
        }
    }

    @Override
    public List<String> listConfigs() throws IOException {
        try (Stream<Path> configs = Files.list(this.configSetBase);){
            List<String> list = configs.map(Path::getFileName).map(Path::toString).sorted().collect(Collectors.toList());
            return list;
        }
    }

    @Override
    public byte[] downloadFileFromConfig(String configName, String fileName) throws IOException {
        Path filePath = this.getConfigDir(configName).resolve(this.normalizePathToOsSeparator(fileName));
        byte[] data = null;
        try {
            data = Files.readAllBytes(filePath);
        }
        catch (NoSuchFileException noSuchFileException) {
            // empty catch block
        }
        return data;
    }

    @Override
    public List<String> getAllConfigFiles(String configName) throws IOException {
        final Path configDir = this.getConfigDir(configName);
        final ArrayList<String> filePaths = new ArrayList<String>();
        Files.walkFileTree(configDir, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                if (!Files.isHidden(file) && !FileSystemConfigSetService.METADATA_FILE.equals(file.getFileName().toString())) {
                    filePaths.add(FileSystemConfigSetService.this.normalizePathToForwardSlash(configDir.relativize(file).toString()));
                    return FileVisitResult.CONTINUE;
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException ioException) {
                String relativePath = configDir.relativize(dir).toString();
                if (!relativePath.isEmpty()) {
                    filePaths.add(relativePath + "/");
                }
                return FileVisitResult.CONTINUE;
            }
        });
        Collections.sort(filePaths);
        return filePaths;
    }

    private String normalizePathToForwardSlash(String path) {
        return path.replace(this.configSetBase.getFileSystem().getSeparator(), "/");
    }

    private String normalizePathToOsSeparator(String path) {
        return path.replace("/", this.configSetBase.getFileSystem().getSeparator());
    }

    protected Path locateInstanceDir(CoreDescriptor cd) {
        String configSet = cd.getConfigSet();
        if (configSet == null) {
            return cd.getInstanceDir();
        }
        Path configSetDirectory = this.configSetBase.resolve(configSet);
        if (!Files.isDirectory(configSetDirectory, new LinkOption[0])) {
            throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "Could not load configuration from directory " + configSetDirectory);
        }
        return configSetDirectory;
    }

    @Override
    public Long getCurrentSchemaModificationVersion(String configSet, SolrConfig solrConfig, String schemaFileName) {
        Path schemaFile = solrConfig.getResourceLoader().getConfigPath().resolve(schemaFileName);
        try {
            return Files.getLastModifiedTime(schemaFile, new LinkOption[0]).toMillis();
        }
        catch (FileNotFoundException e) {
            return null;
        }
        catch (IOException e) {
            log.warn("Unexpected exception when getting modification time of {}", (Object)schemaFile, (Object)e);
            return null;
        }
    }

    protected Path getConfigDir(String configName) throws IOException {
        Path path = this.configSetBase.resolve(configName).normalize();
        if (!path.startsWith(this.configSetBase)) {
            throw new IOException("configName=" + configName + " is not found under configSetBase dir");
        }
        return path;
    }
}

