/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2001-2003 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package coldfusion.mail.mod;

import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.IDN;
import java.net.URL;
import java.security.AccessController;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.PrivateKey;
import java.security.PrivilegedAction;
import java.security.Security;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.activation.FileDataSource;
import javax.activation.URLDataSource;
import javax.mail.BodyPart;
import javax.mail.Header;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.internet.AddressException;
import javax.mail.internet.ContentType;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.mail.internet.MimeUtility;
import jakarta.servlet.jsp.PageContext;

import org.apache.commons.vfs2.FileObject;
import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.cms.AttributeTable;
import org.bouncycastle.asn1.cms.IssuerAndSerialNumber;
import org.bouncycastle.asn1.smime.SMIMECapabilitiesAttribute;
import org.bouncycastle.asn1.smime.SMIMECapability;
import org.bouncycastle.asn1.smime.SMIMECapabilityVector;
import org.bouncycastle.asn1.smime.SMIMEEncryptionKeyPreferenceAttribute;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.cert.jcajce.JcaCertStore;
import org.bouncycastle.cms.CMSAlgorithm;
import org.bouncycastle.cms.SignerInfoGenerator;
import org.bouncycastle.cms.jcajce.JcaSimpleSignerInfoGeneratorBuilder;
import org.bouncycastle.cms.jcajce.JceCMSContentEncryptorBuilder;
import org.bouncycastle.cms.jcajce.JceKeyTransRecipientInfoGenerator;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.mail.smime.SMIMEEnvelopedGenerator;
import org.bouncycastle.mail.smime.SMIMESignedGenerator;
import org.bouncycastle.operator.OutputEncryptor;

import com.ibm.icu.text.IDNA;
import com.ibm.icu.text.IDNA.Info;
import com.sun.mail.smtp.SMTPMessage;

import coldfusion.log.Logger;
import coldfusion.runtime.ApplicationException;
import coldfusion.runtime.RuntimeServiceImpl;
import coldfusion.runtime.StringFunc;
import coldfusion.server.ServiceFactory;
import coldfusion.server.StoreService;
import coldfusion.util.PasswordUtils;
import coldfusion.util.RB;
import coldfusion.util.Utils;
import coldfusion.vfs.VFile;
import coldfusion.vfs.VFileDataSource;
//import coldfusion.vfs.s3.S3FileObject;
import coldfusion.mail.core.*;
import static coldfusion.mail.core.MailExceptions.*;
import sun.security.pkcs.PKCS7;

/**
 * Class that encapsulates a piece of Mail
 *
 * @author Clement Wong
 * @author Tom Jordahl
 */
public class MailImpl
{

    private static final String BC = "BC";

    // private static final String AES256_WRAP = "AES256_WRAP";

    // private static final String AES128_WRAP = "AES128_WRAP";

    private static final String AES256_CBC = "AES256_CBC";

    private static final String AES192_CBC = "AES192_CBC";

    private static final String RC2_CBC = "RC2_CBC";

    // private static final String DES_EDE3_WRAP = "DES_EDE3_WRAP";

    private static final String DES_EDE3_CBC = "DES_EDE3_CBC";

    private static final String AES128_CBC = "AES128_CBC";

    protected HostImpl[] source;

    protected String mimetype = "text/plain";
    protected String charset = null;
    protected String contentType = null;
    protected InternetAddress sender = null;
    protected InternetAddress[] replyTo = null;
    protected InternetAddress[] recipient = null;
    protected InternetAddress[] cc = null;
    protected InternetAddress[] bcc = null;
    protected String failTo = null;
    protected String subject = null;

    protected String username = null;
    protected String password = null;
    protected String message_id = null;
    protected int timeout = -1;
    protected int port = -1;
    protected Boolean useTLS = null;
    protected Boolean useSSL = null;

    protected Vector bodyparts = new Vector();
    protected Vector attachments = new Vector();
    protected ArrayList headers = new ArrayList();
    protected Boolean spoolEnable = null;

    protected int wrapText = -1;

    protected boolean debug = false;
    private boolean multipart = false;  // more than 1 text part?
    private Map removeAttachment = Collections.synchronizedMap(new HashMap()); // remove attachment when email has been sent

    protected Boolean sign = null;
    protected Boolean encrypt = null;
    protected String recipientCert = null;
    protected String encryptionAlgorithm = null;
    protected String keystoreFile = null;
    protected String keyalias = null;
    protected String keystorePassword = null;
    protected String keyPassword = null;
    private static final String JKS_TYPE = "jks";
    private static final String PKCS12_TYPE = "pkcs12";
    private static IDNA idn = IDNA.getUTS46Instance(124);
    private static SMIMECapabilityVector capabilities = new SMIMECapabilityVector();
    static
    {
        Security.addProvider(new BouncyCastleProvider());
        capabilities.addCapability(SMIMECapability.dES_EDE3_CBC);
        capabilities.addCapability(SMIMECapability.rC2_CBC, 128);
        capabilities.addCapability(SMIMECapability.dES_CBC);
    }

    public boolean isRemoveAttachment(String filepath)
    {
        if (removeAttachment.containsKey(filepath))
        {
            Boolean b= (Boolean)removeAttachment.get(filepath);
            return b==null || b.booleanValue();
        }
        return false;
    }

    public boolean isVarAttachment(String filepath)
    {
        return removeAttachment.get(filepath)==null;
    }

    public void addRemoveAttachment(String filepath,Boolean remove)
    {
        if(remove==null)
            remove= Boolean.FALSE;
        if(!removeAttachment.containsKey(filepath))
            removeAttachment.put(filepath,remove);
    }

    /**
     * Special case for implicit vars (Need to be removed evne with sandbox enabled)
     * @param filepath
     */
    public void addRemoveAttachment(String filepath)
    {
        if(!removeAttachment.containsKey(filepath))
            removeAttachment.put(filepath,null);
    }

    /**
     * Set the Content-type of the message.
     *
     * The member variables set by this function are: charset, mimeType, and contentType.
     *
     * @param typeString a Content-type string, possibly including a ";charset=" clause
     * @param cs a charset or null
     */
    public void setContentType(String typeString, String cs)
    {
        String csParam = null;
        try
        {
            ContentType ct = new ContentType(typeString);
            mimetype = ct.getBaseType();
            csParam = ct.getParameter("charset");
            // Use explicit charset if provided and none in the typeString
            if (csParam == null && cs != null)
                ct.setParameter("charset", cs);
            contentType = ct.toString();
        }
        catch (javax.mail.internet.ParseException e)
        {
            // Use fallback defaults
            mimetype = "text/plain";
            contentType = "text/plain";
        }

        // Use charset from typeString, then charset attribute
        if (csParam != null)
            charset = csParam;
        else if (cs != null)
            charset = cs;

    }   // setContentType

    // get method for spoolMgr
    public String getContentType()
    {
        return contentType;
    }

    public HostImpl[] getServers()
    {
        return source;
    }

    /**
     * Clears the existing server list and sets the input as the new list.
     * @param list
     */
    public void setServers(HostImpl[] list)
    {
        source = list;
    }

    /**
     * Set port, timeout, username and password from the message
     * on each of the servers for this messages if not set.
     */
    public void validateServers()
    {
        for (int i = 0; i < source.length; i++)
        {
            HostImpl host = source[i];
            if (username != null && host.getUsername() == null &&
                password != null && host.getPassword() == null)
            {
                host.setUsername(username);
                host.setPassword(password);
            }
            if (timeout != -1 && host.getTimeout() == -1)
                host.setTimeout(timeout);
            if (port != -1 && host.getPort() == -1)
                host.setPort(port);
            if (useTLS != null)
                host.setUseTLS(useTLS.booleanValue());
            if (useSSL!= null)
                host.setUseSSL(useSSL.booleanValue());
        }
    }

    public void setSender(InternetAddress s)
    {
        sender = encodePersonalInfo(s);
    }

    // get method for spoolMgr
    public InternetAddress getSender()
    {
        return sender;
    }

    public void setRecipient(InternetAddress[] r)
    {
        recipient = encodePersonalInfo(r);
    }

    public void setSpoolEnable(boolean bEnable)
    {
        spoolEnable = Boolean.valueOf(bEnable);
    }

    /**
     * @return Boolean or null if we should use admin default
     */
    public Boolean getSpoolEnable()
    {
        return spoolEnable;
    }

    /**
     * @return Boolean or null if we should use admin default
     */
    public Boolean getUseTLS()
    {
        return useTLS;
    }

    public void setUseTLS(boolean useTLS)
    {
        this.useTLS = Boolean.valueOf(useTLS);
    }

    /**
     * @return Boolean or null if we should use admin default
     */
    public Boolean getUseSSL()
    {
        return useSSL;
    }

    public void setUseSSL(boolean useSSL)
    {
        this.useSSL = Boolean.valueOf(useSSL);
    }

    public void setWrapText(int limit)
    {
        wrapText = limit;
    }

    // get method for spoolMgr
    public InternetAddress[] getRecipient()
    {
        return recipient;
    }

    public void setCc(InternetAddress[] r)
    {
        cc = encodePersonalInfo(r);
    }

    //get method for spoolMgr
    public InternetAddress[] getCc()
    {
        return cc;
    }


    public void setBcc(InternetAddress[] r)
    {
        bcc = encodePersonalInfo(r);
    }

    //get method for spoolMgr
    public InternetAddress[] getBcc()
    {
        return bcc;
    }

    public void setReplyTo(InternetAddress[] addr)
    {
        replyTo = encodePersonalInfo(addr);
    }

    // get method for spoolMgr
    public InternetAddress[] getReplyTo()
    {
        return replyTo;
    }

    public void setFailTo(String s)
    {
        failTo = s;
    }

    // get method for spoolMgr
    public String getFailTo()
    {
        return failTo;
    }

    public void setSubject(String s)
    {
        subject = s;
    }

    //get method for spoolMgr
    public String getSubject()
    {
        return subject;
    }

    /**
     * Set the 'default' username for this message
     */
    public void setUsername(String s)
    {
        username = s;
    }

    /**
     * The 'default' username for this message.
     */
    public String getUsername()
    {
        return username;
    }

    /**
     * Set the 'default' password for this message
     */
    public void setPassword(String p)
    {
        password = p;
    }

    /**
     * Set the 'default' password after decrypting for this message
     * This call will come from Alert, as monitoring service stores 
     * it in encrypted form.
     */
    public void setPassword(String p, boolean encrypted)
    {
        password = p;
        if(encrypted)
        {
            String seed = ((MailSpooler)ServiceFactory.getMailSpoolService()).getSeed();
            password =  PasswordUtils.decryptPassword(password, seed);
        }
    }
    
    /**
     * The 'default' password for this message.
     */
    public String getPassword()
    {
        return password;
    }

    /**
     * The timeout for this message.
     * @return timeout, or -1 if none.
     */
    public int getTimeout()
    {
        return timeout;
    }

    /**
     * Set the timeout for this message.
     */
    public void setTimeout(int timeout)
    {
        this.timeout = timeout;
    }

    /**
     * The 'default' port for this message.
     * @return port number or -1 if none
     */
    public int getPort()
    {
        return port;
    }

    public void setMessage_id(String message_id)
    {
        this.message_id = message_id;
    }


    /**
     * Set the 'default' port for this message.
     */
    public void setPort(int port)
    {
        this.port = port;
    }

    public void setDebug(boolean d)
    {
        debug = d;
    }

    public boolean isDebug()
    {
        return debug;
    }

    public boolean isMultipart()
    {
        return multipart;
    }

    public void setMultipart(boolean multipart)
    {
        this.multipart = multipart;
    }
    
    public static InternetAddress[] setInternetAddress(String addr)
    {
        InternetAddress[] rs;
        try
        {
            rs = InternetAddress.parse(addr, false);
        }
        catch (AddressException ex)
        {
            rs = null;
        }
        return rs;
    }

    /**
     * Convert an string to an array of InternetAddreses.
     * @param addr string to convert
     * @return an array of InternetAddress, or null if parse error
     */
    public static InternetAddress[] setInternetAddress(String addr, int idnaVersion)
    {
        InternetAddress[] rs;
        try
        {
            String idnEmail = toIDNEmail(addr, idnaVersion);
            rs = InternetAddress.parse(idnEmail, false);
        }
        catch (AddressException ex)
        {
            rs = null;
        }
        return rs;
    }
    
    private static String toIDNEmail(String address, int idnaVersion) {
        if(address != null) {
            if(StringFunc.isAllASCII(address))
                return address;
            String[] emailAddresses = address.split(",");
            StringBuilder builder = new StringBuilder();
            for(int i = 0; i < emailAddresses.length; i++) {
                   String email = emailAddresses[i];
                   int emailStartIndex = email.lastIndexOf("<");
                   if(emailStartIndex > -1) {
                         int emailEndIndex = email.indexOf(">", emailStartIndex);
                         if(emailEndIndex > -1) {
                                String personalName = email.substring(0, emailStartIndex);
                                String emailAddress = _toIDNEmail(email.substring(emailStartIndex + 1, emailEndIndex), idnaVersion);
                                builder.append(personalName).append("<").append(emailAddress).append(">").append(",");
                                continue;
                         }
                   }
                   builder.append(_toIDNEmail(email, idnaVersion));
                   if(i + 1 != emailAddresses.length) {
                       builder.append(",");
                   }
            }
            return builder.toString();
        }
        return null;
    }
    
    private static String _toIDNEmail(String email, int idnaVersion) {
        int idx = email.indexOf("@");
        if(idx != -1 && idx < email.length() - 1) {
            String local = email.substring(0, idx);
            String domain = email.substring(idx + 1);
            return local + "@" + (idnaVersion == 2003 ? _toIDNA2003(domain) : _toIDNA2008(domain));
        }
        return email;        
    }
    
    private static String _toIDNA2003(String email) {
        return IDN.toASCII(email);        
    }
    
    private static String _toIDNA2008(String email) {
        StringBuilder builder = new StringBuilder();
        IDNA.Info info = new Info();
        StringBuilder idnEmail = idn.nameToASCII(email, builder, info);
        if(!info.hasErrors()) return idnEmail.toString();
        return email;        
    }

    public void setEncrypt(Boolean encrypt)
    {
        this.encrypt = encrypt;
    }

    public Boolean getEncrypt()
    {
        return encrypt;
    }

    public void setRecipientCert(String recipientCert)
    {
        this.recipientCert = recipientCert;
    }

    public String getecipientCert()
    {
        return recipientCert;
    }

    public void setEncryptionAlgorithm(String encryptionAlgorithm)
    {
        this.encryptionAlgorithm = encryptionAlgorithm;
    }

    public String getEncryptionAlgorithm()
    {
        return encryptionAlgorithm;
    }

    public void setSign(Boolean sign)
    {
        this.sign = sign;
    }

    public Boolean getSign()
    {
        return sign;
    }

    public void setKeystoreFile(String keystoreFile)
    {
        this.keystoreFile = keystoreFile;
    }

    public String getKeystoreFile()
    {
        return keystoreFile;
    }

    public void setKeyalias(String keyalias)
    {
        this.keyalias = keyalias;
    }

    public String getKeyalias()
    {
        return keyalias;
    }

    public void setKeystorePassword(String keystorePassword)
    {
        this.keystorePassword = keystorePassword;
    }

    public String getKeystorePassword()
    {
        return keystorePassword;
    }
    
    public void setKeyPassword(String keyPassword)
    {
        this.keyPassword = keyPassword;
    }

    public String getKeyPassword()
    {
        return keyPassword;
    }

    /**
     * Put together the Message and return it for delivery
     *
     * @param session a JavaMail session required to construct a Message
     * @return a JavaMail Message
     * @throws MessagingException
     */
    public Message createMessage(Session session) throws MessagingException
    {
        if (charset == null && contentType == null)
        {
            charset = RuntimeServiceImpl.getDefaultMailCharset();
            contentType = "text/plain; charset=" + charset;
        }

        MimeMessage message = new SMTPMessage(session);
        message.setFrom(sender);
        if (replyTo != null)
        {
            message.setReplyTo(replyTo);
        }
        if (failTo != null)
        {
            ((SMTPMessage)message).setEnvelopeFrom(failTo);
        }
        if (recipient != null)
        {
            message.setRecipients(Message.RecipientType.TO, recipient);
        }
        if (cc != null)
        {
            message.setRecipients(Message.RecipientType.CC, cc);
        }
        if (bcc != null)
        {
            message.setRecipients(Message.RecipientType.BCC, bcc);
        }
        if (subject != null)
        {
            try
            {
                message.setSubject(encodeIfJapanese(subject), charset);
            }
            catch (MessagingException e)
            {
                // This can NOT be good, but skip it and go without a subject
                e.printStackTrace();
            }
        }
        message.setSentDate(new Date());

        Header contentTransferEncoding = null;
        // Add headers to message
        for (int i = 0; i < headers.size(); i++)
        {
            Header header = (Header) headers.get(i);
            message.addHeader(header.getName(), header.getValue());
            // Remember this for later
            if (header.getName().equalsIgnoreCase("Content-Transfer-Encoding"))
                contentTransferEncoding = header;
        }

        // Stuff for handling multipart text content:
        // - If we have a multipart and no attachments, we want the type to be mulitpart/alternative
        // - If we have a multipart with attachments, we have a multipart/mixed,
        //   with a part that is a multipart/alternative
        // - If we have a single text part, use contentType (set/figured out earlier)
        MimeMultipart mixed = null;        
        List<BodyPart> attachmentParts = new ArrayList<>();
        
        if (bodyparts.size() > 1)
        {
            MimeMultipart alternative = new MimeMultipart("alternative");            
            List<BodyPart> inlineBodyParts = new ArrayList<>();
            // Add attachements, if any
            if (attachments.size() > 0)
            {
                // Create a multipart/mixed message to hold the
                // multipart/alternative messages AND the attachements.
                mixed = new MimeMultipart("mixed");

                // Put the alternative message inside a new BodyPart
                MimeBodyPart alternatvePart = new MimeBodyPart();
                alternatvePart.setContent(alternative);
                mixed.addBodyPart(alternatvePart);

                // Now add attachements to the mixed message
                for (int i = 0; i < attachments.size(); i += 1)
                {
                    MimeBodyPart mimeBodyPart = (MimeBodyPart) attachments.elementAt(i);
                    // Set the content type to multipart/related if one (or all)
                    // of the attachments has a content ID.
                    if (mimeBodyPart.getContentID() != null)
                        inlineBodyParts.add(mimeBodyPart);
                    else
                        attachmentParts.add(mimeBodyPart);
                }
            }
            
            for (int i = 0; i < bodyparts.size(); i++)
            {
                BodyPart bodyPart = (BodyPart) bodyparts.elementAt(i);
                String[] header = bodyPart.getHeader("Content-Type");
                if(inlineBodyParts.size() > 0 && header != null && header.length > 0) {
                    ContentType partContentType = new ContentType(header[0]);
                    if(partContentType.match("text/html")){
                        MimeMultipart related = new MimeMultipart("related");
                        related.addBodyPart(bodyPart);
                        bodyPart = new MimeBodyPart();
                        for(BodyPart inlineBodyPart : inlineBodyParts){
                            related.addBodyPart(inlineBodyPart);
                        }
                        bodyPart.setContent(related); 
                    }
                } 
                alternative.addBodyPart(bodyPart);
            }

            // Set up the message
            if(mixed == null)
                message.setContent(alternative);

        }
        else if ((bodyparts.size() == 1) && (attachments.size() > 0))
        {
            // we have a single part with attachements
            // Create a multipart/mixed message to hold the
            // message text and the attachements.
            mixed = new MimeMultipart("mixed");

            List<BodyPart> inlineBodyParts = new ArrayList<>();
            // Now add attachements to the mixed message
            for (int i = 0; i < attachments.size(); i++)
            {
                MimeBodyPart mimeBodyPart = (MimeBodyPart) attachments.elementAt(i);
                // Set the content type to multipart/related if one (or all)
                // of the attachments has a content ID.
                if (mimeBodyPart.getContentID() != null)
                    inlineBodyParts.add(mimeBodyPart);
                else
                    attachmentParts.add(mimeBodyPart);
            }
            if(!inlineBodyParts.isEmpty()) {
                MimeMultipart related = new MimeMultipart("related");
                MimeBodyPart relatedPart = new MimeBodyPart();
                related.addBodyPart((BodyPart) bodyparts.elementAt(0));
                for(BodyPart bodyPart : inlineBodyParts){
                    related.addBodyPart(bodyPart);
                }
                relatedPart.setContent(related);
                mixed.addBodyPart(relatedPart);
            } else {
                // Add the single message part
                mixed.addBodyPart((BodyPart) bodyparts.elementAt(0));
            }           
        }
        else
        {
            if (bodyparts.size() != 1)
            {
                // no body parts, something is wrong
                throw new MessagingException("missing body for message");
            }

            try
            {  // text type, use it with just the body
                BodyPart bodyPart = (BodyPart) bodyparts.elementAt(0);
                String cType = bodyPart.getContentType();
                if(cType == null || cType.trim().isEmpty())
                    cType = contentType;
                message.setContent(bodyPart.getContent(), cType);
                // calling setContext will reset the Content-Transfer-Encoding header
                // Set it again if needed - see bug 69914
                if (contentTransferEncoding != null)
                    message.setHeader("Content-Transfer-Encoding", contentTransferEncoding.getValue());

            }
            catch (Exception e)
            {
                throw new MessagingException(e.getMessage());
            }
        }        
        
        if(mixed != null) {
            if(attachmentParts.size() > 0) {
                for(BodyPart bodyPart: attachmentParts)
                    mixed.addBodyPart(bodyPart);
            }
            message.setContent(mixed);
        }

        // Finalize the message
        message.saveChanges();
        if (sign != null && sign.booleanValue())
        {
           message = signMail(message, session);
        }
        
        // encrypt email
        if (encrypt != null && encrypt.booleanValue())
        {
            if (recipientCert == null || (recipientCert != null && recipientCert.length() <= 0))
            {
                throw new RecipientCertificateException();
            }

            // encrypt the email
            message = encryptMail(message, recipientCert, encryptionAlgorithm);
        }

        // saveChanges method overwrites Message-ID header to default one. To pass user defined value of 
        // Message-ID header, we have to set this header here only. 
        if(message_id != null)
        {
            message.setHeader("Message-ID", message_id);
        }
        return message;
    }

    /**
     * Encrypt the email.
     * 
     * @param message
     * @param recipientCertificate
     * @param recipientCertificate
     * @return
     */
    private MimeMessage encryptMail(MimeMessage message, String recipientCertificate, String algorithm)
    {
        try
        {
            // choose correct algorithm
            ASN1ObjectIdentifier algo = setupAlgoForEncryption(algorithm);

            // load public key certificate.
            X509Certificate[] certs = loadPublicCertificate(recipientCertificate);
            if (certs == null)
            {
                throw new UnrecoverablePublicKeyException(RB.getString(this, "PublicKeyError"));
            }

            // create encryption generator
            SMIMEEnvelopedGenerator genEnc = new SMIMEEnvelopedGenerator();
            genEnc.addRecipientInfoGenerator(new JceKeyTransRecipientInfoGenerator(certs[0]).setProvider("BC"));
            OutputEncryptor encryptor = new JceCMSContentEncryptorBuilder(algo).setProvider("BC").build();
            MimeBodyPart mp = genEnc.generate(message,encryptor);

            MimeMessage encryptedMessage = new MimeMessage(message);
            encryptedMessage.setContent(mp.getContent(), mp.getContentType());
            encryptedMessage.saveChanges();
            return encryptedMessage;
        }
        catch (ApplicationException ae)
        {
            throw ae;
        }
        catch (Exception e)
        {
            throw new MailEncryptionException(e);
        }
    }

    /**
     * @param algorithm
     * @return
     */
    private ASN1ObjectIdentifier setupAlgoForEncryption(String algorithm)
    {
        ASN1ObjectIdentifier algo = null;
        if(algorithm != null && algorithm.length() > 0) 
        {
            if (algorithm.equalsIgnoreCase(DES_EDE3_CBC))
            {
                algo = CMSAlgorithm.DES_EDE3_CBC;
            }
            else if (algorithm.equalsIgnoreCase(RC2_CBC))
            {
                algo = CMSAlgorithm.RC2_CBC;
            }
            else if (algorithm.equalsIgnoreCase(AES128_CBC))
            {
                algo = CMSAlgorithm.AES128_CBC;
            }
            else if (algorithm.equalsIgnoreCase(AES192_CBC))
            {
                algo = CMSAlgorithm.AES192_CBC;
            }
            else if (algorithm.equalsIgnoreCase(AES256_CBC))
            {
                algo = CMSAlgorithm.AES256_CBC;
            }
            // else if (algorithm.equalsIgnoreCase(AES128_WRAP))
            // {
            // algorithm = SMIMEEnvelopedGenerator.AES128_WRAP;
            // }
            // else if (algorithm.equalsIgnoreCase(AES256_WRAP))
            // {
            // algorithm = SMIMEEnvelopedGenerator.AES256_WRAP;
            // }
            else
            {
                throw new UnsupportedAlgorithmException(algorithm);
            }
        }
        else
        {
            algo = CMSAlgorithm.RC2_CBC;
        }
        return algo;
    }

    /**
     * Load public key from given certificate file using PKCS7 or X.509
     * 
     * @param recipientCertificate
     * @return
     * @throws IOException
     */
    private X509Certificate[] loadPublicCertificate(String recipientCertificate)
            throws IOException
    {
        X509Certificate[] certs = null;
        try
        {
            // Get public key of recipient for encryption
            PKCS7 p7 = new PKCS7(new FileInputStream(recipientCertificate));
            certs = p7.getCertificates();
        }
        catch (Exception e)
        {
            Logger logger = ServiceFactory.getLoggingService().getLogger("coldfusion.mail");
            logger.error(e.getMessage());

            // load public key certificate as X.509
            loadX509PublicCertificate(recipientCertificate, certs);
        }
        return certs;
    }

    /**
     * Loads the X.509 public key certificate from given certificate file.
     * 
     * @param recipientCertificate
     * @param certs
     * @return
     * @throws IOException
     */
    private X509Certificate[] loadX509PublicCertificate(String recipientCertificate, X509Certificate[] certs)
            throws IOException
    {
        FileInputStream inStream = null;
        try
        {
            // try reading public key from X.509 certificate.
            inStream = new FileInputStream(recipientCertificate);
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            X509Certificate cert = (X509Certificate) cf.generateCertificate(inStream);
            if (cert != null)
            {
                if (certs == null)
                {
                    certs = new X509Certificate[1];
                }

                certs[0] = cert;
            }
        }
        catch (Exception ex)
        {
            throw new UnrecoverablePublicKeyException(ex);
        }
        finally
        {
            if (inStream != null)
            {
                inStream.close();
            }
        }
        return certs;
    }

    private MimeMessage signMail(MimeMessage message, Session session)
    {
        // Check if the keystore is valid. This is required because the file name can also come from config .
        if(keystoreFile == null || keystoreFile.trim().length() == 0 || new File(keystoreFile).exists() == false)
        {
            throw new KeyStoreNotFoundException(keystoreFile);
        }
        
        // If these properties have not been set, then return the message as it is.
        if(keyPassword == null)
            keyPassword = keystorePassword;

        // Load the keystore. First try with JKS and then with pkcs12.
        KeyStore keystore = getKeyStore(keystoreFile, keystorePassword);

        try
        {
            // If key alias is not defined, then get the first entry in the keystore.
            if(keyalias == null || keyalias.trim().length() == 0)
            {
                keyalias = getKeyAlias(keystore);
            }

            Certificate[] chain = keystore.getCertificateChain(keyalias);
            if(chain == null || chain.length == 0)
                throw new KeyNotFoundException(keyalias);
        
            // Retrieve private key required to sign the mail.
            PrivateKey privateKey;
            try
            {
                privateKey = (PrivateKey) keystore.getKey(keyalias, keyPassword.toCharArray());
            } catch (UnrecoverableKeyException e)
            {
                throw new UnrecoverablePrivateKeyException(e);
            }

            ASN1EncodableVector attributes = new ASN1EncodableVector();
            X509Certificate certificate = (X509Certificate) chain[0];
            String issuerName = certificate.getIssuerDN().getName();
            IssuerAndSerialNumber issuerAndSerialNumber = new IssuerAndSerialNumber(new X500Name(issuerName), certificate.getSerialNumber());
            attributes.add(new SMIMEEncryptionKeyPreferenceAttribute(issuerAndSerialNumber));
            attributes.add(new SMIMECapabilitiesAttribute(capabilities));

            SMIMESignedGenerator signer = new SMIMESignedGenerator();
            String hashingAlgo = "DSA".equals(privateKey.getAlgorithm()) ? SMIMESignedGenerator.DIGEST_SHA1
                                                                            : "SHA1withRSA";

            SignerInfoGenerator signerInfo = new JcaSimpleSignerInfoGeneratorBuilder().setProvider("BC")
                    .setSignedAttributeGenerator(new AttributeTable(attributes))
                    .build(hashingAlgo, privateKey, certificate);
            signer.addSignerInfoGenerator(signerInfo);
            
            // Add the list of certs to the generator
            List<Certificate> certList = new ArrayList<Certificate>();
            certList.add(chain[0]);
            signer.addCertificates(new JcaCertStore(certList));

            // Sign the message
            MimeMultipart signedMsgMultiPart = signer.generate(message);
            
            MimeMessage signedMessage = new MimeMessage(session);

            // Set all original MIME headers in the signed message
            Enumeration headers = message.getAllHeaderLines();
            while (headers.hasMoreElements())
            {
                signedMessage.addHeaderLine((String) headers.nextElement());
            }

            // Set the content of the signed message and save it
            signedMessage.setContent(signedMsgMultiPart);
            signedMessage.saveChanges();
            return signedMessage;
        }catch(ApplicationException ae)
        {
            throw ae;
        }catch (Exception e)
        {
            throw new MailSignException(e);
        }
    }

    private String getKeyAlias(KeyStore keystore) throws KeyStoreException
    {
        Enumeration aliases = keystore.aliases();
        if(aliases.hasMoreElements())
            return (String) aliases.nextElement();
        return null;
    }

    private KeyStore getKeyStore(String keystoreFile, String keystorePassword)
    {
        try
        {
            return getKeyStore(keystoreFile, keystorePassword, JKS_TYPE);
        } catch (ApplicationException e)
        {
            throw e;
        } catch (Exception e)
        {
            try
            {
                return getKeyStore(keystoreFile, keystorePassword, PKCS12_TYPE);
            } catch (ApplicationException e1)
            {
                throw e1;
            } catch(Throwable t)
            {
                throw new InvalidKeystoreException(t);
            }
        }
    }

    private KeyStore getKeyStore(String keystoreFile, String keystorePassword, String storeType) throws Exception
    {
        FileInputStream fis = null;

        try
        {
            fis = new FileInputStream(keystoreFile);
            KeyStore keystore = (storeType == PKCS12_TYPE)? KeyStore.getInstance(storeType, BC) : KeyStore.getInstance(storeType);
            keystore.load(fis, keystorePassword.toCharArray());
            return keystore;
        } finally
        {
            if(fis != null)
                fis.close();
        }
    }

    /**
     * Utility method that checks if the charset is ISO-2022-JP and
     * performs a Base64 encoding ("B") on the text instead of letting JavaMail
     * encode it in Quoted-Printable ("Q").
     */
    private String encodeIfJapanese(String input)
    {
        String encodedString;
        if (!"ISO-2022-JP".equalsIgnoreCase(charset))
            return input;

        try
        {
            // We want this to be Base64 encoded, not Quoted-Printable
            encodedString = MimeUtility.encodeText(input, charset, "B");
        }
        catch (UnsupportedEncodingException e)
        {
            // too bad, ignore it and use original
            return input;
        }

        return encodedString;
    }

    protected void finalize() throws Throwable
    {
        try
        {
            clear();
        }
        finally
        {
            super.finalize();
        }
    }

    public void clear()
    {
        mimetype = "text/plain";
        charset = null;
        contentType = null;
        sender = null;
        replyTo = null;
        recipient = null;
        cc = null;
        bcc = null;
        subject = null;

        username = null;
        password = null;

        bodyparts.clear();
        attachments.clear();
        headers.clear();

        spoolEnable = null;
        useSSL = null;
        useTLS = null;
        wrapText = -1;
        debug = false;
        multipart = false;
    }

    /**
     * Add a body part to the mail message.
     *
     * @param text the content of the part
     * @param type the Mime type of the part, or null
     * @param cs the charset used in the content-type header, or null
     * @param wrap characters to wrap content, or -1 for no wrapping
     * @throws MessagingException
     * @throws UnsupportedEncodingException
     */
    public void addPart(String text, String type, String cs, int wrap)
            throws MessagingException, UnsupportedEncodingException
    {
        // Charset preference
        // - Use the one in the mime type string, if there
        // - Use the charset passed in for the part, if set
        // - Use the charset of the mail tag (which is defaulted to UTF-8)
        if (cs == null)
        {
            cs = charset;
        }

        // Content-type
        if (type == null)
        {
            type = mimetype;
        }

        ContentType ct;
        try
        {
            ct = new ContentType(type);
            String csParam = ct.getParameter("charset");
            // Use explicit charset if provided and none in the typeString
            if (csParam == null && cs != null)
                ct.setParameter("charset", cs);
            // We we found a charset in the type string, always use it
            if (csParam != null)
                cs = csParam;
        }
        catch (javax.mail.internet.ParseException e)
        {
            // Use fallback default
            ct = new ContentType("text/plain");
        }

        // Use default wrapping if not specified (default could also be -1)
        if (wrap == -1)
        {
            wrap = wrapText;
        }

        // Create a new part
        MimeBodyPart mbp = new MimeBodyPart();

        // Wrap lines of the body at specified columns, do not strip existing newlines
        if (wrap != -1 && !type.equalsIgnoreCase("text/html"))
        {
            text = StringFunc.Wrap(text, wrap, false);
        }

        // Set up the text as a data source
        mbp.setDataHandler(new DataHandler(new StringDataSource(text, ct.toString(), cs)));

        // Bug 54673 - 8bit is not suitable. In most cases, it should be quoted-printable or base64.
        // JavaMail should choose the right one automatically, it has logic to do so.
        //mbp.setHeader("Content-Transfer-Encoding", "8bit");

        // Store the content-type header, we may need it later if we are spooling the message
        mbp.setHeader("Content-Type", ct.toString());

        bodyparts.add(mbp);
        if (bodyparts.size() > 1)
            setMultipart(true);
    }

    /**
     * Retrieve the body of the message.
     * If the first BodyPart of the message does not have a Mime type of text/*
     * or if the content object is not a string, null is returned.
     * <P>
     * Used by the MailSpooler.
     *
     * @return a string containing the first BodyPart of the message or null.
     */
    public String getBody() throws MessagingException, IOException
    {
        String sb = null;
        // text message body that got put in first
        BodyPart bp = (BodyPart) bodyparts.get(0);
        if (bp != null)
        {
            if (bp.isMimeType("text/*"))
            {
                Object content = bp.getContent();
                if (content instanceof String)
                {
                    sb = (String) content;
                }
                else if (content instanceof StringInputStream)
                {
                    // This is our special InputStream which gets us the string.
                    StringInputStream sin = (StringInputStream) content;
                    sb = sin.toString();
                }
            }
        }

        return sb;
    }

    /**
     * Retrieve the text body parts a multipart message.
     *
     * @return an string array of body part contents
     */
    public String[] getBodyParts() throws MessagingException, IOException
    {
        int num = bodyparts.size();
        if (num == 0)
        {
            return null;
        }
        String[] parts = new String[num];
        for (int i = 0; i < num; i++)
        {
            BodyPart bp = (BodyPart) bodyparts.get(i);
            if (bp != null)
            {
                if (bp.isMimeType("text/*"))
                {
                    final Object content = bp.getContent();
                    if (content instanceof String)
                    {
                        parts[i] = (String) content;
                    }
                    else if (content instanceof StringInputStream)
                    {
                        // This is our special InputStream which gets us the string.
                        StringInputStream sin = (StringInputStream) content;
                        parts[i] = sin.toString();
                    }
                }
            }
        }
        return parts;
    }

    /**
     * Get the list of content-types for each of the body parts
     * @return a string array of mime types
     */
    public String[] getBodyTypes() throws MessagingException
    {
        int num = bodyparts.size();
        if (num == 0)
        {
            return null;
        }
        String[] types = new String[num];
        for (int i = 0; i < num; i++)
        {
            BodyPart bp = (BodyPart) bodyparts.get(i);
            if (bp != null)
            {
                if (bp.isMimeType("text/*"))
                {
                    final Object ct = bp.getContentType();
                    types[i] = (String) ct;
                }
            }
        }
        return types;
    }
    
    public int getBodyPartsSize()
    {
        return bodyparts.size();
    }

    /**
     * Add an attachment, JavaMail figures out the Mime type.
     *
     * @param file the file contents to add
     */
    public void setAttachment(File file, Boolean remove) throws MessagingException
    {
        setAttachment(file, null, null, null, remove, null);
    }

    /**
     * Add an attachment with a mime type specified.
     * @param file the file to add
     * @param type the MIME type of the attachement
     * @param disposition Content-Disposition header (inline or attachement), may be null.
     * @param id Content-ID header value, may be null.
     */
    public void setAttachment(File file, String type, String disposition, String id, Boolean remove, String filename) throws MessagingException
    {
        // This is better in terms of setting the right MIME type.
        addRemoveAttachment(file.getAbsolutePath(),remove);
        MimeBodyPart mbp = new MimeBodyPart();
        mbp.setDataHandler(new DataHandler(new VFileDataSource(file)));
        if(filename == null || filename.trim().isEmpty())
            filename = getFileName(file);
        RFC2231Util.setFileName(mbp, filename, disposition, charset);
        setAttachment(mbp, type, id);
    }
    
    /**
     * 4019511 when attaching a file from s3 bucket which is inside some subfolder the subfolder
     * name is also getting prepended to the file name. To Prevent it stripping out the subfolders name
     * if the file is of type s3.
     * @param file
     * @return
     */
    private String getFileName(File file){
        String fileName = file.getName();
        if(file instanceof VFile){
            VFile vfile = (VFile) file;
            FileObject fileObject = vfile.getFileObject();
            StoreService storeService = ServiceFactory.getStoreService(false);
            //if(fileObject instanceof S3FileObject) 
            if((storeService != null && storeService.isInstanceOfS3FileObject(fileObject)))
            {
                int index = fileName.lastIndexOf("/");
                if(index > -1 && index < fileName.length() - 1)
                    return fileName.substring(index + 1);
            }
        }
        return fileName;
    }

    /**
     * Set byte[] as attachment
     * @param binAttach
     * @param type
     * @param disposition
     * @param id
     * @throws MessagingException
     */
    public void setAttachment(final byte[] binAttach,final String fileName,final String type,final String disposition,final String id,final Boolean remove,final PageContext pageContext) throws MessagingException
    {
            // we shall use temp file for updater, may revisit for temp folder in cf9
            //String tempDir = ServiceFactory.getMailSpoolService().getEmailTempDir();
            String msg = (String)AccessController.doPrivileged(new PrivilegedAction()
            {
                public Object run()
                {
                    String msg = null;
                    String tempFileStr = Utils.getTempFile(pageContext, "cf");
                    File tempFile = new File(tempFileStr);
                    try
                    {
                        BufferedOutputStream bos = new BufferedOutputStream(new FileOutputStream(tempFile));
                        bos.write(binAttach);
                        bos.close();
                        // This is better in terms of setting the right MIME type.
                        MimeBodyPart mbp = new MimeBodyPart();
                        mbp.setDataHandler(new DataHandler(new FileDataSource(tempFile)));
                        RFC2231Util.setFileName(mbp, fileName, disposition, charset);
                        addRemoveAttachment(tempFileStr);
                        setAttachment(mbp, type, id);
                    } catch (IOException e)
                    {
                        msg = e.getLocalizedMessage();
                    } catch (MessagingException e)
                    {
                        msg = e.getLocalizedMessage();
                    }
                    return msg;
                }
            });
            if(msg !=null)
                throw new MessagingException(msg);

    }

    /**
     * Refractored to add support for byte[] directly
     * @param mbp
     * @param type
     * @param disposition
     * @param id
     * @throws MessagingException
     */
    private void setAttachment(MimeBodyPart mbp, String type, String id) throws MessagingException
    {
        if (type != null)
            mbp.setHeader("Content-Type", type);
        if (id != null)
            mbp.setHeader("Content-ID", id);
        attachments.addElement(mbp);
    }

    /**
     * Add an attachment via a URL, JavaMail figures out the Mime type.
     * @param url the URL of the file contents
     */
    public void setAttachment(URL url) throws MessagingException
    {
        setAttachment(url, null, null, null, null);
    }

    /**
     * Add an attachment via a URL with a mime type specified.
     * @param url the URL of the file contents
     * @param type the MIME type of the attachement
     * @param disposition Content-Disposition header (inline or attachement), may be null.
     * @param id Content-ID header value, may be null.
     */
    public void setAttachment(URL url, String type, String disposition, String id, String filename) throws MessagingException
    {
        // This is better in terms of content caching for performance (the URL is from FileServlet).
        MimeBodyPart mbp = new MimeBodyPart();
        mbp.setDataHandler(new DataHandler(new URLDataSource(url)));
        if(filename == null || filename.trim().isEmpty()){
            String urlstr = url.toString();
            filename = urlstr.substring(urlstr.lastIndexOf("/") + 1);
            if (filename.length() == 0)
                filename="url.txt";
        }
        mbp.setFileName(filename);
        if (type != null)
            mbp.setHeader("Content-Type", type);
        if (disposition != null)
            mbp.setDisposition(disposition);
        if (id != null)
            mbp.setHeader("Content-ID", id);
        attachments.addElement(mbp);
    }

    /**
     * Return the Map of attachment file names and MimeBodyParts
     * Used by the Spool manager.
     *
     * @return a Map of attachement filenames as keys and a MimeBodyPart as the value
     * @throws MessagingException
     */
    public Map getAttachements() throws MessagingException
    {
        // Since spooling is the default, do we want to NOT use
        // a MimeBodyPart object to store the attacmhment info.
        HashMap map = null;
        if (attachments.size() > 0)
        {
            map = new HashMap();
            for (int i = 0; i < attachments.size(); i += 1)
            {
                // Get the body part
                MimeBodyPart mimeBodyPart = (MimeBodyPart) attachments.elementAt(i);

                // Get the filename
                String filename = null;
                DataHandler dh = mimeBodyPart.getDataHandler();
                DataSource ds = dh.getDataSource();
                if (ds instanceof URLDataSource)
                    filename = ((URLDataSource) ds).getURL().toString();
                else if (ds instanceof FileDataSource)
                    filename = ((FileDataSource) ds).getFile().getAbsolutePath();

                // Drop it in the map if we got a filename
                if (filename != null)
                {
                   map.put(filename, mimeBodyPart);
                }
            }
        }
        return map;
    }

    /**
     * Set a header for this message.
     *
     * @param name name of the header
     * @param value value of the header
     */
    public void setHeader(String name, String value)
            throws MessagingException, UnsupportedEncodingException
    {
        // value must contain US-ASCII characters, so...
        if (name.equalsIgnoreCase("reply-to"))
            setReplyTo(setInternetAddress(value));
        else if (name.equalsIgnoreCase("content-type"))
        {
            contentType = MimeUtility.encodeText(value);
        }
        else
        {
            Header h = new Header(name, MimeUtility.encodeText(value));
            headers.add(h);
        }
    }

    /**
     * Return the list of headers for the Spool manager
     * @return an Enumeration of Header objects
     */
    public Header[] getHeaders() throws MessagingException
    {
        Header[] ret = new Header[headers.size()];
        headers.toArray(ret);
        return ret;
    }

    /**
     * Utility to parse an array of addresses and encode the Personal Info with
     * the correct character set.
     **/
    private InternetAddress[] encodePersonalInfo(InternetAddress[] addresses)
    {
        if (addresses == null)
            return null;

        for (int i = 0; i < addresses.length; i++)
        {
            try
            {
                addresses[i].setPersonal(addresses[i].getPersonal(), charset);
            }
            catch (UnsupportedEncodingException e)
            {
                // ignore this and leave the address alone
            }
        }
        return addresses;
    }

    /**
     * Utility to parse an address string and encode the Personal Info with
     * the correct character set.
     **/
    private InternetAddress encodePersonalInfo(InternetAddress address)
    {
        if (address == null)
            return null;

        try
        {
            address.setPersonal(address.getPersonal(), charset);
        }
        catch (UnsupportedEncodingException e)
        {
            // ignore this and leave the address alone
        }
        return address;
    }

   
}
    

/**
 * Special sub class of ByteArrayInputStream so we can retreive the string
 * for spooling without a major stress.
 */
class StringInputStream extends ByteArrayInputStream
{
    String content;
    String charset;

    StringInputStream(String s, String cs) throws IOException
    {
        super(s.getBytes(cs));
        content = s;
        charset = cs;
    }

    public String toString()
    {
        return content;
    }

}

/**
 * Data source for JavaMail to wrap message parts
 */
class StringDataSource implements DataSource
{
    protected StringDataSource(String text, String m, String c)
    {
        mimetype = m;
        charset = MimeUtility.javaCharset(c);
        content = text;
    }

    protected String mimetype = "text/plain";
    protected String charset = RuntimeServiceImpl.getDefaultCharset();
    protected String content;

    public InputStream getInputStream() throws IOException
    {
        return new StringInputStream(content, charset);
    }

    public OutputStream getOutputStream() throws IOException
    {
        throw new IOException("");
    }

    public String getContentType()
    {
        return mimetype;
    }

    public String getName()
    {
        return "StringDataSource";
    }
}