/**
 * MailPartTag
 *
 * Copyright (c) 1995 - 2003 Macromedia, Inc. All rights reserved.
 * DO NOT REDISTRIBUTE THIS SOFTWARE IN ANY WAY WITHOUT THE EXPRESSED
 * WRITTEN PERMISSION OF MACROMEDIA.
 *
 * Created on Mar 6, 2003
 * Author: tomj
 */
package coldfusion.tagext.mail;

import coldfusion.mail.mod.MailImpl;
import coldfusion.tagext.ChildTag;
import coldfusion.tagext.OutputException;

import jakarta.servlet.jsp.tagext.Tag;
import jakarta.servlet.jsp.tagext.BodyTag;
import jakarta.servlet.jsp.JspException;

/**
 * Specify a part mail of a mail message for CFMail tag.
 *
 * @tag mailpart
 * @tagbody JSP
 * @author Tom Jordahl
 */
public class MailPartTag extends ChildTag implements BodyTag
{
    // Attributes
    protected String charset;
    protected String type;
    protected int wrapText = -1;

    // stuff
    private MailTag ancestor;
    private MailImpl impl;

    ///////////////////////////////////////////////////
    // Constructor
    ////////////////////////////////////////////////////
    public MailPartTag()
    {
        super(MailTag.class);
    }

    ////////////////////////////////////////////////////
    // Tag attributes
    ////////////////////////////////////////////////////

    /**
     * The character set of the mail part. Optional.
     * Used in the Content-Type header of the message part.
     * Format: type; charset=foo
     * @tagattribute
     * @required no
     * @rtexprvalue yes
     */
    public void setCharset(String charset)
    {
        this.charset = charset;
    }

    public String getCharset()
    {
        return charset;
    }

    /**
     * The mime type of the message part.  Required.
     * Special values are PLAIN and HTML, otherwise sets the content-type of the part.
     *
     * @tagattribute
     * @required yes
     * @rtexprvalue yes
     */
    public void setType(String t)
    {
        if (t.length() == 0 || t.equalsIgnoreCase("PLAIN") || t.equalsIgnoreCase("TEXT"))
            type = "text/plain";
        else if (t.equalsIgnoreCase("HTML"))
            type = "text/html";
        else
            type = t;
    }

    public String getType()
    {
        return type;
    }

    /**
     * The number of characters to wrap the message body text at. Optional
     *
     * @tagattribute
     * @required no
     * @rtexprvalue yes
     */
    public void setWrapText(int wrapText)
    {
        this.wrapText = wrapText;
    }

    public int getWrapText()
    {
        return wrapText;
    }

    ////////////////////////////////////////////////////
    // ChildTag Interface
    ////////////////////////////////////////////////////
    protected void setAncestor(Tag ancestor)
    {
        this.ancestor = (MailTag) ancestor;
    }

    ////////////////////////////////////////////////////
    // Tag implementation
    ////////////////////////////////////////////////////

    /**
     * Start tag
     */
    public int doStartTag() throws JspException
    {
        onTagStart();
        impl = ancestor.getMailImpl();
        return BodyTag.EVAL_BODY_TAG;
    }

    /**
     * After Body
     */
    public int doAfterBody() throws JspException
    {
        try
        {
            final String body = bodyContent.getString();
            impl.addPart(body, type, charset, wrapText);
            bodyContent.clearBody();
        }
        catch (Exception ex)
        {
            throw new OutputException(ex);
        }

        return super.doAfterBody();
    }

    /**
     * End tag
     */
    public int doEndTag() throws JspException
    {
        onTagEnd();
        return super.doEndTag();
    }

    /**
     * Release tag resources
     */
    public void release()
    {
        // Attributes
        charset = null;
        wrapText = -1;

        // Stuff
        ancestor = null;
        impl = null;

        super.release();
    }

}

