/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 2009 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and may be covered by U.S. and Foreign Patents,
* patents in process, and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
package coldfusion.tagext.mail;

import coldfusion.runtime.ApplicationException;
import coldfusion.runtime.Cast;
import coldfusion.runtime.RequestMonitor;
import coldfusion.runtime.RequestTimedOutException;
import coldfusion.sql.Table;
import coldfusion.tagext.GenericTagPermission;
import coldfusion.util.ExceptionUtils;
import coldfusion.util.Utils;
import coldfusion.vfs.VFSFileFactory;
import coldfusion.tagext.net.*;
import static coldfusion.mail.core.MailExceptions.*;
import coldfusion.mail.core.*;
import coldfusion.mail.mod.HostImpl;
import coldfusion.mail.mod.IMapImpl;

import java.io.File;
import java.security.AccessController;
import java.security.Permission;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.StringTokenizer;

import javax.mail.AuthenticationFailedException;
import javax.mail.search.SearchTerm;
import jakarta.servlet.jsp.JspException;

/**
 * Retrieves and deletes e-mail messages from a IMAP mail server.
 * 
 * @tag imap
 * @tagbody jsp
 * @author Clement Wong
 */
public class IMapTag extends MsgTag implements MailFilterable
{
    private static final String CFIMAP = "cfimap";

    private static final long serialVersionUID = 1L;

    private static final GenericTagPermission tp = new GenericTagPermission(CFIMAP);

    protected static final int DEF_START = 1;
    protected static final int DEF_MAXROWS = -1;
    protected static final int DEF_TIMEOUT = 60000;
    protected static final int DEF_PORT = 143;
    protected static final int DEF_PORT_SSL = 993;

    protected IMapImpl impl;
    protected HostImpl himpl;

    protected String action = "GetHeaderOnly";
    protected String msgnums;
    protected String msgids;
    protected String delimiter = null;
    protected int startrow = DEF_START;
    protected int maxrows = DEF_MAXROWS;
    protected String server;
    protected int port=-1;
    protected String username;
    protected String password;
    protected String attachpath;
    protected String folder=null;
    protected String newFolder =null;
    protected int timeout = DEF_TIMEOUT;
    protected boolean generateuniquefilenames = false;
    protected boolean debug = false;
    private long startTime = 0;
    protected boolean useSSL = false;
    private boolean recurse = false;
    private boolean stopOnError = true;
    private String conName=null;
    private String accept = null;

    // filter information for getting e-mail messages
    MailFilterInfo filterInfo = new MailFilterInfo();


    protected Permission getPermission()
    {
        return tp;
    }

    protected void validate() throws JspException
	{
        if(conName == null || action.equalsIgnoreCase(IMAPUtils.action_OPEN))
        {
            if(port==-1)
            {
                if(useSSL)
                    port=DEF_PORT_SSL;
                else
                    port=DEF_PORT;
            }
            if(useSSL)
                himpl = new HostImpl("imaps");
            else
                himpl = new HostImpl("imap");

            himpl.setHost(server);
            himpl.setPort(port);
            himpl.setTimeout(timeout);
            himpl.setUseSSL(useSSL);
            himpl.setUsername(username);
            himpl.setPassword(password);

            impl = new IMapImpl();

            impl.setMailSource(himpl);
            impl.setUsername(username);
            impl.setPassword(password);
            impl.setPath(attachpath);
            impl.setUniqueFilenames(generateuniquefilenames);
            impl.setDebug(debug);
            impl.setUseSSL(useSSL);
            impl.setAccept(accept);
        }
        else if(conName != null)
        {
            Object value = pageContext.findAttribute(conName);
            if(value == null || !(value instanceof IMapImpl))
                throw new InvalidConnectionException(action,conName);
            impl = (IMapImpl) value;
            if(impl.getFolder()==null || impl.getStore()==null)
                    throw new ClosedConnectionException(action,conName);
            // override the new values
            impl.setPath(attachpath);
            impl.setUniqueFilenames(generateuniquefilenames);
            impl.setDebug(debug);
            impl.setAccept(accept);
        }
    }

    /**
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setAccept(String o)
    {
        accept = o;
    }

    /**
	 * Optional.
	 * @tagattribute
	 * @required no
	 * @rtexprvalue yes
	 */
    public void setStopOnError(boolean stopOnError)
    {
        this.stopOnError = stopOnError;
    }

    /**
	 * Optional.
	 * @tagattribute
	 * @required no
	 * @rtexprvalue yes
	 */
    public void setRecurse(boolean recurse)
    {
        this.recurse = recurse;
    }

    /**
	 * Optional.
	 * @tagattribute
	 * @required no
	 * @rtexprvalue yes
	 */
    public void setConnection(String conName)
    {
        this.conName = conName;
    }

    public boolean isUseSSL()
    {
        return useSSL;
    }

    /**
	 * Optional. Defaults to false
	 * @tagattribute
	 * @required no
	 * @rtexprvalue yes
	 */
    public void setSecure(boolean useSSL)
    {
        this.useSSL = useSSL;
    }

    public String getNewFolder()
    {
        return newFolder;
    }

    /**
	 * Optional. Defaults to false
	 * @tagattribute
	 * @required no
	 * @rtexprvalue yes
	 */
    public void setNewFolder(String newFolder)
    {
        if (newFolder==null || "".equals(newFolder))
        {
            throw new EmptyAttributeException("newfolder");
        }
        this.newFolder = newFolder;
    }

    public String getFolder()
    {
        return folder;
    }

    /**
	 * Optional. Defaults to inbox
	 * @tagattribute
	 * @required no
	 * @rtexprvalue yes
	 */
    public void setFolder(String folder)
    {
        this.folder = folder;
    }

    /**
	 * Required. Host name (biff.upperlip.com) or IP address (192.1.2.225) of the IMAP server
	 *
	 * @tagattribute
	 * @required yes
	 * @rtexprvalue yes
	 */
    public void setServer(String s) {
        server = s;
    }

    public String getServer() {
        return server;
    }
	/**
	 * Optional. Defaults to the standard IMAP port, 110.
	 * @tagattribute
	 * @required no
	 * @rtexprvalue yes
	 */
    public void setPort(int p) {
        port = p;
    }

    public int getPort() {
        return port;
    }

	/**
	 * Optional. Specifies the mail action. Options: getHeaderOnly, getAll, delete
	 * @tagattribute
	 * @required no
	 * @rtexprvalue yes
	 */
    public void setAction(String a) {
        action = a;
    }

	public String getAction() {
        return action;
    }

    /**
	 * Optional. If no user name is specified, the IMAP connection is anonymous
	 * @tagattribute
	 * @required no
	 * @rtexprvalue yes
	 */
    public void setUsername(String u) {
        username = u;
    }

    public String getUsername() {
        return username;
    }
	/**
	 * Optional. Password that corresponds to user name.
	 * @tagattribute
	 * @required no
	 * @rtexprvalue yes
	 */
    public void setPassword(String p) {
        password = p;
    }

    protected String getPassword() {
        return password;
    }

	/**
	 * Optional. Name for the index query. Required for action = "getHeaderOnly" and action = "getAll".
	 * @tagattribute
	 * @required no
	 * @rtexprvalue yes
	 */
    public void setName(String n) {
        setId(n);
    }

    public String getName() {
        return getId();
    }
	/**
	 * Optional. Specifies the message number(s) for the given action
	 * @tagattribute
	 * @required no
	 * @rtexprvalue yes
	 */
    public void setMessagenumber(String num) {
        msgnums = num;
    }

    public String getMessagenumber() {
        return msgnums;
    }
	/**
	 * Optional. Specifies the unique mail id(s) for the given action
	 * @tagattribute
	 * @required no
	 * @rtexprvalue yes
	 */
    public void setUID(String ids) {
        msgids = ids;
    }

    public String getUID() {
        return msgids;
    }

    /**
     * Optional. Specifies the delimiter to be used for parsing mail id(s) for the given action
     * 
     * @tagattribute
     * @required no
     * @rtexprvalue yes
     */
    public void setDelimiter(String delimiter)
    {
        this.delimiter = delimiter;
    }

    public String getDelimiter()
    {
        return delimiter;
    }

	/**
	 * Optional. Allows attachments to be written to the specified directory when action = "getAll".
     * If the path specified is relative, it will be in the temp directory.
	 * @tagattribute
	 * @required no
	 * @rtexprvalue yes
	 */
    public void setAttachmentpath(String p) {
        // Make sure relative paths are put in the temp directory
        attachpath = Utils.getFileFullPath(p, pageContext);
    }

    public String getAttachmentpath() {
        return attachpath;
    }

	/**
	 * Optional. Specifies the maximum time, in seconds, to wait for mail processing.
	 * Defaults is 60 seconds
	 * @tagattribute
	 * @required no
	 * @rtexprvalue yes
	 */
    public void setTimeout(int t) {
        timeout = t;
    }

    public int getTimeout() {
        return timeout;
    }

	/**
	 * Optional. Sets the number of messages returned, starting with the number in the
	 * startRow attribute.  This attribute is ignored if messageNumber is specified.
	 * @tagattribute
	 * @required no
	 * @rtexprvalue yes
	 */
    public void setMaxrows(int r) {
        maxrows = r;
    }

    public int getMaxrows() {
        return maxrows;
    }
	/**
	 * Optional. Specifies the first row number to be retrieved. Default is 1. This attribute is
	 * ignored if messageNumber is specified.
	 * @tagattribute
	 * @required no
	 * @rtexprvalue yes
	 */
    public void setStartrow(int r) {
        startrow = r;
    }

    public int getStartrow() {
        return startrow;
    }

	/**
	 * Optional. Boolean indicating whether to generate unique filenames for the
	 * files attached to an e-mail message to avoid naming conflicts when the files
	 * are saved. Default is NO.
	 * @tagattribute
	 * @required no
	 * @rtexprvalue yes
	 */
    public void setGenerateuniquefilenames(boolean b) {
        generateuniquefilenames = b;
    }

    public boolean isGenerateuniquefilenames() {
        return generateuniquefilenames;
    }
	/**
	 * @tagattribute
	 * @required no
	 * @rtexprvalue yes
	 */
    public void setDebug(boolean d) {
        debug = d;
    }

    public boolean isDebug() {
        return debug;
    }

    public long getElapsedTime()
    {
        return System.currentTimeMillis() - startTime;
    }


    public void release() {
        if(conName==null)
            impl.clear();

        accept = null;
        conName=null;
        action = "GetHeaderOnly";
        msgnums = null;
        msgids = null;
        delimiter = null;
        startrow = DEF_START;
        maxrows = DEF_MAXROWS;
        server = null;
        port = -1;
        username = null;
        password = null;
        attachpath = null;
        folder=null;
        newFolder=null;
        timeout = DEF_TIMEOUT;
        generateuniquefilenames = false;
        recurse= false;
        folder=null;
        startTime = 0;
        filterInfo = new MailFilterInfo();
        useSSL = false;
        stopOnError = true;
        
        super.release();
    }

    public int doStartTag() throws JspException
    {
        startTime = System.currentTimeMillis();
        //Tag Start monitoring event. This exists with onTagEnd() which is invoked at the end of the tag processing.
        // If you remove this make sure you remove the onTagEnd() also
        onTagStart();
        validate();
        
        if(delimiter != null && msgids == null)
        {
            throw new IncorrectUseOfDelimiterException(CFIMAP);
        }
        
        // initialize delimiter for uUIDid tokens
        if (delimiter == null || (delimiter != null && delimiter.trim().isEmpty()))
        {
            delimiter = COMMA;
        }

        // check if end tag is present handling will be done in doEndTag else handle here.
        if (hasEndTag)
        {
            return EVAL_PAGE;
        }
        else
        {
            // no body hence handle request.
            handleRequest();
            return SKIP_BODY;
        }
    }

    public int doEndTag() throws JspException
    {
        super.doEndTag();

        if(hasEndTag)
            handleRequest();
        // invoke monitoring event handlers to capture this tag's monitor data. This exists with onTagStart() at the tagStart.
        // If you remove this make sure you remove the tagStart() invocation near the tag start
        if(impl!=null) impl.clearOperationVar();
        onTagEnd();
        return EVAL_PAGE;
    }

    /**
     * Handle IMap actions.
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    private void handleRequest()
    {
        SearchTerm searchTerm = createFilters(filterInfo, CFIMAP);

        // Verify permissions to attachment path
        if (action.equalsIgnoreCase(IMAPUtils.action_GetAll) && attachpath != null)
        {
            File f = VFSFileFactory.getFileObject(attachpath);
            // This call will throw a security exception
            // if we don't have permission to this directory
            if (! f.exists())
            {
                // Utils.saveFile() does this, which is what we use later
                // We don't document the fact that we will create this directory...
                f.mkdirs();
            }
            // Make sure we have sandbox permissions to write
            // If we actually have attachments to write, that can throw the error.
            f.canWrite();
        }

        checkTimeout();

        // Open the connection
        try {
            if(conName == null || action.equalsIgnoreCase(IMAPUtils.action_OPEN))
                impl.validate_folder();
        } catch (Exception ex) {
            // handle specific authentication failure
            if (ex instanceof AuthenticationFailedException) {
                throw new MailAuthenticationFailedException(ex);
            }
            else {
                throw new MailSessionException(ex);
            }
        }
        checkTimeout();

        // Pass the list of message numbers, UIDs or limits to the implementation class
        // if action is open or close we need not do any of these
        if(action.equalsIgnoreCase(IMAPUtils.action_OPEN))
        {
            pageContext.setAttribute(conName, impl);
        }
        else if(action.equalsIgnoreCase(IMAPUtils.action_CLOSE))
        {
            if(impl!=null)
                impl.clear();
        }
        else
        {
            if (msgids != null)
            {
                // UIDs
                ArrayList<String> uidList = new ArrayList<String>();
                StringTokenizer t = new StringTokenizer(msgids, delimiter);
                while (t.hasMoreTokens()) {
                    uidList.add(t.nextToken());
                }
                String[] uids = (String[]) uidList.toArray(new String[0]);
                impl.setUids(uids);
            }
            else if (searchTerm != null)// set filters
            {
                impl.setSearchTerm(searchTerm);
            }
            else if (msgnums != null)
            {
                // messageNumber
                ArrayList<String> msgList = new ArrayList<String>();
                StringTokenizer t = new StringTokenizer(msgnums, ",");
                while (t.hasMoreTokens()) {
                    msgList.add(t.nextToken());
                }
                String[] msgs = (String[]) msgList.toArray(new String[0]);
                impl.setMsgs(msgs);
            }
            else
            {
                if (startrow != DEF_START)
                    impl.setStartMsg(startrow);
                if (maxrows != DEF_MAXROWS)
                    impl.setMaxRows(maxrows);
            }

            // Let the implementation know what action we are taking
            if (action.equalsIgnoreCase(IMAPUtils.action_GetAll))
                impl.setBody(true);

            // Get or delete messages
            try {
                final IMapImpl tmpImpl = impl;
                Table mailtable = null;
                if (action.equalsIgnoreCase(IMAPUtils.action_GetAll) ||
                    action.equalsIgnoreCase(IMAPUtils.action_GetHeaderOnly)) {
                    mailtable = (Table)AccessController.doPrivileged(new PrivilegedExceptionAction()
                    {
                            public Object run() throws Exception
                            {
                                return tmpImpl.getMails(folder);
                            }
                    });
                } else if (action.equalsIgnoreCase(IMAPUtils.action_Delete))
                {
                    AccessController.doPrivileged(new PrivilegedExceptionAction()
                    {
                            public Object run() throws Exception
                            {
                                tmpImpl.deleteMails(folder);
                                return null;
                            }
                    });
                }
                else if (action.equalsIgnoreCase(IMAPUtils.action_MARKREAD)) {
                    AccessController.doPrivileged(new PrivilegedExceptionAction()
                    {
                            public Object run() throws Exception
                            {
                                tmpImpl.markRead(folder);
                                return null;
                            }
                    });
                }
                else if (action.equalsIgnoreCase(IMAPUtils.action_LISTALLFOLDERS)) {
                    mailtable = (Table)AccessController.doPrivileged(new PrivilegedExceptionAction()
                    {
                            public Object run() throws Exception
                            {
                                return tmpImpl.listAllFolders(folder, recurse);
                            }
                    });
                }
                else if (action.equalsIgnoreCase(IMAPUtils.action_MOVEMAIL)) {
                    AccessController.doPrivileged(new PrivilegedExceptionAction()
                    {
                            public Object run() throws Exception
                            {
                                tmpImpl.moveMails(folder,newFolder);
                                return null;
                            }
                    });
                }
                else if (action.equalsIgnoreCase(IMAPUtils.action_DELETEFOLDER)) {
                    AccessController.doPrivileged(new PrivilegedExceptionAction()
                    {
                            public Object run() throws Exception
                            {
                                tmpImpl.deleteFolder(folder);
                                return null;
                            }
                    });
                }
                else if (action.equalsIgnoreCase(IMAPUtils.action_RENAMEFOLDER)) {
                    AccessController.doPrivileged(new PrivilegedExceptionAction()
                    {
                            public Object run() throws Exception
                            {
                                tmpImpl.renameFolder(folder,newFolder);
                                return null;
                            }
                    });
                }
                else if (action.equalsIgnoreCase(IMAPUtils.action_CREATEFOLDER)) {
                    AccessController.doPrivileged(new PrivilegedExceptionAction()
                    {
                            public Object run() throws Exception
                            {
                                tmpImpl.createFolder(folder);
                                return null;
                            }
                    });
                }


                // Set the result query if we have one
                if (getId() != null && mailtable != null) {
                    pageContext.setAttribute(getId(), mailtable);
                }

            } catch (Exception ex) {
                if (stopOnError)
                {
                    if (ex instanceof PrivilegedActionException) {
                        ex = ((PrivilegedActionException)ex).getException();
                    }
                    if (ex instanceof ArrayIndexOutOfBoundsException) {
                        // We pass the number of messages as the (string) message in this exception
                        // since IMapImpl can't directly throw InvalidStartRowException
                        ArrayIndexOutOfBoundsException aiob = (ArrayIndexOutOfBoundsException) ex;
                        int numrows = 0;
                        try { numrows = (Integer.parseInt(aiob.getMessage())); }
                        catch (NumberFormatException e) {
                            // some other Array problem, don't convert to InvalidStartRow
                            throw new IMapTagException(ex,action);
                        }
                        throw new InvalidStartRowException(startrow, numrows);
                    }
                    else if (ex instanceof ApplicationException)
                    {
                        throw (ApplicationException)ex;
                    }
                    throw new IMapTagException(ex, action);
                }
            }
        }
    }
    
    MailFilterInfo getFilterInfo()
    {
        return filterInfo;
    }

    /**
     * Sets filter attributes for getting e-mail messages. This method is called from MailTagHelper
     * 
     * @param attribName
     *            Name of a filter attribute
     * @param values
     *            HashMap containing all filter attributes
     */
    public void addFilter(String attribName, HashMap<String, Object> values)
    {
        String uAttribName = attribName.toUpperCase();
        if (isValidAttributeName(uAttribName, CFIMAP))
        {
            Object value = values.get(MsgTag.KEY_VALUE);

            Date fromDate = null, toDate = null;

            if (uAttribName.equals(MsgTag.KEY_TIMERECEIVED))
            {
                if (values.containsKey(MsgTag.KEY_FROM))
                    fromDate = Cast._Date(values.get(MsgTag.KEY_FROM));
                
                if (values.containsKey(MsgTag.KEY_TO))
                    toDate = Cast._Date(values.get(MsgTag.KEY_TO));
                
                if (fromDate != null || toDate != null)
                {
                    filterInfo.setFromTimeReceived(fromDate);
                    filterInfo.setToTimeReceived(toDate);
                } else
                {
                    if (value != null)
                        throw new InvalidFilterAttribComboException("CFIMapFilter", MsgTag.KEY_NAME + ","
                                + MsgTag.KEY_FROM + "," + MsgTag.KEY_TO);
                    
                    throw new InvalidAttributeValueException(MsgTag.KEY_TIMERECEIVED);
                }
                return;
            }
            else if (uAttribName.equals(MsgTag.KEY_TIMESENT))
            {
                if (values.containsKey(MsgTag.KEY_FROM))
                    fromDate = Cast._Date(values.get(MsgTag.KEY_FROM));
                if (values.containsKey(MsgTag.KEY_TO))
                    toDate = Cast._Date(values.get(MsgTag.KEY_TO));
                
                if (fromDate != null || toDate != null)
                {
                    filterInfo.setFromTimeSent(fromDate);
                    filterInfo.setToTimeSent(toDate);
                } else
                {
                    if (value != null)
                        throw new InvalidFilterAttribComboException("CFIMapFilter", MsgTag.KEY_NAME
                                + "," + MsgTag.KEY_FROM + "," + MsgTag.KEY_TO);
                    
                    throw new InvalidAttributeValueException(MsgTag.KEY_TIMESENT);
                }
                return;
            }
            if (value == null || value.toString().length() == 0)
                throw new EmptyAttributeValueException(attribName);

            if (!(value instanceof String))
                throw new InvalidStringValueException(attribName);
            
            if (uAttribName.equals(MsgTag.KEY_FROMID))
                filterInfo.setFromId(value.toString());
            else if (uAttribName.equals(MsgTag.KEY_TOID))
                filterInfo.setToId(value.toString());
            else if (uAttribName.equals(MsgTag.KEY_SUBJECT))
                filterInfo.setSubject(value.toString());
            else if (uAttribName.equals(MsgTag.KEY_FLAG))
                filterInfo.setFlag(value.toString());
        }
    }

    public void checkTimeout() throws RequestTimedOutException {
        if (RequestMonitor.isRequestTimedOut()) {
            throw new RequestTimedOutException(tagNameFromClass());
        }
    }

	
}