/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 2008 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and may be covered by U.S. and Foreign Patents,
* patents in process, and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/

package coldfusion.print;

import static coldfusion.print.core.PrintExceptions.*;
import coldfusion.log.CFLogs;
import coldfusion.log.Logger;
import coldfusion.print.core.CFPrintException;
import coldfusion.runtime.Array;
import coldfusion.runtime.ArrayUtil;
//import coldfusion.tagext.lang.PrintTag;
import coldfusion.util.RB;
import coldfusion.vfs.VFSFileFactory;

import javax.print.DocFlavor;
import javax.print.DocPrintJob;
import javax.print.PrintService;
import javax.print.PrintServiceLookup;
import javax.print.attribute.Attribute;
import javax.print.attribute.AttributeSet;
import javax.print.attribute.PrintRequestAttributeSet;
import javax.print.attribute.HashPrintRequestAttributeSet;
import javax.print.attribute.standard.Fidelity;
import javax.print.attribute.standard.JobName;
import javax.print.attribute.standard.Media;
import javax.print.attribute.standard.MediaName;
import javax.print.attribute.standard.MediaPrintableArea;
import javax.print.attribute.standard.MediaSize;
import javax.print.attribute.standard.MediaSizeName;
import javax.print.attribute.standard.MediaTray;
import javax.print.event.PrintJobAdapter;
import javax.print.event.PrintJobEvent;
import jakarta.servlet.jsp.PageContext;

import java.awt.print.Paper;
import java.awt.print.PrinterJob;
import java.awt.*;
import java.io.File;
import java.util.ArrayList;

/**
 * PrintBase
 * <p/>
 * Jan 19, 2007
 *
 * @author cframpto
 */
public class PrintBase
{
    PageContext pageContext;
    String type;                  // for now only PDF is supported
    Object source;
    String password = null;
    String printer = null;
    PrintRequestAttributeSet attributeSet = null;
    DocFlavor docFlavor = null;

    Logger logger = CFLogs.PRINT_LOG;
    boolean isDebug = false;

    private boolean fidelity = false;
    private boolean isFidelityRemoved = false;
    private String defaultPrinter = null;

    public static final String DEFAULT_PRINT_JOB_NAME = "ColdFusion Print Job";

    /**
     * @param pageContext
     * @param type        PrintTag.TYPE_PDF is the only type supported
     */
    protected PrintBase(PageContext pageContext, String type)
    {
        this.pageContext = pageContext;
        this.type = type;

        if (this.type.equals("pdf"))
            docFlavor = DocFlavor.SERVICE_FORMATTED.PAGEABLE;

        if (Boolean.getBoolean("cfprint.debug"))
        {
            isDebug = true;
            logger.setPriority("debug");
        }
    }

    protected PrintBase(PageContext pageContext, String type, Object source, String password)
    {
        this(pageContext, type);
        this.source = source;
        this.password = password;
    }

    /**
     * Should only be used if trying to access printer information.
     *
     * @param type PrintTag.TYPE_PDF is the only type supported
     */
    protected PrintBase(String type)
    {
        this(null, type);
    }

    /**
     */
    protected void setFidelity()
    {
        if (attributeSet != null)
        {
            Fidelity myFidelity = (Fidelity) attributeSet.get(Fidelity.class);
            fidelity = (myFidelity != null && myFidelity == Fidelity.FIDELITY_TRUE);
        }
        else
        {
            fidelity = false;
        }

    }

    /**
     * @return the default printer if there is one, or null.
     */
    protected String getDefaultPrinter()
    {
        if (defaultPrinter == null)
        {
            PrintService defaultPrintService = PrintServiceLookup.lookupDefaultPrintService();
            if (defaultPrintService != null)
            {
                defaultPrinter = defaultPrintService.getName();
            }
        }

        return defaultPrinter;
    }

    /**
     * Get list of configured printers on the system which support the given docFlavor (based on the doc type).
     * The account CF is running in may need privs to access network printers.
     *
     * @return list of all printers  which may be length=0
     */
    protected ArrayList getAllPrinters()
    {
        try
        {
            PrintUtil.refreshPrinterList();
        }
        catch (Exception e)
        {
            logger.debug("Error occurred while refreshing printers.");
        }
        
        ArrayList pList = new ArrayList();

        // *** 
        // on Windows account server is running in must have PRINTER_ACCESS_USE permission on a printer to
        // be able to query it and queue print jobs to it
        // ***
        PrintService[] service = PrintServiceLookup.lookupPrintServices(docFlavor, null);
        for (int i = 0; i < service.length; i++)
        {
            pList.add(service[i].getName());
        }

        return pList;
    }

    /**
     * Get the print service for the printer, if set, or the default printer.
     *
     * @return PrintService if available or throw an exception
     */
    protected PrintService getPrintService()
            throws CFPrintException
    {
        // Make sure either a printer was specified or there is a default printer.
        if (printer == null)
        {
            printer = getDefaultPrinter();
            if (printer == null)
            {
                // Printer not specified and no default printer.
                throw new NoDefaultPrinterException();
            }
        }

        // Get the printing services that can handle the PAGEABLE interface that is needed to print pdfs
        // from JPedal.  Then make sure the printer specified is in the list. If attributeSet is specified,
        // the lookup returns those printers that support those attributes.
        // ****NOTE****
        // If the account this is being run from doesn't have privs to access the network printers they will 
        // not show up in the list.   On Windows, PRINTER_ACCESS_USE is needed.     
        PrintService myPrintService = null;

        //start-kumar

        PrintRequestAttributeSet aset = new HashPrintRequestAttributeSet();

        //you had null as second value which seems to hang JPS
        PrintService[] service = PrintServiceLookup.lookupPrintServices(docFlavor, aset);
        //end-kumar

        ArrayList printerList = new ArrayList();
        for (int i = 0; i < service.length; i++)
        {
            printerList.add(service[i].getName());
            // case-sensitive, exact match now that printer names available on System Information admin page
            if (service[i].getName().equals(printer))
            {
                myPrintService = service[i];
                break;
            }
        }

        // Either a printer wasn't specified, it isn't available for this kind of document or it isn't available at all.
        if (myPrintService == null)
        {
            throw new PrinterUnavailableException(printer, PrintUtil.toDisplayString(printerList));
        }

        return myPrintService;
    }

    /**
     * @param printService
     * @return unsupported attributes in the attributeSet for the particular docFlavor and printService
     */
    protected ArrayList getUnsupportedAttributes(PrintService printService)
    {
        ArrayList attrList = null;

        if (attributeSet != null)
        {
            AttributeSet unsupportedAttributes = printService.getUnsupportedAttributes(docFlavor, attributeSet);
            if (unsupportedAttributes != null)
            {
                // make a list of the unsupported attribute names
                Attribute[] attrArray = unsupportedAttributes.toArray();
                attrList = new ArrayList(attrArray.length);
                for (int i = 0; i < attrArray.length; i++)
                {
                    attrList.add(attrArray[i].getName());
                }
                logger.debug(RB.getString(this, "cfprint.UnsupportedAttributes", printer, type, attrList));
            }
        }

        return attrList;
    }

    /**
     * If fidelity is true, check right now if there are any attributes which aren't supported.  There
     * is no need to proceed.  Some of the printers will return a nice error message about fidelity
     * and some of them don't.
     *
     * @param printService printer to attributes against
     */
    void checkAttributesForFidelity(PrintService printService)
            throws CFPrintRequestAttributeException
    {
        if (fidelity)
        {
            // at least one mac printer and maybe others don't support ipp-attribute-fidelity.
            if (!printService.isAttributeCategorySupported(Fidelity.FIDELITY_TRUE.getClass()))
            {
                logger.debug("cfprint: removing fidelity from attributeSet");
                isFidelityRemoved = attributeSet.remove(Fidelity.FIDELITY_TRUE.getClass());
            }

            ArrayList attrList = getUnsupportedAttributes(printService);
            if (attrList != null)
            {
                throw new CFPrintRequestAttributeException(PrintUtil.toCFAttributeNames(attrList), printService.getName());
            }
        }
    }

    /**
     * Set the JobName attribute for the print job if it wasn't explicitly set by the user.
     */
    protected String setJobNameAttribute(PrintService printService)
    {
        // Set the docPrintJob name if the user didn't explicitly set one.
        if (attributeSet != null)
        {
            Attribute attr = attributeSet.get(JobName.class);
            if (attr == null)
            {
                // If the source is a file, and no explicit docPrintJob name was set then use
                // the file name, else the default.
                String jobNameStr = null;
                if (source instanceof String)
                    jobNameStr = VFSFileFactory.getFileObject((String) source).getName();

                if (jobNameStr == null)
                    jobNameStr = DEFAULT_PRINT_JOB_NAME;

                // Make sure JobName is supported before adding it.
                JobName jobName = new JobName(jobNameStr, null);
                if (printService.isAttributeValueSupported(jobName, docFlavor, attributeSet))
                {
                    attributeSet.add(jobName);
                }
                return jobNameStr;
            }
            else
            {
                return attr.toString();
            }
        }

        return DEFAULT_PRINT_JOB_NAME;
    }

    //    protected boolean setDestination()
    //    {
    //        // Set the Destination name if the user didn't explicitly set one.  We want to write the spool
    //        // file to a place we know the user has access to, even if running within a sandbox.  The
    //        // default is the current directory.
    //        try
    //        {
    //            if (attributeSet != null && !attributeSet.containsKey(Destination.class))
    //            {
    //                String path = Utils.getTempFile(pageContext, Utils.getTempDir(pageContext), "_cfprint", ".prn");
    //                if (File.separatorChar == '\\')
    //                {
    //                    path = "/" + Utils.canonicalizeURI(path);
    //                }
    //                URI uri = new URI("file://", "localhost", path, null);
    //                attributeSet.add(new Destination(uri));
    //                return true;
    //            }
    //        }
    //        catch (URISyntaxException e)
    //        {
    //        }
    //
    //        return false;
    //    }

    /**
     * In most cases, applications will use either MediaSizeName or MediaTray. The MediaSizeName class enumerates the
     * media by size. The MediaTray class enumerates the paper trays on a printer, which usually include a main tray
     * and a manual feed tray. The IPP 1.1 specification does not provide for specifying both the media size and the
     * media tray at the same time, which means, for example, that an application cannot request size A4 paper from
     * the manual tray. A future revision of the IPP specification might provide for a way to request more than one
     * type of media at a time, in which case the JPS API will most likely be enhanced to implement this change.
     * <p/>
     * To set margin specify margin:
     * new MediaPrintableArea (margin, margin, width-margin*2, height-margin*2, MediaPrintableArea.INCHES)
     *
     * @param printService
     * @return Paper
     */
    protected Paper getPaper(PrintService printService)
            throws java.awt.print.PrinterException

    {
        // Get a print docPrintJob to find out the default paper size for the printer.
        PrinterJob printerJob = PrinterJob.getPrinterJob();

        //start-kumar
        //We never call setPrintService
        printerJob.setPrintService(printService);
        //end-kumar

        // Let the printer driver tell us the default paper size for the printer which is typically
        // A4 for non-US (and maybe Canada) locales.  It's isn't possible to request a MediaSize AND
        // a MediaTray so if MediaTray is specified the default paper size is used.       
        Paper paper = printerJob.defaultPage().getPaper();
        
        // Make borderless.
        paper.setImageableArea(0, 0, paper.getWidth(),paper.getHeight());

        // See if the user requested a media type.  If so, get it's size and it's printable area.  If a printable area
        // was also specified, use it only if it is less than or equal to the max printable area allowed.
        // If a media type wasn't requested then default to letter.        
        Media myMedia = attributeSet != null ? (Media) attributeSet.get(Media.class) : null;
        if (myMedia != null)
        {
            // Get the supported media.
            Media[] media = getSupportedMedia(printerJob);

            // Was a media type specified and does it match one of the supported ones?  If so, mediaIndex >= 0.
            int mediaIndex = -1;
            if (media != null)
            {
                for (int i = 0; i < media.length; i++)
                {
                    if (myMedia.equals(media[i]))
                    {
                        mediaIndex = i;
                        break;
                    }
                }
            }

            // Requested media not found.  Use the default if fidelity is false.
            if (mediaIndex == -1)
            {
                // If fidelity, the media has already been checked so we shouldn't get here.
                if (fidelity)
                {
                    // The requested media is not available on the requested printer.
                     throw new CFPrintRequestAttributeException(Media.class.getName(), printerJob.getPrintService().getName());
                 }
                else
                {
                    // Use the default paper size.
                    myMedia = null;
                }
            }

            if (myMedia instanceof MediaSizeName)
            {
                // Get the size of the named media.
                MediaSize mediaSize = MediaSize.getMediaSizeForName((MediaSizeName) myMedia);
                if (mediaSize != null)
                {
                    // A two-element array with the X dimension at index 0 and the Y dimension at index 1.
                    float[] paperSize = mediaSize.getSize(MediaSize.INCH);

                    //  4 values in the order x, y, w, h
                    MediaPrintableArea[] printableArea = getSupportedMediaPrintableArea(printerJob.getPrintService());
                    float[] printArea;
                    if (printableArea != null && mediaIndex >= 0 && printableArea.length == 1)
                    {
                        printArea = printableArea[0].getPrintableArea(MediaPrintableArea.INCH);

                        // See if the user requested a media printable area.
                        MediaPrintableArea myMediaPrintableArea = (MediaPrintableArea) attributeSet.get(MediaPrintableArea.class);
                        if (myMediaPrintableArea != null)
                        {
                            // Only use specified print area if the box it makes fits within the
                            // printable area for the media.
                            //  4 values in the order x, y, w, h
                            float[] myPrintArea = myMediaPrintableArea.getPrintableArea(MediaPrintableArea.INCH);
                            float maxX = printArea[0] + printArea[2];
                            float myMaxX = myPrintArea[0] + myPrintArea[2];
                            float maxY = printArea[1] + printArea[3];
                            float myMaxY = myPrintArea[1] + myPrintArea[3];
                            if (myPrintArea[0] >= printArea[0] && myMaxX <= maxX && myPrintArea[1] >= printArea[1] && myMaxY <= maxY)
                            {
                                printArea[0] = myPrintArea[0];      // X
                                printArea[2] = myPrintArea[2];      // width
                                printArea[1] = myPrintArea[1];      // Y
                                printArea[3] = myPrintArea[3];      // height
                            }
                        }
                    }
                    else
                    {
                        // this case should never happen
                        printArea = new float[4];
                        printArea[0] = 0;
                        printArea[1] = 0;
                        printArea[2] = (float) paper.getWidth();
                        printArea[3] = (float) paper.getHeight();
                    }

                    // Paper diminensions are in 1/72 of an inch.  The print area is borderless unless overridden by
                    // user with smaller diminensions.
                    paper.setSize(paperSize[0] * 72, paperSize[1] * 72);
                    paper.setImageableArea(printArea[0] * 72, printArea[1] * 72, printArea[2] * 72, printArea[3] * 72);
                }
            }
            else if (myMedia instanceof MediaName || myMedia instanceof MediaTray)
            {
                // have to use default Paper setttings because API limitation only allows one to be specified
                // so can't specify MediaSizeName as well
            }
        }

        return paper;
    }

    /**
     * Now that the printer is known, verify that the media is supported on this printer.  If it
     * is supported, add it to the attributeSet, which must have already been initialized.
     *
     * @param printService
     * @throws CFPrintException
     */
    protected void setMedia(PrintService printService, String media)
            throws CFPrintException
    {
        if(media==null){
            Object obj = printService.getDefaultAttributeValue(Media.class);
            if(obj==null)
                media = "iso-a4";
        }
        if (media != null)
        {
            Object obj = printService.getSupportedAttributeValues(Media.class, docFlavor, attributeSet);
            Array mediaStrings = new Array(1);
            if (obj != null && obj instanceof Media[])
            {
                Media[] mediaList = (Media[]) obj;
                for (int i = 0; i < mediaList.length; i++)
                {
                    mediaStrings.add(mediaList[i].toString());
                    if (mediaList[i].toString().equals(media))
                    {
                        // yup, the media specified is supported by the printer
                        attributeSet.add(mediaList[i]);
                        return;
                    }
                }
            }

            // If media isn't supported and we must maintain fidelity, it's an error.
            if (fidelity)
                throw new CFPrintInvalidMediaException(printService.getName(), media, ArrayUtil.ArrayToList(mediaStrings, ","));
        }
    }

    Media[] getSupportedMedia(PrinterJob printJob)
    {
        Object obj = printJob.getPrintService().getSupportedAttributeValues(Media.class, docFlavor, null);
        return (obj instanceof Media[]) ? (Media[]) obj : null;
    }

    MediaPrintableArea[] getSupportedMediaPrintableArea(PrintService service)
    {
        Object obj = service.getSupportedAttributeValues(MediaPrintableArea.class, docFlavor, attributeSet);
        return (obj instanceof MediaPrintableArea[]) ? (MediaPrintableArea[]) obj : null;
    }

    public PrintRequestAttributeSet getAttributeSet()
    {
        return attributeSet;
    }

    public void setAttributeSet(PrintRequestAttributeSet attributeSet)
    {
        this.attributeSet = attributeSet;
    }

    public DocFlavor getDocFlavor()
    {
        return docFlavor;
    }

    public String getPrinter()
    {
        return this.printer;
    }

    public void setPrinter(String printer)
    {
        this.printer = printer;
    }


    class PrintJobWatcher extends PrintJobAdapter
    {
        // true iff it is safe to close the print docPrintJob's input stream
        private boolean done = false;
        int lastEvent = 0;
        private String printer;
        private String jobName;
        private int totalPages;

        PrintJobWatcher(DocPrintJob docPrintJob, String jobName, int totalPages)
        {
            this.jobName = jobName;
            this.printer = docPrintJob.getPrintService().getName();
            this.totalPages = totalPages;
            docPrintJob.addPrintJobListener(this);
            logger.info(RB.getString(this, "cfprint.PrintJobStart", printer, jobName));
        }

        public void printDataTransferCompleted(PrintJobEvent pje)
        {
            lastEvent = pje.getPrintEventType();
            logger.info(RB.getString(this, "cfprint.PrintJobDataTransferCompleted", printer, jobName, String.valueOf(totalPages)));
        }

        public void printJobCanceled(PrintJobEvent pje)
        {
            // The print docPrintJob was cancelled
            lastEvent = pje.getPrintEventType();
            logger.info(RB.getString(this, "cfprint.PrintJobCanceled", printer, jobName));
            allDone();
        }

        public void printJobCompleted(PrintJobEvent pje)
        {
            // The print docPrintJob was completed
            lastEvent = pje.getPrintEventType();
            logger.info(RB.getString(this, "cfprint.PrintJobCompleted", printer, jobName));
            allDone();
        }

        public void printJobFailed(PrintJobEvent pje)
        {
            // The print docPrintJob has failed
            lastEvent = pje.getPrintEventType();
            logger.error(RB.getString(this, "cfprint.PrintJobFailed", printer, jobName));
            allDone();
        }

        public void printJobNoMoreEvents(PrintJobEvent pje)
        {
            // No more events will be delivered from this
            // print service for this print docPrintJob.
            // This event is fired in cases where the print service
            // is not able to determine when the docPrintJob completes.
            if (lastEvent == 0 || logger.isDebugEnabled())
                logger.info(RB.getString(this, "cfprint.PrintJobNoMoreEvents", printer, jobName));
            allDone();
        }

        public void printJobRequiresAttention(PrintJobEvent pje)
        {
            // The print service requires some attention to repair
            // some problem. E.g. running out of paper would
            // cause this event to be fired.
            lastEvent = pje.getPrintEventType();
            logger.error(RB.getString(this, "cfprint.PrintJobRequiresAttention", printer));
        }

        void allDone()
        {
            synchronized (PrintJobWatcher.this)
            {
                done = true;
                PrintJobWatcher.this.notify();
            }
        }

        // timeout in seconds
        public synchronized void waitForDone()
        {
            try
            {
                while (!done)
                {
                    wait();
                }
            }
            catch (InterruptedException e)
            {
            }
        }
    }
}
