/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.cloud.api.collections;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.solr.cloud.api.collections.CollApiCmds;
import org.apache.solr.cloud.api.collections.CollectionCommandContext;
import org.apache.solr.common.SolrException;
import org.apache.solr.common.cloud.ClusterState;
import org.apache.solr.common.cloud.ZkNodeProps;
import org.apache.solr.common.util.NamedList;
import org.apache.solr.common.util.SimpleOrderedMap;
import org.apache.solr.core.CoreContainer;
import org.apache.solr.core.backup.AggregateBackupStats;
import org.apache.solr.core.backup.BackupFilePaths;
import org.apache.solr.core.backup.BackupId;
import org.apache.solr.core.backup.BackupProperties;
import org.apache.solr.core.backup.ShardBackupId;
import org.apache.solr.core.backup.ShardBackupMetadata;
import org.apache.solr.core.backup.repository.BackupRepository;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DeleteBackupCmd
implements CollApiCmds.CollectionApiCommand {
    private static final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private final CollectionCommandContext ccc;

    public DeleteBackupCmd(CollectionCommandContext ccc) {
        this.ccc = ccc;
    }

    @Override
    public void call(ClusterState state, ZkNodeProps message, NamedList<Object> results) throws Exception {
        String backupLocation = message.getStr("location");
        String backupName = message.getStr("name");
        String repo = message.getStr("repository");
        int backupId = message.getInt("backupId", Integer.valueOf(-1));
        int lastNumBackupPointsToKeep = message.getInt("maxNumBackupPoints", Integer.valueOf(-1));
        boolean purge = message.getBool("purgeUnused", false);
        if (backupId == -1 && lastNumBackupPointsToKeep == -1 && !purge) {
            throw new SolrException(SolrException.ErrorCode.BAD_REQUEST, String.format(Locale.ROOT, "%s, %s or %s param must be provided", "backupId", "maxNumBackupPoints", "purgeUnused"));
        }
        CoreContainer cc = this.ccc.getCoreContainer();
        try (BackupRepository repository = cc.newBackupRepository(repo);){
            URI location = repository.createDirectoryURI(backupLocation);
            URI backupPath = BackupFilePaths.buildExistingBackupLocationURI(repository, location, backupName);
            if (repository.exists(repository.resolve(backupPath, "backup.properties"))) {
                throw new SolrException(SolrException.ErrorCode.BAD_REQUEST, "The backup name [" + backupName + "] at location [" + location + "] holds a non-incremental (legacy) backup, but backup-deletion is only supported on incremental backups");
            }
            if (purge) {
                this.purge(repository, backupPath, results);
            } else if (backupId != -1) {
                this.deleteBackupId(repository, backupPath, backupId, results);
            } else {
                this.keepNumberOfBackup(repository, backupPath, lastNumBackupPointsToKeep, results);
            }
        }
    }

    void purge(BackupRepository repository, URI backupPath, NamedList<Object> result) throws IOException {
        PurgeGraph purgeGraph = new PurgeGraph();
        purgeGraph.build(repository, backupPath);
        BackupFilePaths backupPaths = new BackupFilePaths(repository, backupPath);
        repository.delete(backupPaths.getIndexDir(), purgeGraph.indexFileDeletes);
        repository.delete(backupPaths.getShardBackupMetadataDir(), purgeGraph.shardBackupMetadataDeletes);
        repository.delete(backupPath, purgeGraph.backupIdDeletes);
        NamedList details = new NamedList();
        details.add("numBackupIds", (Object)purgeGraph.backupIdDeletes.size());
        details.add("numShardBackupIds", (Object)purgeGraph.shardBackupMetadataDeletes.size());
        details.add("numIndexFiles", (Object)purgeGraph.indexFileDeletes.size());
        result.add("deleted", (Object)details);
    }

    void keepNumberOfBackup(BackupRepository repository, URI backupPath, int maxNumBackup, NamedList<Object> results) throws Exception {
        List<BackupId> backupIds = BackupFilePaths.findAllBackupIdsFromFileListing(repository.listAllOrEmpty(backupPath));
        if (backupIds.size() <= maxNumBackup) {
            return;
        }
        Collections.sort(backupIds);
        List<BackupId> backupIdDeletes = backupIds.subList(0, backupIds.size() - maxNumBackup);
        this.deleteBackupIds(backupPath, repository, new HashSet<BackupId>(backupIdDeletes), results);
    }

    void deleteBackupIds(URI backupUri, BackupRepository repository, Set<BackupId> backupIdsDeletes, NamedList<Object> results) throws IOException {
        BackupFilePaths incBackupFiles = new BackupFilePaths(repository, backupUri);
        URI shardBackupMetadataDir = incBackupFiles.getShardBackupMetadataDir();
        HashSet<String> referencedIndexFiles = new HashSet<String>();
        ArrayList<ShardBackupId> shardBackupIdFileDeletes = new ArrayList<ShardBackupId>();
        List shardBackupIds = Arrays.stream(repository.listAllOrEmpty(shardBackupMetadataDir)).map(sbi -> ShardBackupId.fromShardMetadataFilename(sbi)).collect(Collectors.toList());
        for (ShardBackupId shardBackupId : shardBackupIds) {
            BackupId backupId = shardBackupId.getContainingBackupId();
            if (backupIdsDeletes.contains(backupId)) {
                shardBackupIdFileDeletes.add(shardBackupId);
                continue;
            }
            ShardBackupMetadata shardBackupMetadata = ShardBackupMetadata.from(repository, shardBackupMetadataDir, shardBackupId);
            if (shardBackupMetadata == null) continue;
            referencedIndexFiles.addAll(shardBackupMetadata.listUniqueFileNames());
        }
        HashMap<BackupId, AggregateBackupStats> backupIdToCollectionBackupPoint = new HashMap<BackupId, AggregateBackupStats>();
        ArrayList<String> unusedFiles = new ArrayList<String>();
        for (ShardBackupId shardBackupIdToDelete : shardBackupIdFileDeletes) {
            BackupId backupId = shardBackupIdToDelete.getContainingBackupId();
            ShardBackupMetadata shardBackupMetadata = ShardBackupMetadata.from(repository, shardBackupMetadataDir, shardBackupIdToDelete);
            if (shardBackupMetadata == null) continue;
            backupIdToCollectionBackupPoint.putIfAbsent(backupId, new AggregateBackupStats());
            ((AggregateBackupStats)backupIdToCollectionBackupPoint.get(backupId)).add(shardBackupMetadata);
            for (String uniqueIndexFile : shardBackupMetadata.listUniqueFileNames()) {
                if (referencedIndexFiles.contains(uniqueIndexFile)) continue;
                unusedFiles.add(uniqueIndexFile);
            }
        }
        repository.delete(incBackupFiles.getShardBackupMetadataDir(), shardBackupIdFileDeletes.stream().map(ShardBackupId::getBackupMetadataFilename).collect(Collectors.toList()));
        repository.delete(incBackupFiles.getIndexDir(), unusedFiles);
        try {
            for (BackupId backupId : backupIdsDeletes) {
                repository.deleteDirectory(repository.resolveDirectory(backupUri, BackupFilePaths.getZkStateDir(backupId)));
            }
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
        this.addResult(backupUri, repository, backupIdsDeletes, backupIdToCollectionBackupPoint, results);
        repository.delete(backupUri, backupIdsDeletes.stream().map(id -> BackupFilePaths.getBackupPropsName(id)).collect(Collectors.toList()));
    }

    private void addResult(URI backupPath, BackupRepository repository, Set<BackupId> backupIdDeletes, Map<BackupId, AggregateBackupStats> backupIdToCollectionBackupPoint, NamedList<Object> results) {
        String collectionName = null;
        ArrayList<SimpleOrderedMap> shardBackupIdDetails = new ArrayList<SimpleOrderedMap>();
        results.add("deleted", shardBackupIdDetails);
        for (BackupId backupId : backupIdDeletes) {
            SimpleOrderedMap backupIdResult = new SimpleOrderedMap();
            try {
                BackupProperties props = BackupProperties.readFrom(repository, backupPath, BackupFilePaths.getBackupPropsName(backupId));
                backupIdResult.add("startTime", (Object)props.getStartTime());
                if (collectionName == null) {
                    collectionName = props.getCollection();
                    results.add("collection", (Object)collectionName);
                }
            }
            catch (IOException props) {
                // empty catch block
            }
            AggregateBackupStats cbp = backupIdToCollectionBackupPoint.getOrDefault(backupId, new AggregateBackupStats());
            backupIdResult.add("backupId", (Object)backupId.getId());
            backupIdResult.add("size", (Object)cbp.getTotalSize());
            backupIdResult.add("numFiles", (Object)cbp.getNumFiles());
            shardBackupIdDetails.add(backupIdResult);
        }
    }

    private void deleteBackupId(BackupRepository repository, URI backupPath, int bid, NamedList<Object> results) throws Exception {
        BackupId backupId = new BackupId(bid);
        if (!repository.exists(repository.resolve(backupPath, BackupFilePaths.getBackupPropsName(backupId)))) {
            throw new SolrException(SolrException.ErrorCode.BAD_REQUEST, "Backup ID [" + bid + "] not found; cannot be deleted");
        }
        this.deleteBackupIds(backupPath, repository, Collections.singleton(backupId), results);
    }

    static final class PurgeGraph {
        Map<String, Node> backupIdNodeMap = new HashMap<String, Node>();
        Map<String, Node> shardBackupMetadataNodeMap = new HashMap<String, Node>();
        Map<String, Node> indexFileNodeMap = new HashMap<String, Node>();
        List<String> backupIdDeletes = new ArrayList<String>();
        List<String> shardBackupMetadataDeletes = new ArrayList<String>();
        List<String> indexFileDeletes = new ArrayList<String>();

        PurgeGraph() {
        }

        public void build(BackupRepository repository, URI backupPath) throws IOException {
            BackupFilePaths backupPaths = new BackupFilePaths(repository, backupPath);
            this.buildLogicalGraph(repository, backupPaths);
            this.findDeletableNodes(repository, backupPaths);
        }

        public void findDeletableNodes(BackupRepository repository, BackupFilePaths backupPaths) {
            this.visitExistingNodes(repository.listAllOrEmpty(backupPaths.getShardBackupMetadataDir()), this.shardBackupMetadataNodeMap, this.shardBackupMetadataDeletes);
            this.visitExistingNodes(repository.listAllOrEmpty(backupPaths.getIndexDir()), this.indexFileNodeMap, this.indexFileDeletes);
            this.shardBackupMetadataNodeMap.values().forEach(Node::propagateNotExisting);
            this.indexFileNodeMap.values().forEach(Node::propagateNotExisting);
            this.addDeleteNodesToQueue(this.backupIdNodeMap, this.backupIdDeletes);
            this.addDeleteNodesToQueue(this.shardBackupMetadataNodeMap, this.shardBackupMetadataDeletes);
            this.addDeleteNodesToQueue(this.indexFileNodeMap, this.indexFileDeletes);
        }

        private void visitExistingNodes(String[] existingNodeKeys, Map<String, Node> nodeMap, List<String> deleteQueue) {
            for (String nodeKey : existingNodeKeys) {
                Node node = nodeMap.get(nodeKey);
                if (node == null) {
                    deleteQueue.add(nodeKey);
                    continue;
                }
                node.existing = true;
            }
        }

        private <T> void addDeleteNodesToQueue(Map<T, Node> tNodeMap, List<T> deleteQueue) {
            tNodeMap.forEach((key, value) -> {
                if (value.delete) {
                    deleteQueue.add(key);
                }
            });
        }

        Node getBackupIdNode(String backupPropsName) {
            return this.backupIdNodeMap.computeIfAbsent(backupPropsName, bid -> {
                Node node = new Node();
                node.existing = true;
                return node;
            });
        }

        Node getShardBackupIdNode(String shardBackupId) {
            return this.shardBackupMetadataNodeMap.computeIfAbsent(shardBackupId, s -> new Node());
        }

        Node getIndexFileNode(String indexFile) {
            return this.indexFileNodeMap.computeIfAbsent(indexFile, s -> new IndexFileNode());
        }

        void addEdge(Node node1, Node node2) {
            node1.addNeighbor(node2);
            node2.addNeighbor(node1);
        }

        private void buildLogicalGraph(BackupRepository repository, BackupFilePaths backupPaths) throws IOException {
            URI baseBackupPath = backupPaths.getBackupLocation();
            List<BackupId> backupIds = BackupFilePaths.findAllBackupIdsFromFileListing(repository.listAllOrEmpty(baseBackupPath));
            for (BackupId backupId : backupIds) {
                BackupProperties backupProps = BackupProperties.readFrom(repository, baseBackupPath, BackupFilePaths.getBackupPropsName(backupId));
                Node backupIdNode = this.getBackupIdNode(BackupFilePaths.getBackupPropsName(backupId));
                for (String shardBackupMetadataFilename : backupProps.getAllShardBackupMetadataFiles()) {
                    Node shardBackupMetadataNode = this.getShardBackupIdNode(shardBackupMetadataFilename);
                    this.addEdge(backupIdNode, shardBackupMetadataNode);
                    ShardBackupMetadata shardBackupMetadata = ShardBackupMetadata.from(repository, backupPaths.getShardBackupMetadataDir(), ShardBackupId.fromShardMetadataFilename(shardBackupMetadataFilename));
                    if (shardBackupMetadata == null) continue;
                    for (String indexFile : shardBackupMetadata.listUniqueFileNames()) {
                        Node indexFileNode = this.getIndexFileNode(indexFile);
                        this.addEdge(indexFileNode, shardBackupMetadataNode);
                    }
                }
            }
        }
    }

    static final class IndexFileNode
    extends Node {
        int refCount = 0;

        IndexFileNode() {
        }

        @Override
        void addNeighbor(Node node) {
            super.addNeighbor(node);
            ++this.refCount;
        }

        @Override
        void propagateDelete() {
            if (this.delete || !this.existing) {
                return;
            }
            --this.refCount;
            if (this.refCount == 0) {
                this.delete = true;
            }
        }
    }

    static class Node {
        List<Node> neighbors;
        boolean delete = false;
        boolean existing = false;

        Node() {
        }

        void addNeighbor(Node node) {
            if (this.neighbors == null) {
                this.neighbors = new ArrayList<Node>();
            }
            this.neighbors.add(node);
        }

        void propagateNotExisting() {
            if (this.existing) {
                return;
            }
            if (this.neighbors != null) {
                this.neighbors.forEach(Node::propagateDelete);
            }
        }

        void propagateDelete() {
            if (this.delete || !this.existing) {
                return;
            }
            this.delete = true;
            if (this.neighbors != null) {
                this.neighbors.forEach(Node::propagateDelete);
            }
        }
    }
}

