/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.core.internal.http.pipeline.stages;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.SdkStandardLogger;
import software.amazon.awssdk.core.async.AsyncRequestBody;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.exception.NonRetryableException;
import software.amazon.awssdk.core.exception.SdkException;
import software.amazon.awssdk.core.interceptor.SdkInternalExecutionAttribute;
import software.amazon.awssdk.core.internal.Response;
import software.amazon.awssdk.core.internal.http.HttpClientDependencies;
import software.amazon.awssdk.core.internal.http.RequestExecutionContext;
import software.amazon.awssdk.core.internal.http.TransformingAsyncResponseHandler;
import software.amazon.awssdk.core.internal.http.pipeline.RequestPipeline;
import software.amazon.awssdk.core.internal.retry.ClockSkewAdjuster;
import software.amazon.awssdk.core.internal.retry.RetryHandler;
import software.amazon.awssdk.core.internal.util.CapacityManager;
import software.amazon.awssdk.core.internal.util.ThrowableUtils;
import software.amazon.awssdk.core.retry.RetryPolicy;
import software.amazon.awssdk.http.SdkHttpFullRequest;
import software.amazon.awssdk.http.SdkHttpFullResponse;
import software.amazon.awssdk.http.SdkHttpResponse;
import software.amazon.awssdk.utils.CompletableFutureUtils;

@SdkInternalApi
public final class AsyncRetryableStage<OutputT>
implements RequestPipeline<SdkHttpFullRequest, CompletableFuture<Response<OutputT>>> {
    private static final Logger log = LoggerFactory.getLogger(AsyncRetryableStage.class);
    private final TransformingAsyncResponseHandler<OutputT> responseHandler;
    private final RequestPipeline<SdkHttpFullRequest, CompletableFuture<Response<OutputT>>> requestPipeline;
    private final ScheduledExecutorService scheduledExecutor;
    private final HttpClientDependencies dependencies;
    private final CapacityManager retryCapacity;
    private final RetryPolicy retryPolicy;

    public AsyncRetryableStage(TransformingAsyncResponseHandler<OutputT> responseHandler, HttpClientDependencies dependencies, RequestPipeline<SdkHttpFullRequest, CompletableFuture<Response<OutputT>>> requestPipeline) {
        this.responseHandler = responseHandler;
        this.dependencies = dependencies;
        this.scheduledExecutor = dependencies.clientConfiguration().option(SdkClientOption.SCHEDULED_EXECUTOR_SERVICE);
        this.retryPolicy = dependencies.clientConfiguration().option(SdkClientOption.RETRY_POLICY);
        this.retryCapacity = dependencies.retryCapacity();
        this.requestPipeline = requestPipeline;
    }

    @Override
    public CompletableFuture<Response<OutputT>> execute(SdkHttpFullRequest request, RequestExecutionContext context) throws Exception {
        return new RetryExecutor(request, context).execute();
    }

    private class RetryExecutor {
        private final SdkHttpFullRequest request;
        private final RequestExecutionContext context;
        private final RetryHandler retryHandler;
        private final AsyncRequestBody originalRequestBody;
        private int requestCount = 0;

        private RetryExecutor(SdkHttpFullRequest request, RequestExecutionContext context) {
            this.request = request;
            this.context = context;
            this.originalRequestBody = context.requestProvider();
            this.retryHandler = new RetryHandler(AsyncRetryableStage.this.retryPolicy, AsyncRetryableStage.this.retryCapacity);
        }

        public CompletableFuture<Response<OutputT>> execute() throws Exception {
            CompletableFuture future = new CompletableFuture();
            return this.execute(future);
        }

        public CompletableFuture<Response<OutputT>> execute(CompletableFuture<Response<OutputT>> future) throws Exception {
            this.beforeExecute();
            CompletableFuture executeFuture = this.doExecute();
            executeFuture.whenComplete((resp, err) -> this.retryIfNeeded(future, (Response)resp, (Throwable)err));
            return CompletableFutureUtils.forwardExceptionTo(future, executeFuture);
        }

        private void retryIfNeeded(CompletableFuture<Response<OutputT>> future, Response<OutputT> resp, Throwable err) {
            if (future.isDone()) {
                return;
            }
            try {
                if (resp != null) {
                    this.retryResponseIfNeeded(resp, future);
                } else {
                    if (err instanceof CompletionException) {
                        err = err.getCause();
                    }
                    SdkException sdkException = ThrowableUtils.asSdkException(err);
                    this.retryErrorIfNeeded(sdkException, future);
                }
            }
            catch (Throwable t) {
                future.completeExceptionally(t);
            }
        }

        private void retryResponseIfNeeded(Response<OutputT> resp, CompletableFuture<Response<OutputT>> future) {
            if (resp.isSuccess()) {
                this.retryHandler.releaseRetryCapacity();
                future.complete(resp);
                return;
            }
            SdkException err = resp.exception();
            ClockSkewAdjuster clockSkewAdjuster = AsyncRetryableStage.this.dependencies.clockSkewAdjuster();
            if (clockSkewAdjuster.shouldAdjust(err)) {
                AsyncRetryableStage.this.dependencies.updateTimeOffset(clockSkewAdjuster.getAdjustmentInSeconds((SdkHttpResponse)resp.httpResponse()));
            }
            if (this.shouldRetry(resp.httpResponse(), resp.exception())) {
                AsyncRetryableStage.this.responseHandler.onError(err);
                this.retryHandler.setLastRetriedException(err);
                this.executeRetry(future);
            } else {
                future.completeExceptionally(err);
            }
        }

        private void retryErrorIfNeeded(SdkException err, CompletableFuture<Response<OutputT>> future) {
            if (err instanceof NonRetryableException) {
                future.completeExceptionally(err);
                return;
            }
            if (this.shouldRetry(null, err)) {
                AsyncRetryableStage.this.responseHandler.onError(err);
                this.retryHandler.setLastRetriedException(err);
                this.executeRetry(future);
            } else {
                future.completeExceptionally(err);
            }
        }

        private boolean shouldRetry(SdkHttpFullResponse httpResponse, SdkException exception) {
            return this.retryHandler.shouldRetry(httpResponse, this.request, this.context, exception, this.requestCount);
        }

        private void executeRetry(CompletableFuture<Response<OutputT>> future) {
            Duration delay = this.retryHandler.computeDelayBeforeNextRetry();
            SdkStandardLogger.REQUEST_LOGGER.debug(() -> "Retryable error detected, will retry in " + delay.toMillis() + "ms, attempt number " + this.requestCount);
            AsyncRetryableStage.this.scheduledExecutor.schedule(() -> {
                this.execute(future);
                return null;
            }, delay.toMillis(), TimeUnit.MILLISECONDS);
        }

        private void beforeExecute() {
            this.retryHandler.retryCapacityConsumed(false);
            ++this.requestCount;
        }

        private CompletableFuture<Response<OutputT>> doExecute() throws Exception {
            SdkStandardLogger.REQUEST_LOGGER.debug(() -> (this.retryHandler.isRetry() ? "Retrying " : "Sending ") + "Request: " + this.request);
            this.context.requestProvider(this.originalRequestBody);
            if (this.requestCount > 1 && this.context.asyncResponseTransformerFuture() != null) {
                this.context.executionAttributes().putAttribute(SdkInternalExecutionAttribute.ASYNC_RESPONSE_TRANSFORMER_FUTURE, null);
            }
            return (CompletableFuture)AsyncRetryableStage.this.requestPipeline.execute(this.retryHandler.addRetryInfoHeader(this.request, this.requestCount), this.context);
        }
    }
}

