/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package coldfusion.tagext.mail;

import coldfusion.runtime.ApplicationException;
import coldfusion.tagext.GenericTag;
import coldfusion.tagext.net.*;
import static coldfusion.mail.core.MailExceptions.*;

import java.util.Date;
import java.util.HashMap;

import javax.mail.Flags;
import javax.mail.Flags.Flag;
import javax.mail.Message;
import javax.mail.search.AndTerm;
import javax.mail.search.ComparisonTerm;
import javax.mail.search.FlagTerm;
import javax.mail.search.FromStringTerm;
import javax.mail.search.ReceivedDateTerm;
import javax.mail.search.RecipientStringTerm;
import javax.mail.search.SearchTerm;
import javax.mail.search.SentDateTerm;
import javax.mail.search.SubjectTerm;

/**
 * Handles Filter creation and validation for Pop and IMap tags.
 * 
 * @author shilpik
 * 
 */
public abstract class MsgTag extends GenericTag
{
    private static final long serialVersionUID = 1L;

    protected static final String KEY_FROMID = "FROM";
    protected static final String KEY_TOID = "TO";
    protected static final String KEY_SUBJECT = "SUBJECT";
    protected static final String KEY_FLAG = "FLAG";
    protected static final String KEY_TIMERECEIVED = "TIMERECEIVED";
    protected static final String KEY_TIMESENT = "TIMESENT";

    protected static final String ANSWERED = "ANSWERED"; // Messages with the Answered flag set.
    protected static final String DELETED = "DELETED"; // Messages with the Deleted flag set.
    protected static final String DRAFT = "DRAFT"; // Messages with the Draft flag set.
    protected static final String FLAGGED = "FLAGGED"; // Messages with the Flagged flag set.
    protected static final String SEEN = "SEEN"; // Messages that have the Seen flag set.
    protected static final String RECENT = "RECENT"; // Messages that have the Recent flag set.
    protected static final String NEW = "NEW"; // Messages that have the Recent flag set but not the Seen flag.
    protected static final String OLD = "OLD";  // Messages that do not have the Recent flag set. This is functionally equivalent to "!RECENT" (as opposed to "!NEW").
    protected static final String UNANSWERED = "UNANSWERED";  // Messages that do not have the Answered flag set.
    protected static final String UNDELETED = "UNDELETED";  // Messages that do not have the Deleted flag set.
    protected static final String UNDRAFT = "UNDRAFT";  // Messages that do not have the Draft flag set.
    protected static final String UNFLAGGED = "UNFLAGGED";  // Messages that do not have the Flagged flag set.
    protected static final String UNSEEN = "UNSEEN";  // Messages that do not have the Seen flag set.

    protected static final String KEY_FROM = "from";
    protected static final String KEY_TO = "to";
    protected static final String KEY_VALUE = "value";
    protected static final String KEY_NAME = "name";

    protected static final String COMMA = ",";

    protected static HashMap<String, String> validFilterAttributes = new HashMap<String, String>();
    protected static HashMap<String, String> validFlagFilterValus = new HashMap<String, String>();

    static
    {
        // add the message attributes to the validMessageAttributes set
        validFilterAttributes.put(KEY_FROMID, KEY_FROMID);
        validFilterAttributes.put(KEY_TOID, KEY_TOID);
        validFilterAttributes.put(KEY_SUBJECT, KEY_SUBJECT);
        validFilterAttributes.put(KEY_TIMERECEIVED, KEY_TIMERECEIVED);
        validFilterAttributes.put(KEY_TIMESENT, KEY_TIMESENT);
        validFilterAttributes.put(KEY_FLAG, KEY_FLAG);

        validFlagFilterValus.put(ANSWERED, ANSWERED);
        validFlagFilterValus.put(DELETED, DELETED);
        validFlagFilterValus.put(DRAFT, DRAFT);
        validFlagFilterValus.put(FLAGGED, FLAGGED);
        validFlagFilterValus.put(SEEN, SEEN);
        validFlagFilterValus.put(RECENT, RECENT);
        validFlagFilterValus.put(NEW, NEW);
        validFlagFilterValus.put(OLD, OLD);
        validFlagFilterValus.put(UNANSWERED, UNANSWERED);
        validFlagFilterValus.put(UNDELETED, UNDELETED);
        validFlagFilterValus.put(UNDRAFT, UNDRAFT);
        validFlagFilterValus.put(UNFLAGGED, UNFLAGGED);
        validFlagFilterValus.put(UNSEEN, UNSEEN);
    }

    protected static SearchTerm createFilters(MailFilterInfo filter, String tagName)
    {
        SearchTerm searchTerm = null;

        String value;
        if (null != (value = filter.getFromId()))
        {
            FromStringTerm fromTerm = new FromStringTerm(value);
            searchTerm = andTerm(searchTerm, fromTerm);
        }

        if (null != (value = filter.getToId()))
        {
            RecipientStringTerm toTerm = new RecipientStringTerm(Message.RecipientType.TO, value);
            searchTerm = andTerm(searchTerm, toTerm);
        }

        if (null != (value = filter.getSubject()))
        {
            SubjectTerm subjectTerm = new SubjectTerm(value);
            searchTerm = andTerm(searchTerm, subjectTerm);
        }
        
        if (null != (value = filter.getFlag()))
        {
            isValidFlagValue(value, tagName);
            value = value.toUpperCase();

            FlagTerm flagTerm = null;
            SearchTerm newFlagTerm = null;

            if (value.equals(ANSWERED))
            {
                flagTerm = new FlagTerm(new Flags(Flag.ANSWERED), true);
            }
            else if (value.equals(DELETED))
            {
                flagTerm = new FlagTerm(new Flags(Flag.DELETED), true);
            }
            else if (value.equals(DRAFT))
            {
                flagTerm = new FlagTerm(new Flags(Flag.DRAFT), true);
            }
            else if (value.equals(FLAGGED))
            {
                flagTerm = new FlagTerm(new Flags(Flag.FLAGGED), true);
            }
            else if (value.equals(SEEN))
            {
                flagTerm = new FlagTerm(new Flags(Flag.SEEN), true);
            }
            else if (value.equals(RECENT))
            {
                flagTerm = new FlagTerm(new Flags(Flag.RECENT), true);
            }
            else if (value.equals(NEW))
            {
                FlagTerm flagTermRecent = new FlagTerm(new Flags(Flag.RECENT), true);
                FlagTerm flagTermNotSeen = new FlagTerm(new Flags(Flag.SEEN), false);
                newFlagTerm = andTerm(flagTermRecent, flagTermNotSeen);
            }
            else if (value.equals(OLD))
            {
                flagTerm = new FlagTerm(new Flags(Flag.RECENT), false);
            }
            else if (value.equals(UNANSWERED))
            {
                flagTerm = new FlagTerm(new Flags(Flag.ANSWERED), false);
            }
            else if (value.equals(UNDELETED))
            {
                flagTerm = new FlagTerm(new Flags(Flag.DELETED), false);
            }
            else if (value.equals(UNDRAFT))
            {
                flagTerm = new FlagTerm(new Flags(Flag.DRAFT), false);
            }
            else if (value.equals(UNFLAGGED))
            {
                flagTerm = new FlagTerm(new Flags(Flag.FLAGGED), false);
            }
            else if (value.equals(UNSEEN))
            {
                flagTerm = new FlagTerm(new Flags(Flag.SEEN), false);
            }

            if (newFlagTerm != null)
            {
                searchTerm = andTerm(searchTerm, newFlagTerm);
            }
            else if (flagTerm != null)
            {
                searchTerm = andTerm(searchTerm, flagTerm);
            }
        }

        Date tmpDate = filter.getFromTimeReceived();
        if (null != tmpDate)
        {
            ReceivedDateTerm fromReceivedDateTerm = new ReceivedDateTerm(ComparisonTerm.GE, tmpDate);

            tmpDate = filter.getToTimeReceived();
            ReceivedDateTerm toReceivedDateTerm = null;
            if (null != tmpDate)
            {
                toReceivedDateTerm = new ReceivedDateTerm(ComparisonTerm.LT, tmpDate);
            }
            else
            {
                // if no end date given search till today.
                toReceivedDateTerm = new ReceivedDateTerm(ComparisonTerm.LT, new Date(System.currentTimeMillis()));
            }

            SearchTerm dateTerm = new AndTerm(fromReceivedDateTerm, toReceivedDateTerm);
            searchTerm = andTerm(searchTerm, dateTerm);
        }

        tmpDate = filter.getFromTimeSent();
        if (null != tmpDate)
        {
            SentDateTerm fromSentDateTerm = new SentDateTerm(ComparisonTerm.GE, tmpDate);

            tmpDate = filter.getToTimeSent();
            SentDateTerm toSentDateTerm = null;
            if (null != tmpDate)
            {
                toSentDateTerm = new SentDateTerm(ComparisonTerm.LT, tmpDate);
            }
            else
            {
                // if no end date given search till today.
                toSentDateTerm = new SentDateTerm(ComparisonTerm.LT, new Date(System.currentTimeMillis()));
            }

            SearchTerm dateTerm = new AndTerm(fromSentDateTerm, toSentDateTerm);
            searchTerm = andTerm(searchTerm, dateTerm);
        }
        return searchTerm;
    }

    /**
     * Create a SearchTerm that is AND of given search terms.
     * 
     * @param flagTermRecent
     * @param flagTermNotSeen
     * @return
     */
    private static SearchTerm andTerm(SearchTerm searchTerm, SearchTerm searchTerm2)
    {
        if (searchTerm != null)
        {
            searchTerm = new AndTerm(searchTerm, searchTerm2);
        }
        else
        {
            searchTerm = searchTerm2;
        }
        return searchTerm;
    }

    /**
     * Check if given attribute is allowed as a valid filter.
     * 
     * @param attribName
     */
    protected static boolean isValidAttributeName(String attribName, String tagName)
    {
        if (!validFilterAttributes.containsKey(attribName.toUpperCase()))
            throw new InvalidFilterAttributeKeyException(attribName, tagName);
        else
            return true;
    }

    /**
     * Check if given flag attribute value is valid.
     * 
     * @param attribName
     */
    protected static boolean isValidFlagValue(String attribValue, String tagName)
    {
        if (!validFlagFilterValus.containsKey(attribValue.toUpperCase()))
            throw new InvalidFlagFilterValueException(attribValue, tagName);
        else
            return true;
    }

    /**
     * {attribName} is an invalid filter attribute for the {tagName} tag.
     */
    public static class InvalidFilterAttributeKeyException extends ApplicationException
    {
        private static final long serialVersionUID = 1L;
        public String attribName;
        public String tagName;

        public InvalidFilterAttributeKeyException(String attribName, String tagName)
        {
            this.attribName = attribName;
            this.tagName = tagName;
        }
    }

    /**
     * {attribValue} is an invalid filter value for the {tagName} tag. Valid values are ANSWERED, DELETED, DRAFT,
     * FLAGGED, SEEN, RECENT, NEW, OLD, UNANSWERED, UNDELETED, UNDRAFT, UNFLAGGED, UNSEEN.
     */
    public static class InvalidFlagFilterValueException extends ApplicationException
    {
        private static final long serialVersionUID = 1L;
        public String attribValue;
        public String tagName;

        public InvalidFlagFilterValueException(String attribValue, String tagName)
        {
            this.attribValue = attribValue;
            this.tagName = tagName;
        }
    }

    /**
     * Value for the {attribName} attribute is invalid.
     */
    public static class InvalidAttributeValueException extends ApplicationException
    {
        private static final long serialVersionUID = 1L;
        public String attribName;

        public InvalidAttributeValueException(String attribName)
        {
            this.attribName = attribName;
        }
    }

    /**
     * Invalid attribute combination for the {tagName} tag. Valid combination for this attribute is {validCombination}.
     */
    public static class InvalidFilterAttribComboException extends ApplicationException
    {
        private static final long serialVersionUID = 1L;
        public String validCombination;
        public String tagName;

        public InvalidFilterAttribComboException(String tagName, String validCombo)
        {
            this.tagName = tagName;
            validCombination = validCombo;
        }
    }

    /**
     * The value attribute for {attribName} is not specified or is empty.
     */
    public class EmptyAttributeValueException extends ApplicationException
    {
        private static final long serialVersionUID = 1L;
        public String attribName;

        public EmptyAttributeValueException(String attribName)
        {
            this.attribName = attribName;
        }
    }

    /**
     * Value for the {attribName} attribute is invalid. Value must be a String.
     */
    public class InvalidStringValueException extends ApplicationException
    {
        private static final long serialVersionUID = 1L;
        public String attribName;

        public InvalidStringValueException(String name)
        {
            attribName = name;
        }
    }

    /**
     * Delimiter should be used with UID attirbute only for {tagName}.
     */
    public class IncorrectUseOfDelimiterException extends ApplicationException
    {
        private static final long serialVersionUID = 1L;
        public String tagName;

        public IncorrectUseOfDelimiterException(String tagName)
        {
            this.tagName = tagName;
        }
    }

}
