/*
 * Copyright (c) 1995 - 2003 Macromedia, Inc. All rights reserved.
*/
package coldfusion.mail.mod;

import javax.mail.Part;
import javax.mail.MessagingException;
import javax.mail.internet.ContentDisposition;
import javax.mail.internet.ContentType;
import javax.mail.internet.MimeUtility;
import javax.mail.internet.HeaderTokenizer;
import java.io.UnsupportedEncodingException;

/**
 * RFC2231Util 
 *
 * Author: Shin Kinoshita
 * http://sk-jp.com/java/library/index.html
 * 
 */
public class RFC2231Util
{

	// Support non-ascii characters for attachment file
	// ref. RFC2231
	// These methods (setFileName, encodeParameter, isAllAscii) were got from Japanese JavaMail book.
	protected static void setFileName(Part part, String filename, String contentDispo, String charset) throws MessagingException {
		ContentDisposition disposition;
		String[] strings = part.getHeader("Content-Disposition");
		if (strings == null || strings.length < 1) {
			disposition = new ContentDisposition(getDisposition(contentDispo));
		} else {
			disposition = new ContentDisposition(getDisposition(contentDispo));
			disposition.getParameterList().remove("filename");
		}

		part.setHeader("Content-Disposition",
		        disposition.toString() +
		        encodeParameter("filename", filename, charset, null /* lang */));
		ContentType ctype;
		strings = part.getHeader("Content-Type");
		if (strings == null || strings.length < 1) {
			ctype = new ContentType(part.getDataHandler().getContentType());
		} else {
			ctype = new ContentType(strings[0]);
		}

		try {
			String mimeString = MimeUtility.encodeWord(filename, charset, "B");
			// cut <CRLF>
			StringBuilder sb = new StringBuilder();
			int i;
			while ((i = mimeString.indexOf('\r')) != -1) {
				sb.append(mimeString.substring(0, i));
				mimeString = mimeString.substring(i+2);
			}
			sb.append(mimeString);

			ctype.setParameter("name", new String(sb));
		} catch (UnsupportedEncodingException e) {
			throw new MessagingException("unsupported encoding", e);
		}
		part.setHeader("Content-Type", ctype.toString());
	}

	protected static String encodeParameter(String name, String value, String encoding , String lang) {
		StringBuilder result = new StringBuilder();
		StringBuilder encodedPart = new StringBuilder();
		
		boolean isAllAscii = isAllAscii(value);
		boolean CESWasWritten = false;
		boolean encoded;
		boolean needFolding = false;
		int sequenceNo = 0;
		int column;

		while (value.length() > 0) {
			// index of boundary of ascii/non ascii
			int lastIndex;
			boolean isAscii = value.charAt(0) < 0x80;
			for (lastIndex = 1; lastIndex < value.length(); lastIndex++) {
				if (value.charAt(lastIndex) < 0x80) {
					if (!isAscii) break;
				} else {
					if (isAscii) break;
				}
			}
			if (lastIndex != value.length()) needFolding = true;

			RETRY:      while (true) {
				encodedPart.setLength(0);
				String target = value.substring(0, lastIndex);

				byte[] bytes;
				try {
					if (isAscii) {
						bytes = target.getBytes("us-ascii");
					} else {
						bytes = target.getBytes(encoding);
					}
				} catch (UnsupportedEncodingException e) {
					bytes = target.getBytes(); // use default encoding
					encoding = MimeUtility.mimeCharset(
					        MimeUtility.getDefaultJavaCharset());
				}

				encoded = false;
				// It is not strict.
				column = name.length() + 7; // size of " " and "*nn*=" and ";"

				for (int i = 0; i < bytes.length; i++) {
					if (bytes[i] > ' ' && bytes[i] <= 'z'
					        && HeaderTokenizer.MIME.indexOf((char)bytes[i]) < 0) {
						        encodedPart.append((char)bytes[i]);
						        column++;
					        } else {
						        encoded = true;
						        encodedPart.append('%');
						        String hex  = Integer.toString(bytes[i] & 0xff, 16);
						        if (hex.length() == 1) {
							        encodedPart.append('0');
						        }
						        encodedPart.append(hex);
						        column += 3;
					        }
					if (column > 76) {
						needFolding = true;
						lastIndex /= 2;
						continue RETRY;
					}
				}

				result.append(";\r\n ").append(name);
				if (needFolding) {
					result.append('*').append(sequenceNo);
					sequenceNo++;
				}
				if(encoded || (!CESWasWritten && !isAllAscii)) {
                    result.append("*="); 
                    if(!CESWasWritten && !isAllAscii) {
                        result.append(encoding).append('\'');
                        if (lang != null) result.append(lang);
                        result.append('\'');
                        CESWasWritten =true;
                    }
				} else
				    result.append('=');
				
				result.append(new String(encodedPart));
				value = value.substring(lastIndex);
				break;
			}
		}
		return new String(result);
	}

	/** check if contains only ascii characters in text. */
	protected static boolean isAllAscii(String text) {
		for (int i = 0; i < text.length(); i++) {
			if (text.charAt(i) > 0x7f) { // non-ascii
				return false;
			}
		}
		return true;
	}
	
	private static String getDisposition(String contentDispo){
	    if(contentDispo != null && !contentDispo.trim().isEmpty())
	        return contentDispo;
	    return Part.ATTACHMENT;
	}
}