/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 2008 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and may be covered by U.S. and Foreign Patents,
* patents in process, and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/

package coldfusion.mail.mod;

import coldfusion.log.CFLogs;
import coldfusion.runtime.ApplicationException;
import coldfusion.runtime.OleDateTime;
import coldfusion.runtime.Struct;
import coldfusion.sql.QueryTable;
import coldfusion.sql.QueryTableMetaData;
import coldfusion.util.IOUtils;
import coldfusion.util.Utils;
import coldfusion.vfs.VFSFileFactory;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CodingErrorAction;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.mail.BodyPart;
import javax.mail.Flags;
import javax.mail.Folder;
import javax.mail.Header;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Part;
import javax.mail.internet.AddressException;
import javax.mail.internet.ContentDisposition;
import javax.mail.internet.ContentType;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimePart;
import javax.mail.internet.MimeUtility;
import javax.mail.internet.ParseException;

import com.sun.mail.imap.IMAPBodyPart;
import com.sun.mail.imap.IMAPMessage;
import com.sun.mail.util.MimeUtil;

/**
 * Encapsulate the result of a POP call in a Query table
 *
 */
class EmailTable extends QueryTable
{
    
    private static final boolean faultyServer = Boolean.getBoolean("coldfusion.mail.faultyserver");

    EmailTable()
    {
        super();
    }

    // Bug 50941 - CFMX used comma, CF5 uses TAB.
    private final String attachment_separator = "\t";

    private final String embedded_image_filename = "Embedded Image";
    protected String attachment_path;
    protected boolean unique_filenames = false;

    public void populate(Folder folder, String[] fields, Message[] msgs, String path, boolean unique, String accept)
            throws MessagingException
    {

        attachment_path = path;
        unique_filenames = unique;

        meta = new EmailTableMetaData(fields);
        col_count = meta.getColumnCount();
        col_names = meta.getColumnLabels();
        ensureCapacity(msgs.length);
        String data;
        Struct cidTable = null;

        // parse the tokens
        String[] accepted = null;
        if (attachment_path != null)
        {
            accepted = parseAccept(accept);
        }

        for (int i = 0; i < msgs.length; i += 1)
        {
            try
            {
                // Treat as MimeMessage's so we can use getHeader(name, delim)
                // which returns as strings, not InternetAddresses.
                MimeMessage cmsg = (MimeMessage) msgs[i];
                // Use the MimeMessage copy constructor to make a copy
                // of the entire message, which will fetch the entire
                // message from the server and parse it on the client:
                MimeMessage msg = cmsg;
                if(faultyServer)
                    msg = new MimeMessage(cmsg);
                Flags msgFlgs =msg.getFlags();

                Object[] col_data = new Object[col_count];
                for (int j = 0; j < col_count; j += 1)
                {

                    if ("date".equalsIgnoreCase(col_names[j]))
                    {
                        col_data[j] = msg.getHeader("Date", null);
                    }
                    else if ("from".equalsIgnoreCase(col_names[j]))
                    {
                        String from = msg.getHeader("From", ",");
                        if (from == null)
                            from = msg.getHeader("Sender", ",");
                        col_data[j] = decode(from);
                    }
                    else if ("messagenumber".equalsIgnoreCase(col_names[j]))
                    {
                        col_data[j] = (new Integer(cmsg.getMessageNumber()));
                    }
                    else if ("sentdate".equalsIgnoreCase(col_names[j]))
                    {
                        if(msg.getSentDate() != null) {
                        col_data[j] = new OleDateTime(msg.getSentDate()).toString();
                        }
                    }
                    else if ("rxddate".equalsIgnoreCase(col_names[j]))
                    {
                        col_data[j] = new OleDateTime(cmsg.getReceivedDate()).toString();
                    }
                    else if ("size".equalsIgnoreCase(col_names[j]))
                    {
                        col_data[j] = new Integer(msg.getSize());
                    }
                    else if ("lines".equalsIgnoreCase(col_names[j])) 
                    {
						try {
							if (cmsg.getLineCount() > -1)
								col_data[j] = (new Integer(cmsg.getLineCount()));
							else
								col_data[j] = new Integer(0);
						} catch (MessagingException me) {
							col_data[j] = new Integer(0);
						}
					}
                    else if ("answered".equalsIgnoreCase(col_names[j]))
                    {
                        if(msgFlgs.contains(Flags.Flag.ANSWERED))
                            col_data[j] = "yes";
                        else
                            col_data[j] = "no";
                    }
                    else if ("deleted".equalsIgnoreCase(col_names[j]))
                    {
                        if(msgFlgs.contains(Flags.Flag.DELETED))
                            col_data[j] = "yes";
                        else
                            col_data[j] = "no";
                    }
                    else if ("draft".equalsIgnoreCase(col_names[j]))
                    {
                        if(msgFlgs.contains(Flags.Flag.DRAFT))
                            col_data[j] = "yes";
                        else
                            col_data[j] = "no";
                    }
                    else if ("flagged".equalsIgnoreCase(col_names[j]))
                    {
                        if(msgFlgs.contains(Flags.Flag.FLAGGED))
                            col_data[j] = "yes";
                        else
                            col_data[j] = "no";
                    }
                    else if ("recent".equalsIgnoreCase(col_names[j]))
                    {
                        if(msgFlgs.contains(Flags.Flag.RECENT))
                            col_data[j] = "yes";
                        else
                            col_data[j] = "no";
                    }
                    else if ("seen".equalsIgnoreCase(col_names[j]))
                    {
                        if(msgFlgs.contains(Flags.Flag.SEEN))
                            col_data[j] = "yes";
                        else
                            col_data[j] = "no";
                    }
                    else if ("replyto".equalsIgnoreCase(col_names[j]))
                    {
                        // Get the replty-to header, if not set use From.
                        // Duplicating CFMX, which used msg.getReplyto(),
                        // but not CF5, which only reported Reply-To header.
                        String replyto = msg.getHeader("Reply-To", ",");
                        if (replyto == null)
                        {
                            replyto = msg.getHeader("From", ",");
                        }
                        if (replyto == null)
                        {
                            replyto = msg.getHeader("Sender", ",");
                        }
                        col_data[j] = decode(replyto);
                    }
                    else if ("subject".equalsIgnoreCase(col_names[j]))
                    {
                        col_data[j] = msg.getSubject();  // this does decode for us
                    }
                    else if ("cc".equalsIgnoreCase(col_names[j]))
                    {
                        col_data[j] = decode(msg.getHeader("CC", ","));
                    }
                    else if ("to".equalsIgnoreCase(col_names[j]))
                    {
                        col_data[j] = decode(msg.getHeader("To", ","));
                    }
                    else if ("header".equalsIgnoreCase(col_names[j]))
                    {
                        StringBuffer buf = new StringBuffer();
                        Enumeration headers = msg.getAllHeaders();
                        while (headers.hasMoreElements())
                        {
                            Header hdr = (Header) headers.nextElement();
                            buf.append(hdr.getName());
                            buf.append(": ");
                            buf.append(hdr.getValue());
                            buf.append("\r\n");  // bug 48392
                        }
                        col_data[j] = buf.toString().trim();
                    }
                    else if ("messageid".equalsIgnoreCase(col_names[j]))
                    {
                        col_data[j] = decode(msg.getHeader("Message-ID", ","));
                    }
                    else if ("uid".equalsIgnoreCase(col_names[j]))
                    {
                        String uid = "";
                        if (folder instanceof com.sun.mail.pop3.POP3Folder) {
                            com.sun.mail.pop3.POP3Folder pf = (com.sun.mail.pop3.POP3Folder)folder;
                            uid = pf.getUID(cmsg);
                        }
                        else if (folder instanceof com.sun.mail.imap.IMAPFolder)
                        {
                            com.sun.mail.imap.IMAPFolder pf = (com.sun.mail.imap.IMAPFolder)folder;
                            uid = ""+pf.getUID(cmsg);
                        }
                        col_data[j] = uid;
                    }
                    else if ("body".equalsIgnoreCase(col_names[j]))
                    {
                        try
                        {
                            col_data[j] = getPartBody(msg);
                        }
                        catch (MessagingException e)
                        {
                            // Be robust in the face of bad messages
                            col_data[j] = "";
                        }
                    }
                    else if ("textbody".equalsIgnoreCase(col_names[j]))
                    {
                        try
                        {
                            col_data[j] = getText(msg, "plain");
                        }
                        catch (MessagingException e)
                        {
                            // Be robust in the face of bad messages
                            col_data[j] = "";
                        }
                    }
                    else if ("htmlbody".equalsIgnoreCase(col_names[j]))
                    {
                        try
                        {
                            col_data[j] = getText(msg, "html");
                        }
                        catch (MessagingException e)
                        {
                            // Be robust in the face of bad messages
                            col_data[j] = "";
                        }
                    }
                    else if ("attachments".equalsIgnoreCase(col_names[j]))
                    {
                        data = getAttachmentName(msg);
                        if (data.startsWith(attachment_separator))
                        {
                            data = data.substring(1);
                        }
                        /*
                         * We construct the cid table here since we dont want to avoid reconstrucing the 
                         * data string - we have done this here already and it would be a waste to do it again
                         * for the cids column. Also removing cids from attachments.
                         */
                        cidTable = new Struct();
                        StringTokenizer st = new StringTokenizer(data,attachment_separator);
                        List<String> attachments = new ArrayList<String>();
                        while(st.hasMoreTokens()){
                            String token = st.nextToken();
                            int indx = token.indexOf(" [cid:");
                            if (indx != -1){
                                String key = token.substring(0,indx);
                                String value = token.substring(indx+6,token.length()-1);
                                cidTable.put(key,value.substring(0,value.length()));
                                attachments.add(key);
                            }
                            else
                                attachments.add(token);                               
                        }

                        col_data[j] = Utils.join(attachment_separator, attachments);

                    }
                    else if ("cids".equalsIgnoreCase(col_names[j]))
                    {
                        /*
                               * Use the cidTable that is generated while creating the info for "attachments" column
                               * so we don't waste time redoing everything again.
                               * 
                               * The processing to create cidTable is guaranteed to have run before this assignment since
                               * in the column names array that is passed to the populate() fn. in the class the 
                               * "attachments" string is before the "cids" string.
                               */
                        col_data[j] = cidTable;
                    }
                    else if ("attachmentfiles".equalsIgnoreCase(col_names[j]))
                    {
                        if (attachment_path != null)
                        {
                            data = getAttachmentPath(msg, attachment_path, unique_filenames, true, accept, accepted);

                            if (data.startsWith(attachment_separator))
                            {
                                data = data.substring(1);
                            }
                            col_data[j] = data;
                        }
                    }
                    else
                    {
                        // This gets the header with the specified name as a fall back
                        col_data[j] = decode(msg.getHeader(col_names[j], ","));
                    }
                    /* Not used, save the string compares
                    else if ("original".equalsIgnoreCase(col_names[j]))
                    {
                        col_data[j] = msg;
                    }
                    else if ("sender".equalsIgnoreCase(col_names[j]))
                    {
                        col_data[j] = decode(msg.getHeader("Sender", ","));
                    }
                    else if ("recipient".equalsIgnoreCase(col_names[j]))
                    {
                        col_data[j] = decode(msg.getHeader("To", ","));
                    }
                    else if ("size".equalsIgnoreCase(col_names[j]))
                    {
                        col_data[j] = new Integer(msg.getSize());
                    }
                    else if ("sentdate".equalsIgnoreCase(col_names[j]))
                    {
                        col_data[j] = msg.getSentDate();
                    }
                    else if ("receiveddate".equalsIgnoreCase(col_names[j]))
                    {
                        col_data[j] = msg.getReceivedDate();
                    }
                    */
                }
                addRow(col_data);
            }
            catch (Exception ex)
            {
                if (ex instanceof MessagingException)
                    throw (MessagingException) ex;
                else
                    throw new MessagingException(ex.getMessage(), ex);
            }
        }
    }

    /**
     * Utility to run text through the Mime decoder if not null.
     */
    private String decode(String s) throws Exception
    {
        if (s == null)
            return null;

        // An unfortunate special case where we have encoded
        // text inside quotes, which decodeText wont do the right
        // thing with.  Bug 49907.
        // Example: "=?iso-2022-jp?B?GyRCTD5MNSQ3JE44Iko8MVIbKEI=?=" <user@mail>
        if (s.indexOf("\"=?") != -1)
        {
            try
            {
                InternetAddress[] ia = InternetAddress.parse(s);
                return toString(ia);
            }
            catch (AddressException e)
            {
                // bad address, just return decoded string
                return MimeUtility.decodeText(s);
            }
        }

        return MimeUtility.decodeText(s);
    }

    /**
     * Utility used to convert an array of addresses to a decoded string
     */
    private String toString(Object[] array) throws Exception
    {
        StringBuffer buffer = new StringBuffer();
        if (array != null)
        {
            for (int i = 0; i < array.length; i += 1)
            {
                if (i != 0)
                {
                    buffer.append(',');
                }
                String addr = array[i].toString();
                buffer.append(MimeUtility.decodeText(addr));

            }
        }
        return buffer.toString();
    }



    /**
     * Original code that gets the first body part.
     * This could be replaced with a call to getText("*"), but that might be risky
     * as this version handles rfc822 message parts and 'unknown' parts.
     *
     * @param p the message
     * @return content of the first body part, or an empty string
     * @throws MessagingException
     */
    protected String getPartBody(Part p) throws MessagingException
    {

        if (p == null)
        {
            return "";
        }

        Object content;
        try
        {
            // Get the content checking for UTF-7 as well.
            content = getContentIncludingAdditionalCharsets(p);
        }
        catch (IOException e)
        {
            // probably an unsupported encoding exception,
            // but in any case don't blow up just skip this part
            return "Unable to retrieve message content: " + e.toString();
        }

        String filename = null;
        try
        {
            filename = getFilename(p);
        }
        catch (Exception e)
        {
        }


        if (p.isMimeType("text/*") && filename == null)
        {
            // Just in case something funny is going on
            if (content instanceof String)
            {
                return (String) content;
            }
            else
            {
                return "";
            }
        }
        else if (p.isMimeType("multipart/*"))
        {
            StringBuffer buffer = new StringBuffer();
            // be paranoid
            if (! (content instanceof Multipart))
                return "";
            Multipart mp = (Multipart) content;

            int count = mp.getCount();

            for (int i = 0; i < count; i += 1)
            {

                BodyPart bp = mp.getBodyPart(i);

                buffer.append(getPartBody(bp));

                //  Band-aid fix for 48431: bail out if we're
                //  looking at a multipart/alternative message,
                //  and this is the first part.
                // Also bug 49769 - handle multipart/report.
                if (p.isMimeType("multipart/alternative") || p.isMimeType("multipart/report"))
                {
                    break;
                }
            }
            return buffer.toString();
        }
        else if (p.isMimeType("message/rfc822"))
        {
            if (! (content instanceof Part))
                return "";
            return getPartBody((Part) content);
        }
        else
        {
            if (content instanceof String && filename == null)
            {
                return (String) content;
            }
            else if (content instanceof InputStream)
            {
                return "";
            }
            else if (filename == null)
            {
                return content.toString();
            }
            else
            {
                return "";
            }
        }
    }

    /**
     * Get the content from the part.  If a UnsupportedEncodingException is thrown check if it's
     * UTF-7.  As of Java 1.6, UTF-7 isn't a natively supported Charset.  Use the decoder from sourceforge jutf7.jar.
     * If this jar is placed in {JAVA_HOME}/jre/lib/ext then it becomes a "native" charset and this code
     * won't be reached because an UnsupportedEncodingException won't be thrown.
     * @param p mail part to get content from
     * @return the content - if charset not recognized will fall back to ascii
     */
    private Object getContentIncludingAdditionalCharsets(Part p) throws IOException, MessagingException
    {
        UnsupportedEncodingException savedException;
        try
        {
            return p.getContent();
        }
        catch (UnsupportedEncodingException e)
        {
            // See if we can do something with the content rather than return an error.
            savedException = e;
        }

        // Read in the content.
        InputStream in = p.getInputStream();
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        int c;

        while ((c = in.read()) != -1)
        {
            out.write(c);
        }
        byte bytes[] = out.toByteArray();

        ContentType ct = new ContentType(p.getContentType());
        String charset = ct.getParameter("charset");

        // If the charset is UTF-7 or one of it's aliases, attempt to decode the message.
        com.beetstra.jutf7.CharsetProvider provider = new com.beetstra.jutf7.CharsetProvider();
        Charset cs = provider.charsetForName(charset);
        if (cs != null)
        {
            CharsetDecoder d = cs.newDecoder();

            // Just do the best we can.
            d.onMalformedInput(CodingErrorAction.IGNORE);
            d.onUnmappableCharacter(CodingErrorAction.IGNORE);

            CharBuffer outBytes;
            try
            {
                outBytes = d.decode(ByteBuffer.wrap(bytes));
                return outBytes.toString();
            }
            catch (CharacterCodingException e)
            {
                // Just convert to ascii and hope for the best.
                return new String(bytes, "US-ASCII");
            }
        }

        // Still can't support the encoding so throw the original exception.
        throw savedException;
    }

    /**
     * Return the text body (if any) of this Part with the specified subtype (plain, html, etc).
     *
     * @param p the message
     * @param type the subtype of text we are looking for (plain, html, etc).
     * @return content of message part or null if not found.
     * @throws MessagingException
     */
    private String getText(Part p, String type) throws MessagingException
    {
        if (p == null)
        {
            return null;
        }

        // Check the disposition of this part, ignore if it is an attachment
        String disp = null;
        try
        {
            disp = p.getDisposition();
        }
        catch (MessagingException e)
        {
            // ignore this, as this is probably a malformed Content-Disposition header
        }
        if (disp != null && disp.equalsIgnoreCase(Part.ATTACHMENT))
        {
            return null;
        }

        try
        {
            // If the content is of type text/<type>, this is it
            String t = "text/" + type;
            if (p.isMimeType(t))
            {
                Object content;
                try
                {
                    // Get the content checking for UTF-7 as well.
                    content = getContentIncludingAdditionalCharsets(p);
                }
                catch (IOException e)
                {
                    // probably an unsupported encoding exception,
                    // but in any case don't blow up just skip this part
                    return "Unable to retrieve message content: " + e.toString();
                }

                if (content instanceof String)
                    return (String) content;
                else
                    return "";
            }

            // handle multipart messages
            if (p.isMimeType("multipart/*"))
            {
                // be paranoid
                Object content = p.getContent();
                if (! (content instanceof Multipart))
                    return "";
                Multipart mp = (Multipart) content;
                int count = mp.getCount();
                for (int i = 0; i < count; i += 1)
                {
                    BodyPart bp = mp.getBodyPart(i);
                    String ret = getText(bp, type);
                    // we found the part of the right type
                    if (ret != null)
                        return ret;
                }
            }

            // We don't want to look in a message/rfc822 part.
            // We don't want to scrounge around in anything else

            // Didn't find it
            return null;
        }
        catch (IOException e)
        {
            // probably an unsupported encoding exception, but in any case don't blow up
            // just skip this part
            return null;
        }
    }

    /**
     * Return the filename associated with this message part
     * from the MIME headers.  May return a list of names.
     *
     * @param p the message part to extract name(s) from
     * @return one or more names, or an empty string
     */
    protected String getAttachmentName(Part p)
    {
        try
        {
            if (p == null)
            {
                return "";
            }

            String filename = null;

            try
            {
                    filename = getFilename(p);
            }
            catch (Exception e)
            {
                //e.printStackTrace();
            }

            if (p.isMimeType("text/*") && filename != null)
            {
                return attachment_separator + filename;
            }
            else if (p.isMimeType("message/rfc822"))
            {
                // Bug 49769 fix: Doesn't make sense to look for attachments in the rfc822 message...
                //return getAttachmentName((Part) p.getContent());
                return "";
            }
            else if (p.isMimeType("multipart/*"))
            {
                StringBuffer buffer = new StringBuffer();
                try
                {
                    Multipart mp = (Multipart) p.getContent();
                    int count = mp.getCount();
                    for (int i = 0; i < count; i += 1)
                    {
                        BodyPart bodyPart;
                        try
                        {
                            bodyPart = mp.getBodyPart(i);
                        }
                        catch (MessagingException e)
                        {
                            // be robust in the face of bad messages, skip this part
                            continue;
                        }
                        buffer.append(getAttachmentName(bodyPart));
                    }
                }
                catch (MessagingException e)
                {
                    // be robust in the face of bad messages
                    // ignore this multipart
                }
                return buffer.toString();
            }
            else if (p.isMimeType("image/*")){
                /*
                     * If the mimeType is an image we need to add cid information to the attachment string
                     * that is generated so that information could be extracted from it to create a cid struct
                     * 
                     * At present only images are being considered for this cid table, however other media types 
                     * might be added in future.
                     */
                String cid = null;

                try{
                    String contentId = ((MimeBodyPart)p).getContentID();
                    if(contentId != null)
                        cid = " [cid:" + contentId + "]";
                }catch (MessagingException e){
                    cid = " [cid:error]";
                }

                if(filename == null){
                    filename = embedded_image_filename;
                }

                return attachment_separator + filename + (cid != null ? cid : "");
            }
            else
            {
                if (filename != null)
                {
                    return attachment_separator + filename;
                }
                else
                {
                    return "";
                }
            }
        }
        catch (MessagingException me)
        {
            // if we can't get check the mime type, forget the whole thing
            return "";
        }
        catch (IOException e)
        {
            // probably an unsupported encoding exception, but in any case don't blow up
            // just skip this part
            return "";
        }
    }

    /**
     * Return the list of pathnames for the saved attachements. In the process, write the attachments to disk.
     * 
     * @param p
     *            the part to examine for attachments
     * @param path
     *            the local path to store attachments files
     * @param unique
     *            true if we should make unique filenames to avoid overwriting existing files
     * @param isRoot
     * @param accept
     * @param accepted
     * @return a path or list of paths, or an emtpy string (if no attachments).
     */
    protected String getAttachmentPath(Part p, String path, boolean unique, boolean isRoot, String accept,
            String[] accepted)
    {
        if (p == null)
        {
            return "";
        }

        String filename = null;
        try
        {
            filename = getFilename(p);
        }
        catch (Exception e)
        {
        }

        try
        {
            if (p.isMimeType("text/*") && filename != null)
            {
                File f = Utils.getFullName(path, filename, unique);
                // check for MimeType or allowed File extensions
                if (isSafe(p, f.getCanonicalPath(), accept, accepted))
                {
                    IOUtils.saveFile(f, p.getInputStream());
                    return attachment_separator + f.getCanonicalPath();
                }
            }
            else if (p.isMimeType("message/rfc822"))
            {
                // Bug 49769 fix: Doesn't make sense to look for attachments in the rfc822 message...
                //return getAttachmentPath((Part) p.getContent(), path, unique);
                if(isRoot)
                    return "";
                else
                {
                    // Inside writeMessageToFile we check for allowed extension.
                    return writeMessageToFile(p, filename, path, unique, accept, accepted);
                }
            }
            else if (p.isMimeType("multipart/*"))
            {
                StringBuffer buffer = new StringBuffer();
                try
                {
                    Multipart mp = (Multipart) p.getContent();
                    int count = mp.getCount();
                    for (int i = 0; i < count; i += 1)
                    {
                        BodyPart bodyPart;
                        try
                        {
                            bodyPart = mp.getBodyPart(i);
                        }
                        catch (MessagingException e)
                        {
                            // be robust in the face of bad messages, skip this part
                            continue;
                        }
                        buffer.append(getAttachmentPath(bodyPart, path, unique, false, accept, accepted));
                    }
                }
                catch (MessagingException e)
                {
                    // be robust in the face of bad messages
                    // ignore this multipart
                }
                return buffer.toString();
            }
            else
            {
                try
                {
                    File f ;
                    if(filename == null )
                    {
                        if(isRoot || !(p.getContent() instanceof InputStream))
                            return "";
                        filename = "ATT" + System.currentTimeMillis() + ".att";
                        f = Utils.getFullName(path, filename, true);
                    }
                    else
                    {
                        f = Utils.getFullName(path, filename, unique);
                    }
                    
                    // check for MimeType or allowed File extensions
                    if (isSafe(p, f.getCanonicalPath(), accept, accepted))
                    {
                        InputStream in = p.getInputStream();
                        IOUtils.saveFile(f, in);
                        return attachment_separator + f.getCanonicalPath();
                    }
                }
                catch (IOException e)
                {
                    // be robust in the face of bad messages
                    // ignore this part
                    return "";
                }
                catch (MessagingException e)
                {
                    // be robust in the face of bad messages
                    // ignore this part
                    return "";
                }
            }
        }
        catch (MessagingException me)
        {
            // if we can't process the mime type, just skip it
            return "";
        }
        catch (IOException e)
        {
            // probably an unsupported encoding exception, but in any case don't blow up
            // just skip this part
            return "";
        }
        return "";
    }

    /**
     * parse accept attribute value and return the array
     * 
     * @param accept
     * @return
     */
    private String[] parseAccept(String accept)
    {
        if (accept != null && accept.length() < 2 && !accept.equals("*") && !accept.trim().equals(""))
        {
            throw new InvalidValException();
        }

        if (accept == null || accept.equals("*") || accept.trim().equals(""))
        {
            return new String[] { "*" };
        }
        
        return accept.split(",");
    }

    /**
     * Check if the file name extension or mime type matches with the values given in accept attribute.
     * 
     * @param p
     * @param filename
     * @param accepted
     */
    private boolean isSafe(Part p, String filename, String accept, String[] accepted)
    {
        if (accepted == null || (accepted.length == 1 && accepted[0].equals("*")))
        {
            return true;
        }

        boolean isSafe = false;

        // allow Mime type filtering with "accept" attribute
        for (String thisTok : accepted)
        {
            // everything matches */*
            if (thisTok.startsWith("."))
            {
                String ext = thisTok.substring(1, thisTok.length()).toLowerCase();
                String lFilename = filename.toLowerCase();
                isSafe = lFilename.endsWith(ext);
            }
            else
            {
                try
                {
                    isSafe = p.isMimeType(thisTok);
                }
                catch (MessagingException e)
                {
                    // ignore
                }
            }
            if (isSafe)
                break;
        }

        // "accept" attribute filter used
        if (!isSafe)
        {
            try
            {
                // TODO need to get it reviewed that should we throw error or just log it. Throwing the
                // error is resulting in not retrieving any emails which may not be a good idea.
                String contentType = p.getContentType();
                if(p instanceof IMAPMessage || p instanceof IMAPBodyPart){
                    ContentType cType = new ContentType(contentType);
                    cType.setParameterList(null);
                    contentType = cType.toString();
                }
                InvalidAttachmentTypeException e = new InvalidAttachmentTypeException(contentType, accept);
                // throw e;
                CFLogs.SERVER_LOG.error(e);
            }
            catch (MessagingException e)
            {
                // TODO need to get it reviewed that should we throw error or just log it. Throwing the
                // error is resulting in not retrieving any emails which may not be a good idea.
                InvalidAttachmentTypeException e1 = new InvalidAttachmentTypeException("", accept);
                // throw e;
                CFLogs.SERVER_LOG.error(e1);
            }
        }

        return isSafe;
    }

    private String writeMessageToFile(Part p, String filename, String path, boolean unique, String accept,
            String[] accepted)
            throws IOException, MessagingException
    {
        Object msg = p.getContent();
        if(msg != null && msg instanceof MimeMessage)
        {
            MimeMessage mimeMessage = (MimeMessage) msg;
            File f ;
            if(filename == null)
            {
                filename = "ATT" + System.currentTimeMillis() + ".eml";
                f = Utils.getFullName(path, filename, true);
            }
            else
            {
                f = Utils.getFullName(path, filename, unique);
            }
            
            OutputStream fos = null;
            try
            {
                // check for MimeType or allowed File extensions
                if (isSafe(p, f.getCanonicalPath(), accept, accepted))
                {
                    fos = VFSFileFactory.getOutputStream(f.getAbsolutePath());
                    mimeMessage.writeTo(fos);
                    return attachment_separator + f.getCanonicalPath();
                }
            } 
            catch (Exception e)
            {
                e.printStackTrace();
            }
            finally
            {
                try
                {
                    if(fos != null)
                        fos.close();
                } 
                catch(IOException ioe){}
            }
        }
        return "";
    }

    private String getFilename(Part p) throws Exception
    {
        String filename = _getFileName(p);

        /*
         * For images that are directly embedded into the mail message e.g. from programs written using JavaMail API
         * the filename information might not be present, such an embedded image is given the filename - 
         * "Embedded Image", suffixed with its image format. However, in case of more than one embedded images
         * the user would need to pass the attribute generateUniqueFilenames = "yes" to prevent overwriting 
         * the previous embedded image saved to the disk. 
         * 
         * TODO: In case of more than one embedded images in the mail, the cid table contains cid entry
         * 		of the last embedded image in the mail.
         */
        if (p.isMimeType("image/*") && filename == null){
            String imgFormat = p.getContentType().substring(6); //6 is length of "image/"
            filename = "Embedded Image." + imgFormat;
        }

        if (filename != null)
        {
            try
            {
                String conType = p.getContentType();
                ContentType cType = new ContentType(conType);
                String charset = cType.getParameter("charset");

                String save_filename = new String(filename);
                int len = save_filename.length();
                filename = MimeUtility.decodeText(filename);

                if (len > 0 && filename.equals(""))
                {
                    if (charset != null && charset.length() > 0)
                        filename = new String(save_filename.getBytes(charset));
                    else
                        filename = new String(save_filename.getBytes("UTF8"));
                }

            }
            catch (UnsupportedEncodingException e)
            {
               //assume if it is unknown it will be unicode, since we can't find out what it is.
                try
                {
                    filename = new String(filename.getBytes("UTF8"));
                }
                catch (Exception ex)
                {
                }

            }
            catch (ParseException parseex)
            {
                // Must be a strange content-type, but we did get a
                // filename, so ignore this and return the filename.
            }
            catch (Exception e)
            {
                // Since we have a filename at this point, we don't
                // want to throw an exception just because we are playing around
                // with character sets, just return it.
            }
        }

        return filename;
    }
    
    private static String _getFileName(Part part) throws MessagingException{

        // In case of IMAP after upgrading to mail library 1.5.x if the attachment file name
        // is Quotable encoding or Base64 encoded they are not getting decoded. This is because of 
        // the bug in ImapMessage & IMapbodypart getfilename issue. So instead of getting it from 
        // getfilename manually getting them from headers internally the same is used to get the part filename.
        if(part instanceof IMAPMessage || part instanceof IMAPBodyPart){
            String filename = null;
            String s = null;
            String[] dispos = part.getHeader("Content-Disposition");
            if(dispos != null && dispos.length > 0)
                s = dispos[0];
            
            if (s != null) {
                // Parse the header ..
                ContentDisposition cd = new ContentDisposition(s);
                filename = cd.getParameter("filename");
            }
            if (filename == null) {
                // Still no filename ? Try the "name" ContentType parameter
                String[] ctypes = part.getHeader("Content-Type");
                if(ctypes != null && ctypes.length > 0){
                    s = ctypes[0];
                    if(part instanceof MimePart)
                        s = MimeUtil.cleanContentType((MimePart) part, s);
                    if (s != null) {
                    try {
                        ContentType ct = new ContentType(s);
                        filename = ct.getParameter("name");
                    } catch (ParseException pex) { }    // ignore it
                    }
                }
            }
            return filename;
        }
        return part.getFileName();
                
    }

    /**
     * Value for attribute ACCEPT is invalid. It should be either a valid MIME type or file extension.
     */
    public static class InvalidValException extends ApplicationException
    {
        private static final long serialVersionUID = 1L;

        InvalidValException()
        {
            super();
        }
    }

    /**
     * The MIME type or the Extension for attachment {mimeType} was not accepted by the server. Only attachments of type
     * {accept} can be downloaded.
     */
    public static class InvalidAttachmentTypeException extends ApplicationException
    {
        private static final long serialVersionUID = 1L;

        private String accept;
        private String mimeType;

        public InvalidAttachmentTypeException(String mimeType)
        {
            this.mimeType = mimeType;
        }

        public InvalidAttachmentTypeException(String mimeType, String accept)
        {
            this.accept = accept;
            this.mimeType = mimeType;
        }

        public void setAccept(String accept)
        {
            this.accept = accept;
        }

        public String getAccept()
        {
            return accept;
        }

        public String getMimeType()
        {
            return mimeType;
        }
    }
}


class EmailTableMetaData extends QueryTableMetaData
{

    protected EmailTableMetaData()
    {
        super();
    }

    protected String[] fields;

    /**
     * Initialize the QueryTableMetaData with column types, names, case, count, etc.
     *
     * @param fs a list of columns
     */
    protected EmailTableMetaData(String[] fs)
    {

        fields = fs;

        Vector list = new Vector();
        for (int i = 0; fields != null && i < fields.length; i += 1)
        {
            list.addElement(fields[i]);
        }

        if (fs == null)
        {
            list.addElement("body");
        }

        column_count = list.size();
        column_label = new String[column_count];
        list.copyInto(column_label);
        column_case = new boolean[column_count];
        column_type = new int[column_count];
        column_type_names = new String[column_count];
        for (int j = 0; j < column_count; j++)
        {
            column_case[j] = false;
            if ("size".equalsIgnoreCase(column_label[j])
                    || "messagenumber".equalsIgnoreCase(column_label[j]))
            {
                column_type[j] = java.sql.Types.BIGINT;
                column_type_names[j] = "BIGINT";
            }
            else if ("sentdate".equalsIgnoreCase(column_label[j])
                    || "receiveddate".equalsIgnoreCase(column_label[j]))
            {
                column_type[j] = java.sql.Types.DATE;
                column_type_names[j] = "DATE";
            }
            else
            {
                column_type[j] = java.sql.Types.VARCHAR;
                column_type_names[j] = "VARCHAR";
            }
        }
    }
}
