/**
 * Copyright 2007 Adobe Systems Incorporated
 * All Rights Reserved.
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

package coldfusion.print;

import coldfusion.log.CFLogs;
import coldfusion.runtime.Array;
import coldfusion.runtime.ArrayUtil;
import coldfusion.runtime.Struct;
//import coldfusion.tagext.lang.PrintTag;

import javax.print.PrintService;
import javax.print.attribute.Attribute;
import javax.print.attribute.PrintRequestAttribute;
import javax.print.attribute.standard.Chromaticity;
import javax.print.attribute.standard.CopiesSupported;
import javax.print.attribute.standard.Destination;
import javax.print.attribute.standard.Fidelity;
import javax.print.attribute.standard.Finishings;
import javax.print.attribute.standard.JobHoldUntil;
import javax.print.attribute.standard.JobName;
import javax.print.attribute.standard.JobPriority;
import javax.print.attribute.standard.JobSheets;
import javax.print.attribute.standard.Media;
import javax.print.attribute.standard.MediaPrintableArea;
import javax.print.attribute.standard.MediaSize;
import javax.print.attribute.standard.MediaSizeName;
import javax.print.attribute.standard.NumberUp;
import javax.print.attribute.standard.OrientationRequested;
import javax.print.attribute.standard.PageRanges;
import javax.print.attribute.standard.PresentationDirection;
import javax.print.attribute.standard.PrintQuality;
import javax.print.attribute.standard.PrinterResolution;
import javax.print.attribute.standard.RequestingUserName;
import javax.print.attribute.standard.SheetCollate;
import javax.print.attribute.standard.Sides;
import javax.servlet.jsp.PageContext;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;

public class PrinterInfo
{
    String defaultPrinter = null;

    final private static String INCH_DISPLAY_STR = "in";
    final private static String MM_DISPLAY_STR = "mm";

    PrinterInfo()
    {
    }

    /**
     * Is printer the default printer?
     *
     * @param printer - to test if it's the default
     * @return true if the printer is the current default printer for the system
     */
    public boolean isDefaultPrinter(String printer)
    {
        return getDefaultPrinter().equals(printer);
    }

    /**
     * For the ColdFusion Administrator System Information page.
     *
     * @return the default printer if there is one, or the empty string
     */
    public static String getDefaultPrinter()
    {
        PrintBase pr = new PrintBase("pdf");
        String defaultPrinter = pr.getDefaultPrinter();
        return defaultPrinter != null ? defaultPrinter : "";
    }

    /**
     * For the ColdFusion Administrator System Information page.
     * Return list of configured printers that can print pdf files.
     * If CF is running as a Windows System Service it must be running under an account which has
     * PRINTER_ACCESS_USE permission for each printer it wants to use.  Even if the printer is configured
     * locally on the system it won't show up here if the account doesn't have the permission to query the printer.
     *
     * @return array of printer names
     */
    public static Object[] getPrinters()
    {
        PrintBase pr = new PrintBase("pdf");
        ArrayList pList = pr.getAllPrinters();

        if (pList.size() == 0)
            pList.add("");

        return pList.toArray();
    }

    /**
     * For the ColdFusion function GetPrinterInfo()
     *
     * @param pageContext
     * @param printer
     * @return Struct containing supported and unsupported attributes for the given printer
     */

    public static Struct getInfo(PageContext pageContext, String printer)
    {
        Struct result = new Struct();

        PrintBase pr = new PrintBase(pageContext,"pdf");
        pr.setPrinter(printer);

        PrinterInfo info = new PrinterInfo();
        PrintService printService = pr.getPrintService();
        result.put("printer", printService.getName());

        // add supported attribute values
        Class[] classes = printService.getSupportedAttributeCategories();
        for (int i = 0; i < classes.length; i++)
        {
            Object obj = printService.getSupportedAttributeValues(classes[i], pr.getDocFlavor(), null);
            if (obj != null)
            {
                try
                {
                    Object defaultObj = printService.getDefaultAttributeValue(classes[i]);
                    Attribute defaultAttr = (defaultObj != null && defaultObj instanceof Attribute) ? (Attribute) defaultObj : null;
                    info.addSupportedAttributeValue(obj, defaultAttr, result);
                }
                catch (Exception e)
                {
                    // skip it
                    CFLogs.PRINT_LOG.debug(e);
                }
            }
        }

        // this doesn't seem to be needed anymore now that the defaults are reported
        // get the default media name and set the default entry in the media struct
        //        try
        //        {
        //            // Let the printer driver tell us the default paper size for the printer which is typically
        //            // A4 for non-US (and maybe Canada) locales.     
        //            Paper paper = pr.getPaper(printService);
        //            MediaSizeName paperName = MediaSize.findMedia((float) (paper.getWidth() / 72.0), (float) (paper.getHeight() / 72.0), MediaSize.INCH);
        //            if (paperName != null)
        //            {
        //                Struct names = (Struct) result.get(CFPrintAttribute.MEDIA);
        //                if (names != null)
        //                {
        //                    Struct details = (Struct) names.get(paperName.toString());
        //                    details.put("default", "yes");
        //                }
        //            }
        //        }
        //        catch (PrinterException e)
        //        {
        //            // skip it
        //            CFLogs.PRINT_LOG.debug(e);
        //        }

        return result;
    }

    /**
     * Only add standard print attributes that are supported in the attributeStruct attribute.  There are
     * some that are only accessible via the printRequestAttributeSet attribute which isn't being documented.
     *
     * @param obj    from javax.print.attribute.standard
     * @param result
     */
    private void addSupportedAttributeValue(Object obj, Attribute defaultAttr, Struct result)
    {
        if (obj instanceof CopiesSupported)
        {
            // CopiesSupported is not a PrintRequestAttribute.
            setDefaultValue(CFPrintAttribute.COPIES, null, defaultAttr, result);
            result.put(CFPrintAttribute.COPIES, obj.toString());
        }
        else if (obj instanceof Destination)
        {
            //            Destination value = (Destination) obj;
            //            result.put(value.getName(), value.getURI().toString());
        }
        else if (obj instanceof JobHoldUntil)
        {
            setDefaultValue(CFPrintAttribute.JOB_HOLD_UNITL, null, defaultAttr, result);
            result.put(CFPrintAttribute.JOB_HOLD_UNITL, obj.toString());
        }
        else if (obj instanceof JobName)
        {
            // return the CF default, not the system default
            setDefaultValue(CFPrintAttribute.JOBNAME, PrintBase.DEFAULT_PRINT_JOB_NAME, defaultAttr, result);
            result.put(CFPrintAttribute.JOBNAME, PrintBase.DEFAULT_PRINT_JOB_NAME);
        }
        else if (obj instanceof JobPriority)
        {
            setDefaultValue(CFPrintAttribute.JOB_PRIORITY, null, defaultAttr, result);
            result.put(CFPrintAttribute.JOB_PRIORITY, obj.toString());
        }
        else if (obj instanceof RequestingUserName)
        {
            setDefaultValue(CFPrintAttribute.REQUESTING_USERNAME, null, defaultAttr, result);
            result.put(CFPrintAttribute.REQUESTING_USERNAME, obj.toString());
        }
        else if (obj instanceof Chromaticity[])
        {
            Chromaticity[] values = (Chromaticity[]) obj;
            if (values.length > 0)
            {
                Array cfarray = new Array(1, values.length);
                for (int i = 0; i < values.length; i++)
                {

                    String value = getCFColorValue(values[i]);
                    if (value != null)
                        cfarray.add(value);
                }
                Collections.sort(cfarray);
                setDefaultValue(CFPrintAttribute.COLOR, getCFColorValue((Chromaticity) defaultAttr), defaultAttr, result);
                result.put(CFPrintAttribute.COLOR, ArrayUtil.ArrayToList(cfarray, ","));
            }
        }
        else if (obj instanceof Fidelity[])
        {
            Fidelity[] values = (Fidelity[]) obj;
            if (values.length > 0)
            {
                Array cfarray = new Array(1, values.length);
                for (int i = 0; i < values.length; i++)
                {
                    String value = getCFFidelityValue(values[i]);
                    if (value != null)
                        cfarray.add(value);
                }
                Collections.sort(cfarray);
                setDefaultValue(CFPrintAttribute.FIDELITY, getCFFidelityValue((Fidelity) defaultAttr), defaultAttr, result);
                result.put(CFPrintAttribute.FIDELITY, ArrayUtil.ArrayToList(cfarray, ","));
            }
        }
        else if (obj instanceof Finishings[])
        {
            setDefaultValue(CFPrintAttribute.FINISHINGS, null, defaultAttr, result);
            setAttribute(CFPrintAttribute.FINISHINGS, (Finishings[]) obj, result);
        }
        else if (obj instanceof JobSheets[])
        {
            JobSheets[] values = (JobSheets[]) obj;
            if (values.length > 0)
            {
                Array cfarray = new Array(1, values.length);
                for (int i = 0; i < values.length; i++)
                {
                    String value = getCFCoverpageValue(values[i]);
                    if (value != null)
                        cfarray.add(value);
                }
                Collections.sort(cfarray);
                setDefaultValue(CFPrintAttribute.COVERPAGE, getCFCoverpageValue((JobSheets) defaultAttr), defaultAttr, result);
                result.put(CFPrintAttribute.COVERPAGE, ArrayUtil.ArrayToList(cfarray, ","));
            }
        }
        else if (obj instanceof NumberUp[])
        {
            setDefaultValue(CFPrintAttribute.NUMBERUP, null, defaultAttr, result);
            setAttribute(CFPrintAttribute.NUMBERUP, (NumberUp[]) obj, result);
        }
        else if (obj instanceof OrientationRequested[])
        {
            setDefaultValue(CFPrintAttribute.ORIENTATION, null, defaultAttr, result);
            setAttribute(CFPrintAttribute.ORIENTATION, (OrientationRequested[]) obj, result);
        }
        else if (obj instanceof PageRanges[])
        {
            setDefaultValue(CFPrintAttribute.PAGES, null, defaultAttr, result);
            setAttribute(CFPrintAttribute.PAGES, (PageRanges[]) obj, result);
        }
        else if (obj instanceof PresentationDirection[])
        {
            setDefaultValue(CFPrintAttribute.PRESENTATION_DIRECTION, null, defaultAttr, result);
            setAttribute(CFPrintAttribute.PRESENTATION_DIRECTION, (PresentationDirection[]) obj, result);
        }
        else if (obj instanceof PrintQuality[])
        {
            setDefaultValue(CFPrintAttribute.QUALITY, null, defaultAttr, result);
            setAttribute(CFPrintAttribute.QUALITY, (PrintQuality[]) obj, result);
        }
        else if (obj instanceof Sides[])
        {
            setDefaultValue(CFPrintAttribute.SIDES, null, defaultAttr, result);
            setAttribute(CFPrintAttribute.SIDES, (Sides[]) obj, result);
        }
        else if (obj instanceof SheetCollate[])
        {
            SheetCollate[] values = (SheetCollate[]) obj;
            if (values.length > 0)
            {
                Array cfarray = new Array(1, values.length);
                for (int i = 0; i < values.length; i++)
                {
                    String value = getCFCollateValue(values[i]);
                    if (value != null)
                        cfarray.add(value);
                }
                Collections.sort(cfarray);
                setDefaultValue(CFPrintAttribute.COLLATE, getCFCollateValue((SheetCollate) defaultAttr), defaultAttr, result);
                result.put(CFPrintAttribute.COLLATE, ArrayUtil.ArrayToList(cfarray, ","));
            }
        }
        else if (obj instanceof Media[])
        {
            Media[] values = (Media[]) obj;
            if (values.length > 0)
            {
                Struct allMediaStruct = new Struct();
                for (int i = 0; i < values.length; i++)
                {
                    // the size for the name if there is one
                    Struct mediaDetailsStruct = new Struct();
                    if (values[i] instanceof MediaSizeName)
                    {
                        MediaSize mediaSize = MediaSize.getMediaSizeForName((MediaSizeName) values[i]);
                        if (mediaSize != null)
                        {
                            mediaDetailsStruct.put(mediaSize.getName() + " (NA)",
                                    mediaSize.toString(MediaSize.INCH, INCH_DISPLAY_STR));
                            mediaDetailsStruct.put(mediaSize.getName() + " (ISO)",
                                    mediaSize.toString(MediaSize.MM, MM_DISPLAY_STR));
                        }
                    }

                    mediaDetailsStruct.put("index", new Integer(i + 1));

                    // key is the name
                    allMediaStruct.put(values[i].toString(), mediaDetailsStruct);
                }
                setDefaultValue(CFPrintAttribute.PAPER, null, defaultAttr, result);
                result.put(CFPrintAttribute.PAPER, allMediaStruct);
            }
        }
        else if (obj instanceof MediaPrintableArea[])
        {
            // this will only work if Media has already been seen but that appears to be the case
            // and this is only for display purposes so if it isn't show it's not horrible
            MediaPrintableArea[] values = (MediaPrintableArea[]) obj;
            Struct allMediaStruct = (Struct) result.get("media");
            if (allMediaStruct != null)
            {
                for (int i = 0; i < values.length; i++)
                {
                    if (values[i] != null)
                    {
                        // find the struct with id = i + 1
                        Struct detailsStruct;
                        Iterator iter = allMediaStruct.valuesIterator();
                        while (iter.hasNext())
                        {
                            detailsStruct = (Struct) iter.next();
                            if (detailsStruct != null)
                            {
                                Integer id = (Integer) detailsStruct.get("index");
                                if (id != null && id.intValue() == (i + 1))
                                {
                                    detailsStruct.put(values[i].getName() + " (NA)",
                                            values[i].toString(MediaPrintableArea.INCH, INCH_DISPLAY_STR));
                                    detailsStruct.put(values[i].getName() + " (ISO)",
                                            values[i].toString(MediaPrintableArea.MM, MM_DISPLAY_STR));
                                    break;
                                }
                            }
                        }
                    }
                }
            }
        }
        else if (obj instanceof PrinterResolution[])
        {
            //            PrinterResolution[] values = (PrinterResolution[]) obj;
            //            ArrayList list = new ArrayList(values.length);
            //            for (int i = 0; i < values.length; i++)
            //            {
            //                Struct resStruct = new Struct();
            //                resStruct.put("DPI", values[i].toString(PrinterResolution.DPI, "dpi"));
            //                resStruct.put("DPCM", values[i].toString(PrinterResolution.DPCM, "dpcm"));
            //                list.add(resStruct);
            //            }
            //            if (values.length > 0)
            //                result.put(values[0].getName(), list.toString());
        }
        else
        {
            // not one of the standard print attributes that CF supports so don't display
            if (Boolean.getBoolean("cfprint.debug"))
            {
                setDefaultValue(null, null, defaultAttr, result);
                result.put(obj.toString() + " (debug)", "class name: " + obj.getClass().getName());
            }
        }
    }

    /**
     * Convert Chromaticity Attribute value to CF values
     *
     * @param c Chromaticity
     * @return value of c translated to CF values
     */
    String getCFColorValue(Chromaticity c)
    {
        String value = null;

        if (c == Chromaticity.COLOR)
        {
            value = "yes";
        }
        else if (c == Chromaticity.MONOCHROME)
        {
            value = "no";
        }

        return value;
    }

    /**
     * Convert Fidelity Attribute value to CF values
     *
     * @param f Fidelity
     * @return value of f translated to CF values
     */
    String getCFFidelityValue(Fidelity f)
    {
        String value = null;

        if (f == Fidelity.FIDELITY_TRUE)
        {
            value = "yes";
        }
        else if (f == Fidelity.FIDELITY_FALSE)
        {
            value = "no";
        }

        return value;
    }

    /**
     * Convert SheetCollate Attribute value to CF values
     *
     * @param s SheetCollate
     * @return value of s translated to CF values
     */
    String getCFCollateValue(SheetCollate s)
    {
        String value = null;

        if (s == SheetCollate.COLLATED)
        {
            value = "yes";
        }
        else if (s == SheetCollate.UNCOLLATED)
        {
            value = "no";
        }

        return value;
    }

    /**
     * Convert JobSheets Attribute value to CF values
     *
     * @param s JobSheets
     * @return value of s translated to CF values
     */
    String getCFCoverpageValue(JobSheets s)
    {
        String value = null;

        if (s == JobSheets.NONE)
        {
            value = "no";
        }
        else if (s == JobSheets.STANDARD)
        {
            value = "yes";
        }

        return value;
    }

    /**
     * Add entry in result for the attribute array.
     *
     * @param name   String
     * @param obj    Object[] array of PrintRequestAttribute[]
     * @param result Struct
     */
    void setAttribute(String name, Object[] obj, Struct result)
    {
        PrintRequestAttribute[] values = (PrintRequestAttribute[]) obj;
        Array cfarray = new Array(1, values.length);
        for (int i = 0; i < values.length; i++)
        {
            cfarray.add(values[i].toString());
        }
        if (values.length > 0)
        {
            result.put(name, ArrayUtil.ArrayToList(cfarray, ","));
        }
    }

    /**
     * Put the default attribute into the defaults object of the result.
     *
     * @param cfName  the ColdFusion name for the attribute which may or may not be different from a.getName()
     * @param cfValue the ColdFusion value for the attribute which may or may not be different from a.toString()
     * @param a       the default attribute value
     * @param result  Struct to return GetPrinterInfo results
     */
    void setDefaultValue(String cfName, String cfValue, Attribute a, Struct result)
    {
        if (a != null)
        {
            Struct defaultStruct = (Struct) result.get("defaults");
            if (defaultStruct == null)
            {
                defaultStruct = new Struct();
                result.put("defaults", defaultStruct);
            }
            if (cfName == null)
                cfName = a.getName();

            if (cfValue == null)
                cfValue = a.toString();

            defaultStruct.put(cfName, cfValue);
        }
    }
}
