/**
 * Copyright 2007 Adobe Systems Incorporated
 * All Rights Reserved.
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

package coldfusion.print;

import coldfusion.util.FastHashtable;

import javax.print.PrintServiceLookup;
import javax.print.attribute.PrintRequestAttribute;
import javax.print.attribute.standard.Chromaticity;
import javax.print.attribute.standard.Finishings;
import javax.print.attribute.standard.JobSheets;
import javax.print.attribute.standard.MediaSizeName;
import javax.print.attribute.standard.OrientationRequested;
import javax.print.attribute.standard.PresentationDirection;
import javax.print.attribute.standard.PrintQuality;
import javax.print.attribute.standard.SheetCollate;
import javax.print.attribute.standard.Sides;


import sun.awt.AppContext;

import java.util.ArrayList;
import java.util.Enumeration;

/**
 * PrintUtil
 * <p/>
 * Created on Jan 18, 2007
 *
 * @author cframpto
 */
public class PrintUtil
{
    /**
     * *********************************************************
     * Finishings Map
     */
    private static FastHashtable finishingsMap = null;

    private static FastHashtable getFinishingsMap()
    {
        if (finishingsMap == null)
        {
            finishingsMap = new FastHashtable(18);
            finishingsMap.put(Finishings.NONE.toString(), Finishings.NONE);
            finishingsMap.put(Finishings.STAPLE.toString(), Finishings.STAPLE);
            finishingsMap.put(Finishings.EDGE_STITCH.toString(), Finishings.EDGE_STITCH);
            finishingsMap.put(Finishings.BIND.toString(), Finishings.BIND);
            finishingsMap.put(Finishings.SADDLE_STITCH.toString(), Finishings.SADDLE_STITCH);
            finishingsMap.put(Finishings.COVER.toString(), Finishings.COVER);

            finishingsMap.put(Finishings.STAPLE_TOP_LEFT.toString(), Finishings.STAPLE_TOP_LEFT);
            finishingsMap.put(Finishings.STAPLE_BOTTOM_LEFT.toString(), Finishings.STAPLE_BOTTOM_LEFT);
            finishingsMap.put(Finishings.STAPLE_TOP_RIGHT.toString(), Finishings.STAPLE_TOP_RIGHT);
            finishingsMap.put(Finishings.STAPLE_BOTTOM_RIGHT.toString(), Finishings.STAPLE_BOTTOM_RIGHT);

            finishingsMap.put(Finishings.EDGE_STITCH_LEFT.toString(), Finishings.EDGE_STITCH_LEFT);
            finishingsMap.put(Finishings.EDGE_STITCH_TOP.toString(), Finishings.EDGE_STITCH_TOP);
            finishingsMap.put(Finishings.EDGE_STITCH_RIGHT.toString(), Finishings.EDGE_STITCH_RIGHT);
            finishingsMap.put(Finishings.EDGE_STITCH_BOTTOM.toString(), Finishings.EDGE_STITCH_BOTTOM);

            finishingsMap.put(Finishings.STAPLE_DUAL_LEFT.toString(), Finishings.STAPLE_DUAL_LEFT);
            finishingsMap.put(Finishings.STAPLE_DUAL_TOP.toString(), Finishings.STAPLE_DUAL_TOP);
            finishingsMap.put(Finishings.STAPLE_DUAL_RIGHT.toString(), Finishings.STAPLE_DUAL_RIGHT);
            finishingsMap.put(Finishings.STAPLE_DUAL_BOTTOM.toString(), Finishings.STAPLE_DUAL_BOTTOM);
        }

        return finishingsMap;
    }


    public static PrintRequestAttribute getFinishings(String finish)
    {
        return (PrintRequestAttribute) getFinishingsMap().get(finish);
    }

    public static String getFinishingsValues()
    {
        return getValuesFromMap(getFinishingsMap());
    }

    /**
     * *********************************************************
     * Orientation Map
     */

    private static FastHashtable orientationMap = null;

    private static FastHashtable getOrientationMap()
    {
        if (orientationMap == null)
        {
            orientationMap = new FastHashtable(4);
            orientationMap.put(OrientationRequested.PORTRAIT.toString(), OrientationRequested.PORTRAIT);
            orientationMap.put(OrientationRequested.LANDSCAPE.toString(), OrientationRequested.LANDSCAPE);
            orientationMap.put(OrientationRequested.REVERSE_LANDSCAPE.toString(), OrientationRequested.REVERSE_LANDSCAPE);
            orientationMap.put(OrientationRequested.REVERSE_PORTRAIT.toString(), OrientationRequested.REVERSE_PORTRAIT);
        }

        return orientationMap;
    }


    public static PrintRequestAttribute getOrientation(String orientation)
    {
        return (PrintRequestAttribute) getOrientationMap().get(orientation);
    }

    public static String getOrientationValues()
    {
        return getValuesFromMap(getOrientationMap());
    }

    /**
     * *********************************************************
     * PresentationDirection Map
     */
    private static FastHashtable presentationDirectionMap = null;

    private static FastHashtable getPresentationDirectionMap()
    {
        if (presentationDirectionMap == null)
        {
            presentationDirectionMap = new FastHashtable(7);
            presentationDirectionMap.put(PresentationDirection.TOBOTTOM_TOLEFT.toString(), PresentationDirection.TOBOTTOM_TOLEFT);
            presentationDirectionMap.put(PresentationDirection.TOBOTTOM_TORIGHT.toString(), PresentationDirection.TOBOTTOM_TORIGHT);
            presentationDirectionMap.put(PresentationDirection.TOLEFT_TOBOTTOM.toString(), PresentationDirection.TOLEFT_TOBOTTOM);
            presentationDirectionMap.put(PresentationDirection.TOLEFT_TOTOP.toString(), PresentationDirection.TOLEFT_TOTOP);
            presentationDirectionMap.put(PresentationDirection.TORIGHT_TOBOTTOM.toString(), PresentationDirection.TORIGHT_TOBOTTOM);
            presentationDirectionMap.put(PresentationDirection.TORIGHT_TOTOP.toString(), PresentationDirection.TORIGHT_TOTOP);
            presentationDirectionMap.put(PresentationDirection.TOTOP_TOLEFT.toString(), PresentationDirection.TOTOP_TOLEFT);
            presentationDirectionMap.put(PresentationDirection.TOTOP_TORIGHT.toString(), PresentationDirection.TOTOP_TORIGHT);
        }

        return presentationDirectionMap;
    }

    public static PrintRequestAttribute getPresentationDirection(String dir)
    {
        return (PrintRequestAttribute) getPresentationDirectionMap().get(dir);
    }

    public static String getPresentationDirectionValues()
    {
        return getValuesFromMap(getPresentationDirectionMap());
    }

    /**
     * *********************************************************
     * MediaName Map
     */

    //    private static FastHashtable mediaNameMap = null;
    //
    //    private static FastHashtable getMediaNameMap()
    //    {
    //        if (mediaNameMap == null)
    //        {
    //            mediaNameMap = new FastHashtable(4);
    //            mediaNameMap.put(MediaName.ISO_A4_TRANSPARENT.toString(), MediaName.ISO_A4_TRANSPARENT);
    //            mediaNameMap.put(MediaName.ISO_A4_WHITE.toString(), MediaName.ISO_A4_WHITE);
    //            mediaNameMap.put(MediaName.NA_LETTER_TRANSPARENT.toString(), MediaName.NA_LETTER_TRANSPARENT);
    //            mediaNameMap.put(MediaName.NA_LETTER_WHITE.toString(), MediaName.NA_LETTER_WHITE);
    //        }
    //
    //        return mediaNameMap;
    //    }
    //
    //    public static PrintRequestAttribute getMediaName(String name)
    //    {
    //        return (PrintRequestAttribute) getMediaNameMap().get(name);
    //    }
    //
    //    public static String getMediaNameValues()
    //    {
    //        return getValuesFromMap(getMediaNameMap());
    //    }

    /**
     * *********************************************************
     * MediaTray Map
     */

    //    private static FastHashtable mediaTrayMap = null;
    //
    //    private static FastHashtable getMediaTrayMap()
    //    {
    //        if (mediaTrayMap == null)
    //        {
    //            mediaTrayMap = new FastHashtable(8);
    //            mediaTrayMap.put(MediaTray.BOTTOM.toString(), MediaTray.BOTTOM);
    //            mediaTrayMap.put(MediaTray.ENVELOPE.toString(), MediaTray.ENVELOPE);
    //            mediaTrayMap.put(MediaTray.LARGE_CAPACITY.toString(), MediaTray.LARGE_CAPACITY);
    //            mediaTrayMap.put(MediaTray.MAIN.toString(), MediaTray.MAIN);
    //            mediaTrayMap.put(MediaTray.MANUAL.toString(), MediaTray.MANUAL);
    //            mediaTrayMap.put(MediaTray.MIDDLE.toString(), MediaTray.MIDDLE);
    //            mediaTrayMap.put(MediaTray.SIDE.toString(), MediaTray.SIDE);
    //            mediaTrayMap.put(MediaTray.TOP.toString(), MediaTray.TOP);
    //        }
    //
    //        return mediaTrayMap;
    //    }
    //
    //    public static PrintRequestAttribute getMediaTray(String tray)
    //    {
    //        return (PrintRequestAttribute) getMediaTrayMap().get(tray);
    //    }
    //
    //    public static String getMediaTrayValues()
    //    {
    //        return getValuesFromMap(getMediaTrayMap());
    //    }

    /**
     * *********************************************************
     * PrintQuality Map
     */

    private static FastHashtable printQualityMap = null;

    private static FastHashtable getPrintQualityMap()
    {
        if (printQualityMap == null)
        {
            printQualityMap = new FastHashtable(3);
            printQualityMap.put(PrintQuality.DRAFT.toString(), PrintQuality.DRAFT);
            printQualityMap.put(PrintQuality.NORMAL.toString(), PrintQuality.NORMAL);
            printQualityMap.put(PrintQuality.HIGH.toString(), PrintQuality.HIGH);
        }

        return printQualityMap;
    }

    public static PrintRequestAttribute getPrintQuality(String quality)
    {
        return (PrintRequestAttribute) getPrintQualityMap().get(quality);
    }

    public static String getPrintQualityValues()
    {
        return getValuesFromMap(getPrintQualityMap());
    }

    /**
     * *********************************************************
     * Sides Map
     */

    private static final String SIDES_DUPLEX = "duplex";
    private static final String SIDES_TUMBLE = "tumble";

    private static FastHashtable sidesMap = null;

    private static FastHashtable getSidesMap()
    {
        if (sidesMap == null)
        {
            // Duplex is synonym for TWO_SIDED_LONG_EDGE
            // Tumble is synonym for TWO_SIDED_SHORT_EDGE
            sidesMap = new FastHashtable(5);
            sidesMap.put(Sides.ONE_SIDED.toString(), Sides.ONE_SIDED);
            sidesMap.put(Sides.TWO_SIDED_LONG_EDGE.toString(), Sides.TWO_SIDED_LONG_EDGE);
            sidesMap.put(Sides.TWO_SIDED_SHORT_EDGE.toString(), Sides.TWO_SIDED_SHORT_EDGE);
            sidesMap.put(SIDES_DUPLEX, Sides.DUPLEX);
            sidesMap.put(SIDES_TUMBLE, Sides.TUMBLE);
        }

        return sidesMap;
    }

    public static PrintRequestAttribute getSides(String sides)
    {
        return (PrintRequestAttribute) getSidesMap().get(sides);
    }

    public static String getSidesValues()
    {
        return getValuesFromMap(getSidesMap());
    }

    /**
     * *********************************************************
     * PaperSize Map
     */

    private static final String PAPER_SIZE_LETTER = "letter";
    private static final String PAPER_SIZE_LEGAL = "legal";
    private static final String PAPER_SIZE_A4 = "a4";
    private static final String PAPER_SIZE_A5 = "a5";
    private static final String PAPER_SIZE_B4 = "b4";
    private static final String PAPER_SIZE_B5 = "b5";
    private static final String PAPER_SIZE_B4_JIS = "b4-jis";
    private static final String PAPER_SIZE_B5_JIS = "b5-jis";

    private static FastHashtable cfDocumentMap = null;

    private static FastHashtable getCFDocumentMap()
    {
        if (cfDocumentMap == null)
        {
            cfDocumentMap = new FastHashtable();

            // these are synonyms for cfdocument pageSizes
            cfDocumentMap.put(PAPER_SIZE_LETTER, MediaSizeName.NA_LETTER);
            cfDocumentMap.put(PAPER_SIZE_LEGAL, MediaSizeName.NA_LEGAL);
            cfDocumentMap.put(PAPER_SIZE_A4, MediaSizeName.ISO_A4);
            cfDocumentMap.put(PAPER_SIZE_A5, MediaSizeName.ISO_A5);
            cfDocumentMap.put(PAPER_SIZE_B4, MediaSizeName.ISO_B4);
            cfDocumentMap.put(PAPER_SIZE_B5, MediaSizeName.ISO_B5);
            cfDocumentMap.put(PAPER_SIZE_B4_JIS, MediaSizeName.JIS_B4);
            cfDocumentMap.put(PAPER_SIZE_B5_JIS, MediaSizeName.JIS_B5);
        }

        return cfDocumentMap;
    }

    public static PrintRequestAttribute getMediaFromCFDocumentPapersize(String name)
    {
        return (PrintRequestAttribute) getCFDocumentMap().get(name);
    }

    //    private static FastHashtable mediaSizeMap = null;
    //    
    //    private static FastHashtable getMediaSizeMap()
    //    {
    //        if (mediaSizeMap == null)
    //        {
    //            mediaSizeMap = new FastHashtable();
    //
    //            // these are synonyms for cfdocument pageSizes
    //            mediaSizeMap.put(PAPER_SIZE_LETTER, MediaSizeName.NA_LETTER);
    //            mediaSizeMap.put(PAPER_SIZE_LEGAL, MediaSizeName.NA_LEGAL);
    //            mediaSizeMap.put(PAPER_SIZE_A4, MediaSizeName.ISO_A4);
    //            mediaSizeMap.put(PAPER_SIZE_A5, MediaSizeName.ISO_A5);
    //            mediaSizeMap.put(PAPER_SIZE_B4, MediaSizeName.ISO_B4);
    //            mediaSizeMap.put(PAPER_SIZE_B5, MediaSizeName.ISO_B5);
    //            mediaSizeMap.put(PAPER_SIZE_B4_JIS, MediaSizeName.JIS_B4);
    //            mediaSizeMap.put(PAPER_SIZE_B5_JIS, MediaSizeName.JIS_B5);
    //
    //            mediaSizeMap.put(MediaSizeName.A.toString(), MediaSizeName.A);
    //            mediaSizeMap.put(MediaSizeName.B.toString(), MediaSizeName.B);
    //            mediaSizeMap.put(MediaSizeName.C.toString(), MediaSizeName.C);
    //            mediaSizeMap.put(MediaSizeName.D.toString(), MediaSizeName.D);
    //            mediaSizeMap.put(MediaSizeName.E.toString(), MediaSizeName.E);
    //            mediaSizeMap.put(MediaSizeName.EXECUTIVE.toString(), MediaSizeName.EXECUTIVE);
    //            mediaSizeMap.put(MediaSizeName.FOLIO.toString(), MediaSizeName.FOLIO);
    //            mediaSizeMap.put(MediaSizeName.INVOICE.toString(), MediaSizeName.INVOICE);
    //            mediaSizeMap.put(MediaSizeName.ISO_A0.toString(), MediaSizeName.ISO_A0);
    //            mediaSizeMap.put(MediaSizeName.ISO_A1.toString(), MediaSizeName.ISO_A1);
    //            mediaSizeMap.put(MediaSizeName.ISO_A10.toString(), MediaSizeName.ISO_A10);
    //            mediaSizeMap.put(MediaSizeName.ISO_A2.toString(), MediaSizeName.ISO_A2);
    //            mediaSizeMap.put(MediaSizeName.ISO_A3.toString(), MediaSizeName.ISO_A3);
    //            mediaSizeMap.put(MediaSizeName.ISO_A4.toString(), MediaSizeName.ISO_A4);
    //            mediaSizeMap.put(MediaSizeName.ISO_A5.toString(), MediaSizeName.ISO_A5);
    //            mediaSizeMap.put(MediaSizeName.ISO_A6.toString(), MediaSizeName.ISO_A6);
    //            mediaSizeMap.put(MediaSizeName.ISO_A7.toString(), MediaSizeName.ISO_A7);
    //            mediaSizeMap.put(MediaSizeName.ISO_A8.toString(), MediaSizeName.ISO_A8);
    //            mediaSizeMap.put(MediaSizeName.ISO_A9.toString(), MediaSizeName.ISO_A9);
    //            mediaSizeMap.put(MediaSizeName.ISO_B0.toString(), MediaSizeName.ISO_B0);
    //            mediaSizeMap.put(MediaSizeName.ISO_B1.toString(), MediaSizeName.ISO_B1);
    //            mediaSizeMap.put(MediaSizeName.ISO_B10.toString(), MediaSizeName.ISO_B10);
    //            mediaSizeMap.put(MediaSizeName.ISO_B2.toString(), MediaSizeName.ISO_B2);
    //            mediaSizeMap.put(MediaSizeName.ISO_B3.toString(), MediaSizeName.ISO_B3);
    //            mediaSizeMap.put(MediaSizeName.ISO_B4.toString(), MediaSizeName.ISO_B4);
    //            mediaSizeMap.put(MediaSizeName.ISO_B5.toString(), MediaSizeName.ISO_B5);
    //            mediaSizeMap.put(MediaSizeName.ISO_B6.toString(), MediaSizeName.ISO_B6);
    //            mediaSizeMap.put(MediaSizeName.ISO_B7.toString(), MediaSizeName.ISO_B7);
    //            mediaSizeMap.put(MediaSizeName.ISO_B8.toString(), MediaSizeName.ISO_B8);
    //            mediaSizeMap.put(MediaSizeName.ISO_B9.toString(), MediaSizeName.ISO_B9);
    //            mediaSizeMap.put(MediaSizeName.ISO_C0.toString(), MediaSizeName.ISO_C0);
    //            mediaSizeMap.put(MediaSizeName.ISO_C1.toString(), MediaSizeName.ISO_C1);
    //            mediaSizeMap.put(MediaSizeName.ISO_C2.toString(), MediaSizeName.ISO_C2);
    //            mediaSizeMap.put(MediaSizeName.ISO_C3.toString(), MediaSizeName.ISO_C3);
    //            mediaSizeMap.put(MediaSizeName.ISO_C4.toString(), MediaSizeName.ISO_C4);
    //            mediaSizeMap.put(MediaSizeName.ISO_C5.toString(), MediaSizeName.ISO_C5);
    //            mediaSizeMap.put(MediaSizeName.ISO_C6.toString(), MediaSizeName.ISO_C6);
    //            mediaSizeMap.put(MediaSizeName.ISO_DESIGNATED_LONG.toString(), MediaSizeName.ISO_DESIGNATED_LONG);
    //            mediaSizeMap.put(MediaSizeName.ITALY_ENVELOPE.toString(), MediaSizeName.ITALY_ENVELOPE);
    //            mediaSizeMap.put(MediaSizeName.JAPANESE_DOUBLE_POSTCARD.toString(), MediaSizeName.JAPANESE_DOUBLE_POSTCARD);
    //            mediaSizeMap.put(MediaSizeName.JAPANESE_POSTCARD.toString(), MediaSizeName.JAPANESE_POSTCARD);
    //            mediaSizeMap.put(MediaSizeName.JIS_B0.toString(), MediaSizeName.JIS_B0);
    //            mediaSizeMap.put(MediaSizeName.JIS_B1.toString(), MediaSizeName.JIS_B1);
    //            mediaSizeMap.put(MediaSizeName.JIS_B10.toString(), MediaSizeName.JIS_B10);
    //            mediaSizeMap.put(MediaSizeName.JIS_B2.toString(), MediaSizeName.JIS_B2);
    //            mediaSizeMap.put(MediaSizeName.JIS_B3.toString(), MediaSizeName.JIS_B3);
    //            mediaSizeMap.put(MediaSizeName.JIS_B4.toString(), MediaSizeName.JIS_B4);
    //            mediaSizeMap.put(MediaSizeName.JIS_B5.toString(), MediaSizeName.JIS_B5);
    //            mediaSizeMap.put(MediaSizeName.JIS_B6.toString(), MediaSizeName.JIS_B6);
    //            mediaSizeMap.put(MediaSizeName.JIS_B7.toString(), MediaSizeName.JIS_B7);
    //            mediaSizeMap.put(MediaSizeName.JIS_B8.toString(), MediaSizeName.JIS_B8);
    //            mediaSizeMap.put(MediaSizeName.JIS_B9.toString(), MediaSizeName.JIS_B9);
    //            mediaSizeMap.put(MediaSizeName.LEDGER.toString(), MediaSizeName.LEDGER);
    //            mediaSizeMap.put(MediaSizeName.MONARCH_ENVELOPE.toString(), MediaSizeName.MONARCH_ENVELOPE);
    //            mediaSizeMap.put(MediaSizeName.NA_10X13_ENVELOPE.toString(), MediaSizeName.NA_10X13_ENVELOPE);
    //            mediaSizeMap.put(MediaSizeName.NA_10X14_ENVELOPE.toString(), MediaSizeName.NA_10X14_ENVELOPE);
    //            mediaSizeMap.put(MediaSizeName.NA_10X15_ENVELOPE.toString(), MediaSizeName.NA_10X15_ENVELOPE);
    //            mediaSizeMap.put(MediaSizeName.NA_5X7.toString(), MediaSizeName.NA_5X7);
    //            mediaSizeMap.put(MediaSizeName.NA_6X9_ENVELOPE.toString(), MediaSizeName.NA_6X9_ENVELOPE);
    //            mediaSizeMap.put(MediaSizeName.NA_7X9_ENVELOPE.toString(), MediaSizeName.NA_7X9_ENVELOPE);
    //            mediaSizeMap.put(MediaSizeName.NA_8X10.toString(), MediaSizeName.NA_8X10);
    //            mediaSizeMap.put(MediaSizeName.NA_9X11_ENVELOPE.toString(), MediaSizeName.NA_9X11_ENVELOPE);
    //            mediaSizeMap.put(MediaSizeName.NA_9X12_ENVELOPE.toString(), MediaSizeName.NA_9X12_ENVELOPE);
    //            mediaSizeMap.put(MediaSizeName.NA_LEGAL.toString(), MediaSizeName.NA_LEGAL);
    //            mediaSizeMap.put(MediaSizeName.NA_LETTER.toString(), MediaSizeName.NA_LETTER);
    //            mediaSizeMap.put(MediaSizeName.NA_NUMBER_10_ENVELOPE.toString(), MediaSizeName.NA_NUMBER_10_ENVELOPE);
    //            mediaSizeMap.put(MediaSizeName.NA_NUMBER_11_ENVELOPE.toString(), MediaSizeName.NA_NUMBER_11_ENVELOPE);
    //            mediaSizeMap.put(MediaSizeName.NA_NUMBER_12_ENVELOPE.toString(), MediaSizeName.NA_NUMBER_12_ENVELOPE);
    //            mediaSizeMap.put(MediaSizeName.NA_NUMBER_14_ENVELOPE.toString(), MediaSizeName.NA_NUMBER_14_ENVELOPE);
    //            mediaSizeMap.put(MediaSizeName.NA_NUMBER_9_ENVELOPE.toString(), MediaSizeName.NA_NUMBER_9_ENVELOPE);
    //            mediaSizeMap.put(MediaSizeName.PERSONAL_ENVELOPE.toString(), MediaSizeName.PERSONAL_ENVELOPE);
    //            mediaSizeMap.put(MediaSizeName.QUARTO.toString(), MediaSizeName.QUARTO);
    //            mediaSizeMap.put(MediaSizeName.TABLOID.toString(), MediaSizeName.TABLOID);
    //
    //        }
    //
    //        return mediaSizeMap;
    //    }
    //
    //    public static PrintRequestAttribute getMediaSize(String size)
    //    {
    //        return (PrintRequestAttribute) getMediaSizeMap().get(size);
    //    }
    //
    //    public static String getMediaSizeValues()
    //    {
    //        return getValuesFromMap(getMediaSizeMap());
    //    }

    /**
     * To convert all the values in a map of Strings to a single string for display.
     *
     * @param map
     * @return single string of map values for display
     */
    private static String getValuesFromMap(FastHashtable map)
    {
        String values = "";
        Enumeration e = map.keys();
        while (e.hasMoreElements())
            values += e.nextElement() + " ";
        return values.trim();
    }

    /**
     * Convert ArrayList to string for display, removing the enclosing brackets.
     *
     * @param list
     * @return string representation of list with enclosing brackets removed
     */
    public static String toDisplayString(ArrayList list)
    {
        String str = "";
        if (list != null)
        {
            str = list.toString();
            int len = str.length();
            // Remove '[' and ']' from toString.
            if (len >= 2)
                str = str.substring(1, len - 1);
        }
        return str;
    }

    /**
     * @param attribute Java attribute name
     * @return CF attribute name
     */
    public static String toCFAttributeName(String attribute)
    {
        if (attribute.equals(Chromaticity.COLOR.getName()))
            return CFPrintAttribute.COLOR;

        if (attribute.equals("media"))
            return CFPrintAttribute.PAPER;

        if (attribute.equals(JobSheets.STANDARD.getName()))
            return CFPrintAttribute.COVERPAGE;

        if (attribute.equals(PrintQuality.NORMAL.getName()))
            return CFPrintAttribute.QUALITY;

        if (attribute.equals(SheetCollate.COLLATED.getName()))
            return CFPrintAttribute.COLLATE;

        // remove all '-' characters in attribute name

        int i = 0;
        while (true)
        {
            i = attribute.indexOf('-', i);
            if (i >= 0)
                attribute = attribute.substring(0, i) + attribute.substring(i + 1);
            else
                break;
        }

        return attribute;
    }

    /***
     * javax.print.PrintServiceLookup is held up once print service is called. 
     * And to refresh the printer list we have to unload this class and reload 
     * it back.
     */
    public static void refreshPrinterList()
    {
      Class[] arrayOfClass1 = PrintServiceLookup.class.getDeclaredClasses();
      for (Class localClass : arrayOfClass1) {
        if (localClass.getName().equals("javax.print.PrintServiceLookup$Services"))
        {
          AppContext.getAppContext().remove(localClass);
          break;
        }
      }
    }
    
    /**
     * @param alist Java attribute names
     * @return CF attribute names
     */
    public static String toCFAttributeNames(ArrayList alist)
    {
        for (int i = 0; i < alist.size(); i++)
        {
            alist.set(i, toCFAttributeName((String) alist.get(i)));
        }

        return toDisplayString(alist);
    }
}
