/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 2008 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and may be covered by U.S. and Foreign Patents,
* patents in process, and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/

package coldfusion.tagext.print;

import static coldfusion.print.core.PrintExceptions.*;
import coldfusion.print.core.*;
import coldfusion.featurerouter.EFRConstants;
import coldfusion.featurerouter.FeatureRouter;
import coldfusion.print.CFPrintAttribute;
import coldfusion.print.PrintPDF;
import coldfusion.print.PrintPDFAttributes;
import coldfusion.print.PrintUtil;
import coldfusion.runtime.Cast;
import coldfusion.runtime.Struct;
import coldfusion.runtime.StructUtils;
import coldfusion.runtime.java.JavaProxy;
import coldfusion.server.ServiceFactory;
import coldfusion.tagext.GenericTag;
import coldfusion.tagext.GenericTagPermission;
import coldfusion.tagext.io.FileUtils;
import coldfusion.util.Utils;
import coldfusion.vfs.VFSFileFactory;

import javax.print.attribute.Attribute;
import javax.print.attribute.HashPrintRequestAttributeSet;
import javax.print.attribute.PrintRequestAttribute;
import javax.print.attribute.standard.Chromaticity;
import javax.print.attribute.standard.Copies;
import javax.print.attribute.standard.Fidelity;
import javax.print.attribute.standard.JobHoldUntil;
import javax.print.attribute.standard.JobName;
import javax.print.attribute.standard.JobPriority;
import javax.print.attribute.standard.JobSheets;
import javax.print.attribute.standard.NumberUp;
import javax.print.attribute.standard.PageRanges;
import javax.print.attribute.standard.RequestingUserName;
import javax.print.attribute.standard.SheetCollate;
import javax.servlet.jsp.JspException;

import java.security.AccessController;
import java.security.Permission;
import java.security.PrivilegedAction;
import java.util.Date;
import java.util.Enumeration;

/**
 * PrintTag
 * <p/>
 * Created on January 16, 2007
 *
 * @author cframpton
 * @tag print
 * @body JSP
 */
public class PrintTag extends GenericTag
{
    public static final String CFPRINT_TAG = "cfprint";

    public static final String TYPE_PDF = "pdf";
    public static final String TYPE_DEFAULT = TYPE_PDF;
    public static final String ATTR_SOURCE = "source";
    public static final String ATTR_TYPE = "type";
    public static final String ATTR_DEBUG = "debug";
    public static final String MEDIA_DEFAULT = "iso-a4";

    private static final String ATTR_ATTRIBUTE_STRUCT = "attributeStruct";
    private static final String ATTR_PRINT_REQUEST_ATTRIBUTE_SET = "printRequestAttributeSet";

    private Struct attributeStruct;
    private String password;
    private String printer;
    private HashPrintRequestAttributeSet printRequestAttributeSet;
    private Object source;
    private String type;
    private PrintPDFAttributes pdfAttributes;
    private String media;

    private HashPrintRequestAttributeSet prSet;

    public PrintTag()
    {
        init();
    }

    /**
     * Called by constructor and release.  Initialize fields to defaults here.
     */
    private void init()
    {
        attributeStruct = null;
        password = null;
        printer = null;
        printRequestAttributeSet = null;
        source = null;
        type = TYPE_DEFAULT;
        pdfAttributes = new PrintPDFAttributes();
        media = null;

        prSet = new HashPrintRequestAttributeSet();
    }

    public void release()
    {
        init();
        super.release();
    }

    /**
     * ************** these setters are visible from the tag ********************
     * ************** keep in alphabetical order by attribute ********************
     */

    /**
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setAttributestruct(Object cfStruct)
    {
        if (StructUtils.IsStruct(cfStruct))
        {
            this.attributeStruct = (Struct) cfStruct;
        }
        else
        {
            throw new CFPrintEmptyAttributeException(ATTR_ATTRIBUTE_STRUCT);
        }
    }

    /**
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setColor(boolean isColor)
    {
        addAttr(isColor ? Chromaticity.COLOR : Chromaticity.MONOCHROME);
    }

    /**
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setCopies(int copies)
    {
        try
        {
            addAttr(new Copies(copies));
        }
        catch (IllegalArgumentException e)
        {
            throw new CFPrintInvalidValueException(CFPrintAttribute.COPIES, String.valueOf(copies), e.getLocalizedMessage());
        }
    }

    /**
     * fidelity means "the quality or state of being faithful or accurate"
     * the quality or state of being faithful
     * <p/>
     * If FIDELITY_TRUE is specified and a service cannot print the job exactly as specified it must
     * reject the job. If FIDELITY_FALSE is specified a reasonable attempt to print the job is acceptable.
     * If not supplied the default is FIDELITY_FALSE.
     *
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setFidelity(boolean isExact)
    {
        addAttr(isExact ? Fidelity.FIDELITY_TRUE : Fidelity.FIDELITY_FALSE);
    }

    /**
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setPages(String pages)
    {
        if (pages.trim().length() == 0)
            throw new CFPrintEmptyAttributeException(CFPrintAttribute.PAGES);

        try
        {
            addAttr(new PageRanges(pages));
        }
        catch (IllegalArgumentException e)
        {
            // todo: invalid 10-5 range message is "members is zero-length"
            // make more friendly
            throw new CFPrintInvalidValueException(CFPrintAttribute.PAGES, pages, e.getLocalizedMessage());
        }
    }

    /**
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setPaper(String paper)
    {
        setMedia(CFPrintAttribute.PAPER, paper);
    }

    /**
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setPassword(String password)
    {
        // don't touch it
        this.password = password;
    }

    /**
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setPrinter(String printer)
    {
        if (printer.trim().length() == 0)
            throw new CFPrintEmptyAttributeException(CFPrintAttribute.PRINTER);

        this.printer = printer.trim();
    }

    /**
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setPrintrequestattributeset(Object javaObject)
    {
        if (javaObject instanceof JavaProxy)
        {
            Object o = ((JavaProxy) javaObject).getObject();
            if (o instanceof HashPrintRequestAttributeSet)
            {
                this.printRequestAttributeSet = (HashPrintRequestAttributeSet) o;
                return;
            }
        }

        throw new CFPrintEmptyAttributeException(ATTR_PRINT_REQUEST_ATTRIBUTE_SET);
    }

    /**
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setSource(String sourceStr)
    {
        // See if this is a CF variable - a string, a PDF variable or a Document variable.
        Object value = null;
        try
        {
            value = pageContext.findAttribute(sourceStr);
        }
        catch (Exception e)
        {
           value = sourceStr;
        }
        if (value != null)
        {
            if (value instanceof String)
            {
                sourceStr = (String) value;
            }
            else if (ServiceFactory.getPDFService().isPDFObject(value) || value instanceof byte[])
            {
                this.source = value;
                return;
            }
            else
            {
                throw new CFPrintEmptyAttributeException(ATTR_SOURCE);
            }
        }

        // Assume it's a file.
        if ("".equals((sourceStr).trim()))
        {
            throw new CFPrintEmptyAttributeException(ATTR_SOURCE);
        }
        if ((sourceStr).indexOf(",") == -1)
        {
            sourceStr = Utils.getFileFullPath((sourceStr).trim(), pageContext, true);
        }

        // Make sure we can access the source file from the sandbox if that's being used.
        if (!fileExists(sourceStr) || VFSFileFactory.getFileObject(sourceStr).isDirectory())
        {
            throw new FileUtils.FileNotFoundException(sourceStr);
        }

        this.source = sourceStr;
    }
    
    static boolean fileExists(final String filename)
    {
        if (System.getSecurityManager() == null)
            return VFSFileFactory.getFileObject(filename).exists();
        else
        {
            Boolean bExists = (Boolean) AccessController.doPrivileged(new PrivilegedAction()
            {
                public Object run()
                {
                    return Boolean.valueOf(VFSFileFactory.getFileObject(filename).exists());
                }
            });
            return bExists.booleanValue();
        }
    }

    /**
     * @tagattribute
     * @required true
     * @rtexprvalue false
     */
    public void setType(String type)
    {
        if (type.trim().length() == 0)
            throw new CFPrintEmptyAttributeException(ATTR_TYPE);

        type = type.trim().toLowerCase();

        if (!type.equalsIgnoreCase(TYPE_PDF))
            throw new CFPrintInvalidValueException(ATTR_TYPE, type, TYPE_PDF);

        this.type = type;
    }

    /**
     * ************** these aren't visible from the tag ********************
     * ************** keep in alphabetical order by attribute ********************
     */

    public void setAutoRotateAndCenter(boolean isAutoRotateAndCenter)
    {
        this.pdfAttributes.setAutoRotateAndCenter(isAutoRotateAndCenter);
    }

    private void setCollate(boolean isCollate)
    {
        addAttr(isCollate ? SheetCollate.COLLATED : SheetCollate.UNCOLLATED);
    }

    private void setCoverpage(boolean isCoverPage)
    {
        addAttr(isCoverPage ? JobSheets.STANDARD : JobSheets.NONE);
    }

    private void setFinishings(String finish)
    {
        if (finish.trim().length() == 0)
            throw new CFPrintEmptyAttributeException(CFPrintAttribute.FINISHINGS);

        PrintRequestAttribute a = PrintUtil.getFinishings(finish.trim());
        if (a != null)
        {
            addAttr(a);
        }
        else
        {
            throw new CFPrintInvalidValueException(CFPrintAttribute.FINISHINGS, finish, PrintUtil.getFinishingsValues());
        }
    }

    private void setJobHoldUntil(Date dateTime)
    {
        try
        {
            addAttr(new JobHoldUntil(dateTime));
        }
        catch (IllegalArgumentException e)
        {
            throw new CFPrintInvalidValueException(CFPrintAttribute.JOB_HOLD_UNITL, dateTime.toString(), e.getLocalizedMessage());
        }
    }

    private void setJobname(String jobName)
    {
        if (jobName.trim().length() == 0)
            throw new CFPrintEmptyAttributeException(CFPrintAttribute.JOBNAME);

        // Don't trim.  It should be just what was specified since it is a client string.
        addAttr(new JobName(jobName, null));
    }

    private void setJobPriority(int priority)
    {
        try
        {
            addAttr(new JobPriority(priority));
        }
        catch (IllegalArgumentException e)
        {
            throw new CFPrintInvalidValueException(CFPrintAttribute.JOB_PRIORITY, String.valueOf(priority), e.getLocalizedMessage());
        }
    }

    private void setMedia(String tagname, String name)
    {
        name = name.trim();
        if (name.length() == 0)
            throw new CFPrintEmptyAttributeException(tagname);

        // see if it is one of the cfdocument papersize names that has to be converted to a Media name
        PrintRequestAttribute a = PrintUtil.getMediaFromCFDocumentPapersize(name);
        if (a != null)
        {
            media = a.toString();
        }
        else
        {
            // will be verified and set in attribute set once printer is known
            media = name;
        }
    }

    private void setNumberUp(int num)
    {
        try
        {
            addAttr(new NumberUp(num));
        }
        catch (IllegalArgumentException e)
        {
            throw new CFPrintInvalidValueException(CFPrintAttribute.NUMBERUP, String.valueOf(num), e.getLocalizedMessage());
        }
    }

    /**
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    private void setOrientation(String orientation)
    {
        if (orientation.trim().length() == 0)
            throw new CFPrintEmptyAttributeException(CFPrintAttribute.ORIENTATION);

        PrintRequestAttribute a = PrintUtil.getOrientation(orientation.trim());
        if (a != null)
        {
            addAttr(a);
        }
        else
        {
            throw new CFPrintInvalidValueException(CFPrintAttribute.ORIENTATION, orientation, PrintUtil.getOrientationValues());
        }
    }

    private void setPageScaling(String pageScaling)
    {
        if (!this.pdfAttributes.setPageScaling(pageScaling))
            throw new CFPrintInvalidValueException(CFPrintAttribute.PAGE_SCALING, pageScaling, this.pdfAttributes.getPageScalingValues());
    }

    private void setPageSubset(String pageSubset)
    {
        if (!this.pdfAttributes.setPageSubset(pageSubset))
            throw new CFPrintInvalidValueException(CFPrintAttribute.PAGE_SUBSET, pageSubset, this.pdfAttributes.getPageSubsetValues());
    }


    private void setPresentationDirection(String dir)
    {
        if (dir == null || dir.trim().length() == 0)
            throw new CFPrintEmptyAttributeException(CFPrintAttribute.PRESENTATION_DIRECTION);

        PrintRequestAttribute a = PrintUtil.getPresentationDirection(dir.trim());
        if (a != null)
        {
            addAttr(a);
        }
        else
        {
            throw new CFPrintInvalidValueException(CFPrintAttribute.PRESENTATION_DIRECTION, dir, PrintUtil.getPresentationDirectionValues());
        }
    }

    private void setQuality(String quality)
    {
        if (quality.trim().length() == 0)
            throw new CFPrintEmptyAttributeException(CFPrintAttribute.QUALITY);

        PrintRequestAttribute a = PrintUtil.getPrintQuality(quality.trim());
        if (a != null)
        {
            addAttr(a);
        }
        else
        {
            throw new CFPrintInvalidValueException(CFPrintAttribute.QUALITY, quality, PrintUtil.getPrintQualityValues());
        }
    }

    /**
     * Used by the print job to set JobOriginatingUserName only if System.property("user.name") is null.
     *
     * @param name
     */
    private void setRequestingUsername(String name)
    {
        if (name != null)
            addAttr(new RequestingUserName(name, null));
        else
            throw new CFPrintEmptyAttributeException(CFPrintAttribute.REQUESTING_USERNAME);
    }

    private void setReversePages(boolean isReversePages)
    {
        this.pdfAttributes.setReversePages(isReversePages);
    }

    private void setSides(String sides)
    {
        if (sides.trim().length() == 0)
            throw new CFPrintEmptyAttributeException(CFPrintAttribute.SIDES);

        PrintRequestAttribute a = PrintUtil.getSides(sides.trim());
        if (a != null)
        {
            addAttr(a);
        }
        else
        {
            throw new CFPrintInvalidValueException(CFPrintAttribute.SIDES, sides, PrintUtil.getSidesValues());
        }
    }

    private void setUsePdfPageSize(boolean isUsePdfPageSize)
    {
        this.pdfAttributes.setUsePdfPageSize(isUsePdfPageSize);
    }

    private boolean addAttr(Attribute a)
    {
        boolean ret = false;
        if (!prSet.containsKey(a.getCategory()))
        {
            ret = prSet.add(a);
        }

        return ret;
    }

    public int doStartTag() throws JspException
    {
        // Tag Start monitoring event. This exists with onTagEnd() which is invoked at the end of the tag processing.
        // If you remove this make sure you remove the onTagEnd() also.
        onTagStart();

        // Is printing allowed in this edition?
        FeatureRouter.getInstance().allowFeature(
                EFRConstants.server_side_printing.intValue(),
                tagNameFromClass(), null);

        validateAttributes();

        // todo: what does this mean
        return SKIP_BODY;

    }

    public int doEndTag() throws JspException
    {
        if (type.equals(TYPE_PDF))
        {
            PrintPDF pdf = new PrintPDF(pageContext, source, password);
            pdf.print(printer, prSet, media, pdfAttributes);
        }

        release();

        // Invoke monitoring event handlers to capture this tag's monitor data. This exists with onTagStart() 
        // at the tagStart.  If you remove this make sure you remove the tagStart() invocation near the tag start
        onTagEnd();

        return EVAL_PAGE;
    }

    private void validateAttributes()
    {
        // The CF struct attribute

        if (this.attributeStruct != null)
        {
            // add any attributes that aren't already in the set from the top-level attributes since
            // they have precendence over attributes specified in this structure
            Enumeration keyEnum = attributeStruct.keys();
            while (keyEnum.hasMoreElements())
            {
                String key = ((String) keyEnum.nextElement()).toLowerCase();

                // Make sure the key is the name of a valid print request attribute for the CFStruct attribute.
                if (!CFPrintAttribute.isCFStructAttribute(key))
                {
                    throw new CFPrintInvalidAttributeException(key, ATTR_ATTRIBUTE_STRUCT,
                            CFPrintAttribute.getCFStructAttributes());
                }

                // Make sure there is a value.
                Object value = attributeStruct.get(key);
                if (value == null)
                    throw new CFPrintEmptyAttributeException(key);

                // Translate from the CF key/value to the equivalent PrintRequestAttribute.
                // If using the CF Cast operator to cast to a string, a null value is translated
                // to a "" value so the setters don't need to check for null.
                try
                {
                    if (key.equals(CFPrintAttribute.AUTO_ROTATE_AND_CENTER))
                    {
                        setAutoRotateAndCenter(Cast._boolean(value));
                    }
                    else if (key.equals(CFPrintAttribute.COLLATE) || key.equals(CFPrintAttribute.SHEET_COLLATE))
                    {
                        setCollate(Cast._boolean(value));
                    }
                    else if (key.equals(CFPrintAttribute.COLOR) || key.equals(CFPrintAttribute.CHROMATICITY))
                    {
                        setColor(Cast._boolean(value));
                    }
                    else if (key.equals(CFPrintAttribute.COPIES))
                    {
                        setCopies(Cast._int(value));
                    }
                    else if (key.equals(CFPrintAttribute.COVERPAGE) || key.equals(CFPrintAttribute.JOBSHEETS))
                    {
                        setCoverpage(Cast._boolean(value));
                    }
                    else if (key.equals(CFPrintAttribute.FIDELITY))
                    {
                        setFidelity(Cast._boolean(value));
                    }
                    else if (key.equals(CFPrintAttribute.FINISHINGS))
                    {
                        setFinishings(Cast._String(value));
                    }
                    else if (key.equals(CFPrintAttribute.JOB_HOLD_UNITL))
                    {
                        setJobHoldUntil(Cast._Date(value));
                    }
                    else if (key.equals(CFPrintAttribute.JOBNAME))
                    {
                        setJobname(Cast._String(value));
                    }
                    else if (key.equals(CFPrintAttribute.JOB_PRIORITY))
                    {
                        setJobPriority(Cast._int(value));
                    }
                    else if (key.equals(CFPrintAttribute.MEDIA))
                    {
                        setMedia(key, Cast._String(value));
                    }
                    else if (key.equals(CFPrintAttribute.ORIENTATION) || key.equals(CFPrintAttribute.ORIENTATION_REQUESTED))
                    {
                        setOrientation(Cast._String(value));
                    }
                    else if (key.equals(CFPrintAttribute.PAGES))
                    {
                        setPages(Cast._String(value));
                    }
                    else if (key.equals(CFPrintAttribute.PAPER))
                    {
                        setPaper(Cast._String(value));
                    }
                    else if (key.equals(CFPrintAttribute.PAGE_SCALING))
                    {
                        setPageScaling(Cast._String(value));
                    }
                    else if (key.equals(CFPrintAttribute.PAGE_SUBSET))
                    {
                        setPageSubset(Cast._String(value));
                    }
                    else if (key.equals(CFPrintAttribute.NUMBERUP))
                    {
                        setNumberUp(Cast._int(value));
                    }
                    if (key.equals(CFPrintAttribute.USE_PDF_PAGE_SIZE))
                    {
                        setUsePdfPageSize(Cast._boolean(value));
                    }
                    else if (key.equals(CFPrintAttribute.PRESENTATION_DIRECTION))
                    {
                        setPresentationDirection(Cast._String(value));
                    }
                    else if (key.equals(CFPrintAttribute.PRINTER))
                    {
                        setPrinter(Cast._String(value));
                    }
                    else if (key.equals(CFPrintAttribute.REQUESTING_USERNAME))
                    {
                        setRequestingUsername(Cast._String(value));
                    }
                    else if (key.equals(CFPrintAttribute.REVERSE_PAGES))
                    {
                        setReversePages(Cast._boolean(value));
                    }
                    else if (key.equals(CFPrintAttribute.QUALITY))
                    {
                        setQuality(Cast._String(value));
                    }
                    else if (key.equals(CFPrintAttribute.SIDES))
                    {
                        setSides(Cast._String(value));
                    }
                    else
                    {
                        //System.err.println("action for attribute " + key + " missing");
                    }
                }
                catch (Cast.BooleanStringConversionException e1)
                {
                    throw new CFPrintInvalidValueException(key, e1.value, "yes | no | true | false");
                }
                catch (Cast.NumberConversionException e2)
                {
                    throw new CFPrintInvalidValueException(key, e2.source, "numbers");
                }
            }
        }


        // The Java PrintRequestAttributeSet attribute.
        if (this.printRequestAttributeSet != null)
        {
            // add any attributes that aren't already in the set from either top-level tags or the
            // attributeStruct attribute which take precedence.           
            Attribute[] javaAttributeList = this.printRequestAttributeSet.toArray();
            for (int i = 0; i < javaAttributeList.length; i++)
            {
                addAttr(javaAttributeList[i]);
            }
        }

        // Make sure there is a source specified, (if debug output isn't requested) since it is required.
        if (type.equals(TYPE_PDF) && source == null)
        {
            throw new CFPrintEmptyAttributeException(ATTR_SOURCE);
        }

    }

    

    // permission for sandbox security
    private static final GenericTagPermission tp = new GenericTagPermission(CFPRINT_TAG);

    protected Permission getPermission()
    {
        return tp;
    }
}
 
