/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2001-2003 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package coldfusion.mail;

import java.security.*;
import java.security.cert.*;
import javax.security.auth.x500.*;
import java.math.*;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilePermission;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.net.IDN;
import java.net.URL;
import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Observable;
import java.util.Observer;

import javax.activation.DataSource;
import javax.mail.Address;
import javax.mail.AuthenticationFailedException;
import javax.mail.Header;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.NoSuchProviderException;
import javax.mail.SendFailedException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.AddressException;
import javax.mail.internet.ContentType;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.ParseException;
import javax.mail.util.ByteArrayDataSource;
import javax.naming.InitialContext;
import javax.servlet.ServletContext;
import javax.servlet.jsp.tagext.Tag;
import org.bouncycastle.crypto.generators.BCrypt;
import org.bouncycastle.crypto.generators.SCrypt;

import coldfusion.log.CFLogs;
import coldfusion.log.Logger;
import coldfusion.mail.core.MailDeliveryException;
import coldfusion.mail.core.MailExceptions.InvalidSpoolFileException;
import coldfusion.mail.core.MailExceptions.ServerMissingException;
import coldfusion.mail.core.MailExceptions.SpoolLockTimeoutException;
import coldfusion.mail.core.MailSessionException;
import coldfusion.tagext.mail.MailTag;
import coldfusion.runtime.AppHelper;
import coldfusion.runtime.ApplicationSettings;
import coldfusion.runtime.RuntimeServiceImpl;
import coldfusion.server.ConfigMap;
import coldfusion.server.LicenseService;
import coldfusion.server.MailSpoolService;
import coldfusion.server.SchedulerService;
import coldfusion.server.SecurityService;
import coldfusion.server.ServiceBase;
import coldfusion.server.ServiceException;
import coldfusion.server.ServiceFactory;
import coldfusion.tagext.validation.CFTypeValidatorFactory.InvalidEmailTypeException;
import coldfusion.util.PasswordUtils;
import coldfusion.util.RB;
import coldfusion.util.SoftCache;
import coldfusion.util.Utils;
import coldfusion.vfs.VFSFileFactory;
import coldfusion.wddx.Base64Encoder;

import com.sun.mail.smtp.SMTPSendFailedException;
import org.bouncycastle.asn1.x509.KeyUsage;
import org.bouncycastle.asn1.x509.X509Extensions;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.x509.X509V3CertificateGenerator;
import java.util.Random;
import java.util.concurrent.TimeUnit;
import java.util.Date;
import coldfusion.util.*;


/**
 * Manages the ColdFusion mail spool files and performs the delivery
 * of both spooled and non-spool mail.
 *
 * @author Xu Chen
 * @author Tom Jordahl
 */
public class MailSpooler extends ServiceBase implements MailSpoolService, Runnable, Observer
{
    private static final String KEYPASSWORD = "keypassword";
	private static final String KEYSTOREPASSWORD = "keystorepassword";
	private static final String PASSWORD = "password";
    private static final String OLDSEEDVALFORSM = "0yJ!@1$r8p0L@r1$6yJ!@1rj";
	private String emailSpoolDir = null;
    //private static String emailPath = null;
    private String emailUndeliverDir = null;
    private String mailServer = null;
    private String username = "";
    private String password = "";
    private int mailPort = -1;
    private int mailTimeout = -1;
    private Boolean useTLS = Boolean.FALSE;
    private Boolean useSSL = Boolean.FALSE;
    private boolean bMailSentLoggingEnable = true;
    private boolean bSpoolEnable = true;
    private File spoolDir = null;
    private File undeliverDir = null;
    private Logger logger = null;
    private Logger sentlogger = null;
    private boolean bMaintainConnections = true;
    private int maxDeliveryThreads = 10;
    private boolean spoolToMemory = false;
    private int spoolMessagesLimit = 50000;
    private boolean allowDownload = true;

    private final String separator = ":  ";  //separator for name value pair in the file
    private final String rootDirPattern = "{neo.rootdir}";
    private final int rootDirLen = rootDirPattern.length();
    private String severity = "warning";
    private long schedule_time = 0;
    private ConfigMap settings;
    private boolean bFastMail = false;
    private boolean bypassSoftCache = false;

    private File configfile = null;
    private String rootdir = null;
    private boolean mailSpoolerStarted=false;
    // Synchronized objects
    private final SpoolerSoftCache mailStorage = new SpoolerSoftCache();
    private final ConnectionPool connectionPool = new ConnectionPool();
    private final MailRWLock spoolLock = new MailRWLock();
    private final int SPOOL_LOCK_TIMEOUT = Integer.getInteger("coldfusion.spooltimeout", 60).intValue();
    private final List spoolFiles = Collections.synchronizedList(new LinkedList());
    private final Map badServers = Collections.synchronizedMap(new HashMap());
    private final LinkedList spoolMessages = new LinkedList();     // not wrapped so we can use first/last apis
    private Boolean sign = Boolean.FALSE;
    private String keystore = "";
    private String keystorePassword = "";
    private String keyPassword = "";
    private String keyAlias="";
    private String seed = null;
    private static final boolean IS_JVM_SECURITY_ENABLED = (System.getSecurityManager() != null);
    private FilePermission tempFolderPermission;
    
    public MailSpooler(File file, String rootdir) {
        this(file, rootdir, null);
    }
    
    public MailSpooler(File file, String rootdir, ServletContext application)
    {
        this.configfile = file;
        this.rootdir = rootdir;

        //set the watch flag and watchfile
        setEnableWatch(true);
        setWatchFile(configfile);
        if(application != null && IS_JVM_SECURITY_ENABLED) {
            tempFolderPermission = new FilePermission(Utils.getTempDir(application) + "*", "read");
        }
    }

    public void start() throws ServiceException
    {
        super.start();
        PasswordUtils.getInstance().addObserver(this);
        try
        {
            //set for the scheduler
            setSchedule(((Number) settings.get("schedule")).intValue());
            SchedulerService ss = ServiceFactory.getSchedulerService();
            ss.cancel(this);
            ss.schedule(this, (System.currentTimeMillis() + getSchedule()));

            settings.setUnchanged();
            settings.setConfigMapListener(this);

        }
        catch (Throwable ex)
        {
            throw new ServiceException(ex);
        }
    }

    public void load() throws ServiceException
    {
        try
        {
            settings = (ConfigMap) deserialize(configfile);
            settings.init(this, "configuration");
            try
            {
                if(ServiceFactory.getRuntimeService().isCommandLineCompile())
                {
                 Map appSetting = (Map)AppHelper.getApplicationSetting();
                 ApplicationSettings appSettings = new ApplicationSettings(appSetting);
                 Map appMapSettings = appSettings.loadAppMailSettings();
                 if(appMapSettings != null)
                 {
                    settings.putAll(appMapSettings);
                 }
                }
            }
            catch (Throwable e)
            {
                //For cli services only.
            }
            setSettings(settings);
            //Startup only at the start. Subsequent load shouldn't try to call doStartup(). Added for WatchService
            if(!mailSpoolerStarted)
                doStartup();
        }
        catch (Throwable ex)
        {
            throw new ServiceException(ex);
        }
    }

    public void stop() throws ServiceException
    {
        mailSpoolerStarted=false;
        try
        {
            SchedulerService ss = ServiceFactory.getSchedulerService();
            ss.cancel(this);
        }
        catch (Throwable t)
        {
            throw new ServiceException(t);
        }
    }

    public void store() throws ServiceException
    {
        serialize(settings, configfile);
        settings.setUnchanged();
    }

	public void addBCProvider(){
         BouncyCastleProvider bouncyCastleProvider = new BouncyCastleProvider();
         Security.addProvider(bouncyCastleProvider);
	}

     public X509Certificate generateCertificate(KeyPair keyPair,SamlKeyPairGeneratorData keyPairData) throws GeneralSecurityException{
 X509V3CertificateGenerator certGen = new X509V3CertificateGenerator();
        X500Principal dnName = new X500Principal(keyPairData.getSubjectDN()); //$NON-NLS-1$

        certGen.setSerialNumber(BigInteger.valueOf(Math.abs(new Random().nextInt())));
        certGen.setSubjectDN(dnName);
        certGen.setIssuerDN(dnName); // use the same
        certGen.setNotBefore(new Date(System.currentTimeMillis()));

        certGen.setNotAfter(new Date(System.currentTimeMillis() + TimeUnit.MILLISECONDS.convert(keyPairData.getValidityPeriod(), TimeUnit.DAYS)));
        certGen.setPublicKey(keyPair.getPublic());
        certGen.setSignatureAlgorithm(keyPairData.getSignatureAlgorithm()); //$NON-NLS-1$
        KeyUsage usage = new KeyUsage(KeyUsage.digitalSignature | KeyUsage.dataEncipherment);
        certGen.addExtension(X509Extensions.ExtendedKeyUsage, false, usage);
        return certGen.generate(keyPair.getPrivate());
	 }
    public Map getSettings()
    {
        return settings;
    }

    public void setSettings(Map settings)
    {
        // Get edition to set "FastMail" functionality
        LicenseService ls =  ServiceFactory.getLicenseService();
        if (ls != null)
            bFastMail = ls.allowFastMail();

        // Spool directory
        emailSpoolDir = (String) settings.get("spooldir");
        if (emailSpoolDir != null)
        {
            if (emailSpoolDir.indexOf(rootDirPattern) != -1)
            {
                emailSpoolDir = rootdir + emailSpoolDir.substring(emailSpoolDir.indexOf(rootDirPattern) + rootDirLen);
            }
        }

        // Undeliverable mail directory
        emailUndeliverDir = (String) settings.get("undeliverdir");
        if (emailUndeliverDir != null && emailUndeliverDir.length() > rootDirLen && emailUndeliverDir.indexOf(rootDirPattern) >= 0)
        {
            if (emailUndeliverDir.indexOf(rootDirPattern) != -1)
            {
                emailUndeliverDir = rootdir + emailUndeliverDir.substring(emailUndeliverDir.indexOf(rootDirPattern) + rootDirLen);
            }

        }

        logger = CFLogs.MAIL_LOG; //ServiceFactory.getLoggingService().getLogger("coldfusion.mail");
        sentlogger = CFLogs.MAILSENT_LOG; //ServiceFactory.getLoggingService().getLogger("coldfusion.mailsent");

        setServer((String) settings.get("server"));
        setPort(((Number) settings.get("port")).intValue());
        setTimeout(((Number) settings.get("timeout")).intValue());
        setMailSentLoggingEnable(((Boolean) settings.get("mailsentloggingenable")).booleanValue());
        setSeverity( ((settings.get("severity") != null) ? (String) settings.get("severity") : "warning") );
        setSchedule(((Number) settings.get("schedule")).intValue());

        // New additions, be defensive
        Object val = settings.get("spoolenable");
        if (val != null){
            bSpoolEnable = ((Boolean) val).booleanValue();
            setSpoolEnable(bSpoolEnable);
        }

        val = settings.get("allowdownload");
        if (val != null)
        {
            allowDownload = ((Boolean) val).booleanValue();
        }
        else
        {
            settings.put("allowdownload", Boolean.TRUE);
        }
        setAllowDownload(((Boolean) settings.get("allowdownload")).booleanValue());

        val = settings.get("usetls");
        if (val != null){
            useTLS = (Boolean) val;
            setUseTLS(useTLS.booleanValue());
        }
        val = settings.get("usessl");
        if (val != null){
            useSSL = (Boolean) val;
            setUseSSL(useSSL.booleanValue());
        }
        val = settings.get("username");
        if (val != null){
            username = (String) val;
            setUsername(username);
        }
        val = settings.get(PASSWORD);
        if (val != null)
            password = (String) val;

        val = settings.get("sign");
        if(val != null){
            sign = ((Boolean)val);
            setSign(sign.booleanValue());
        }

        val = settings.get("keystore");
        if(val != null){
            keystore = (String)val;
            setKeystore(keystore);
        }

        val = settings.get(KEYSTOREPASSWORD);
        if(val != null)
            keystorePassword = (String)val;

        val = settings.get(KEYPASSWORD);
        if(val != null)
            keyPassword = (String)val;

        val = settings.get("keyalias");
        if(val != null){
            keyAlias = (String)val;
            setKeyAlias(keyAlias);
        }

        if (bFastMail)
        {
            // Maintain mail server connections?
            val = settings.get("maintainconnections");
            if (val != null)
            {
                bMaintainConnections = ((Boolean) val).booleanValue();
            }
            // spool to memory or disk?
            val = settings.get("spooltomemory");
            if (val != null)
            {
                spoolToMemory = ((Boolean) val).booleanValue();
            }
            // How many delivery threads max? (Enterprise only)
            val = settings.get("maxthreads");
            if (val != null)
                maxDeliveryThreads = ((Number) val).intValue();

            // How many messages in memory? (Enterprise only)
            val = settings.get("spoolmessageslimit");
            if (val != null)
                spoolMessagesLimit = ((Number) val).intValue();
        }
        else
        {
            // Always set it to false if the license doesn't allow it
            // Don't user setter functions as they are noop when fastmail == false!
            bMaintainConnections = false;
            spoolToMemory = false;
        }
    }

    /**
     * Startup routine for the Mail Spooler thread.
     * Perform any initialization or object creation here.
     * Called by the Service load() function.
     */
    private void doStartup()
    {
        // Create spool directory
        spoolDir = new File(emailSpoolDir);
        if (!spoolDir.exists())
        {
            spoolDir.mkdirs();
        }

        // Create undelivered directory
        undeliverDir = new File(emailUndeliverDir);
        if (!undeliverDir.exists())
        {
            undeliverDir.mkdirs();
        }
        mailSpoolerStarted = true;
    }


    public synchronized boolean isMailSentLoggingEnable()
    {
        return bMailSentLoggingEnable;
    }

    public synchronized void setMailSentLoggingEnable(boolean bEnable)
    {
        bMailSentLoggingEnable = bEnable;
        settings.put("mailsentloggingenable", (bEnable) ? Boolean.TRUE : Boolean.FALSE);
    }

    public synchronized String getServer()
    {
    	Map appSetting = (Map)AppHelper.getApplicationSetting();
    	if(appSetting !=null)
    	{
        	Map smtpServerSettings = (Map) appSetting.get("smtpServerSettings");
        	if(smtpServerSettings != null)
        	{
            	String smtpserver = (String) smtpServerSettings.get("server");
            	if(smtpserver != null)
            			return smtpserver;
        	}
    	}
        return mailServer;
    }

    public synchronized String getServerFromAdministrator()
    {
        return mailServer;
    }

    public synchronized void setServer(String server)
    {
        mailServer = server;
        settings.put("server", server);
    }
    
    public synchronized void deleteServer (String server)
    {
    	String currentServer = (String)settings.get("server");
    	if (currentServer != null && currentServer.equalsIgnoreCase(server))
    	{
    		settings.put("server", "");
    		settings.put("usetls", false);
    		settings.put("usessl", false);
    		settings.put("username", "");
    		settings.put(PASSWORD, "");
    		settings.put("sign", false);
    		settings.put("keystore", "");
    		settings.put(KEYSTOREPASSWORD, "");
    		settings.put(KEYPASSWORD, "");
    		settings.put("keyalias", "");
    		resetSettings();
    		settings.setChanged(settings, "server", "");
    	}
    }

    private void resetSettings()
    {
        mailServer = null;
        username = "";
        password = "";
        useTLS = Boolean.FALSE;
        useSSL = Boolean.FALSE;
        sign = Boolean.FALSE;
        keystore = "";
        keystorePassword = "";
        keyPassword = "";
        keyAlias="";
        allowDownload = true;
    }
    
    /**
     * This method should not be used by anyone other than Server Manager
     * 
     * @return
     */
    public String getSMPassword()
    {
        // check if it is admin, as CAR/Migration should be in admin
        SecurityService security = ServiceFactory.getSecurityService();
        security.authenticateAdmin();

        Map appSetting = (Map) AppHelper.getApplicationSetting();
        if (appSetting != null)
        {
            Map smtpServerSettings = (Map) appSetting.get("smtpServerSettings");
            if (smtpServerSettings != null)
            {
                String password = (String) smtpServerSettings.get(PASSWORD);
                if (password != null)
                    return PasswordUtils.encryptWith3DES(password, OLDSEEDVALFORSM);
            }
        }
        return PasswordUtils.reEncryptForSM(password, this.seed, OLDSEEDVALFORSM);
    }

    public String getPassword()
    {
    	Map appSetting = (Map)AppHelper.getApplicationSetting();
    	if(appSetting != null)
    	{
        	Map smtpServerSettings = (Map) appSetting.get("smtpServerSettings");
        	if(smtpServerSettings != null)
        	{
            	String password = (String) smtpServerSettings.get(PASSWORD);
            	if(password != null)
                    return encryptPassword(password);
        	}
    	}
        return password;
    }
    
    public String getKeystorePassword()
    {
        return keystorePassword;
    }

    public String getKeyPassword()
    {
        return keyPassword;
    }
    
    /**
     * Required for mailImpl to return decrypted password.
     * @return
     */
    protected String getSeed()
    {
    	return this.seed;
    }

    /**
     * Method gets invoked by Observable PasswordUtils to update seed.
     */
    @Override
    public void update(Observable o, Object arg)
    {
        String oldSeed = this.seed;

        if (o instanceof PasswordUtils)
        {
            if (arg != null && arg instanceof String)
            {
                String seedVal = (String) arg;
                if (seedVal != null && seedVal.length() > 0)
                {
                    // No need to re-encrypt, just set seed.
                    // This will happen only at start time.
                    this.seed = seedVal;
                    if (oldSeed == null)
                    {
                        return;
                    }
                    else
                    {
                        reEncryptPassword(oldSeed);
                    }
                }
            }
        }
    }

    /**
     * Encrypt the password.
     * 
     * @param p
     * @return
     */
    private String encryptPassword(String p)
    {
        return PasswordUtils.encryptPassword(p, seed);
    }

    /**
     * Decrypt the password.
     * 
     * @param password
     * @return decrypted password
     */
    private String decryptPassword(String password)
    {
        return PasswordUtils.decryptPassword(password, this.seed);
    }

    /**
     * encrypt and set the appropriate password Type can be "password", "keystorepassword", "keyPassword"
     * 
     * @param oldSeed
     */
    private void reEncryptPassword(String oldSeed)
    {
        if (oldSeed != null && oldSeed.equalsIgnoreCase(this.seed))
        {
            return;
        }

        // check if it is admin, as set seed should be in admin
        SecurityService security = ServiceFactory.getSecurityService();
        security.authenticateAdmin();

        String oldPassword = getPassword();
        String newPassword = null;

        synchronized (settings)
        {
            try
            {
                if (oldPassword != null && oldPassword.length() > 0)
                {
                    newPassword = PasswordUtils.reEncryptWithNewSeed(oldPassword, oldSeed, this.seed);
                    this.password = newPassword;
                    settings.put(PASSWORD, newPassword);
                }

                oldPassword = getKeystorePassword();
                if (oldPassword != null && oldPassword.length() > 0)
                {
                    newPassword = PasswordUtils.reEncryptWithNewSeed(oldPassword, oldSeed, this.seed);
                    this.keystorePassword = newPassword;
                    settings.put(KEYSTOREPASSWORD, newPassword);
                }

                oldPassword = getKeyPassword();
                if (oldPassword != null && oldPassword.length() > 0)
                {
                    newPassword = PasswordUtils.reEncryptWithNewSeed(oldPassword, oldSeed, this.seed);
                    this.keyPassword = newPassword;
                    settings.put(KEYPASSWORD, newPassword);
                }
            }
            catch (Exception e)
            {
                CFLogs.SERVER_LOG.error(e);
            }
        }
    }

    /**
     * encrypt and set the appropriate password Type can be "password", "keystorepassword", "keyPassword"
     * 
     * @param oldSeed
     * @param oldAlgoValue
     * @param majorVersion
     * @param minorVersion
     */
    public void reEncryptPasswordForMigration(String oldSeed, String oldAlgoValue, int majorVersion, int minorVersion)
    {
        // check if it is admin, as CAR/Migration should be in admin
        SecurityService security = ServiceFactory.getSecurityService();
        security.authenticateAdmin();

        synchronized (settings)
        {
            if (!PasswordUtils.isAESS(majorVersion, minorVersion))
            {
                // this call will encrypt the password and then set it.
                setPassword(password, PASSWORD);
                setPassword(keystorePassword, KEYSTOREPASSWORD);
                setPassword(keyPassword, KEYPASSWORD);
                return;
            }

            if (oldSeed == null || (oldSeed != null && oldSeed.equalsIgnoreCase(this.seed)))
            {
                return;
            }

            String oldPassword = getPassword();
            String newPassword = null;
            try
            {
                if (oldPassword != null && oldPassword.length() > 0)
                {
                    newPassword = PasswordUtils.reEncryptWithNewSeed(oldPassword, oldSeed, this.seed, oldAlgoValue,
                            majorVersion, minorVersion);
                    this.password = newPassword;
                    settings.put(PASSWORD, newPassword);
                }

                oldPassword = getKeystorePassword();
                if (oldPassword != null && oldPassword.length() > 0)
                {
                    newPassword = PasswordUtils.reEncryptWithNewSeed(oldPassword, oldSeed, this.seed, oldAlgoValue,
                            majorVersion, minorVersion);
                    this.keystorePassword = newPassword;
                    settings.put(KEYSTOREPASSWORD, newPassword);
                }

                oldPassword = getKeyPassword();
                if (oldPassword != null && oldPassword.length() > 0)
                {
                    newPassword = PasswordUtils.reEncryptWithNewSeed(oldPassword, oldSeed, this.seed, oldAlgoValue,
                            majorVersion, minorVersion);
                    this.keyPassword = newPassword;
                    settings.put(KEYPASSWORD, newPassword);
                }
            }
            catch (Exception e)
            {
                CFLogs.SERVER_LOG.error(e);
            }
        }
    }

    /*
     * Set the settings for archive.CF9 and before, password needs to be encrypted, CF10 and above re-encrypted.
     */
    public void setSettingsForArchive(Map settings, String archiveSeed, String oldAlgoValue, int majorVersion,
            int minorVersion)
    {
        setSettings(settings);
        reEncryptPasswordForMigration(archiveSeed, oldAlgoValue, majorVersion, minorVersion);
    }

    public void setPassword(String password)
    {
        setPassword(password, PASSWORD);
    }

    public void setKeystorePassword(String keystorePassword)
    {
        setPassword(keystorePassword, KEYSTOREPASSWORD);
    }

    public void setKeyPassword(String keyPassword)
    {
        setPassword(keyPassword, KEYPASSWORD);
    }

    /**
     * encrypt and set the appropriate password Type can be "password", "keystorepassword", "keyPassword"
     * 
     * @param password
     */
    private void setPassword(String password, String type)
    {
        password = encryptPassword(password);
        if (type.equalsIgnoreCase(PASSWORD))
        {
            this.password = password;
            settings.put(PASSWORD, password);
        }
        else if (type.equalsIgnoreCase(KEYSTOREPASSWORD))
        {
            this.keystorePassword = password;
            settings.put(KEYSTOREPASSWORD, password);
        }
        else if (type.equalsIgnoreCase(KEYPASSWORD))
        {
            this.keyPassword = password;
            settings.put(KEYPASSWORD, password);
        }
    }
	
    public String getUsername()
    {
    	Map appSetting = (Map)AppHelper.getApplicationSetting();
    	if(appSetting != null)
    	{
        	Map smtpServerSettings = (Map) appSetting.get("smtpServerSettings");
        	if(smtpServerSettings != null)
        	{
            	String username = (String) smtpServerSettings.get("username");
            	if(username != null)
            			return username;
        	}
    	}
        return username;
    }

    public void setUsername(String username)
    {
        this.username = username;
        settings.put("username", username);
    }

    public synchronized String getSeverity()
    {
        return severity;
    }

    public synchronized void setSeverity(String level)
    {
        severity = level;
        logger.setPriority(level);
        settings.put("severity", level);
    }

    public synchronized int getPort()
    {
        return mailPort;
    }

    public synchronized void setPort(int port)
    {
        mailPort = port;
        settings.put("port", new Integer(port));
    }

    public synchronized void setPort(double port)
    {
        mailPort = (new Double(port)).intValue();
        settings.put("port", new Integer(mailPort));
    }

    public synchronized int getTimeout()
    {
        return mailTimeout;
    }

    public synchronized void setTimeout(int timeout)
    {
        mailTimeout = timeout;
        settings.put("timeout", new Integer(timeout));
    }

    public synchronized void setTimeout(double timeout)
    {
        mailTimeout = (new Double(timeout)).intValue();
        settings.put("timeout", new Integer(mailTimeout));
    }

    public synchronized long getSchedule()
    {
        return schedule_time;
    }

    public synchronized void setSchedule(int sched)
    {
        if (sched <= 0) sched = 1;  // enforce 1 second minimum
        schedule_time = sched * 1000;
        settings.put("schedule", new Integer(sched));
    }

    public synchronized void setSchedule(double sched)
    {
        int s = (new Double(sched)).intValue();
        if (s <= 0) s = 1;  // enforce 1 second minimum
        schedule_time = s * 1000;
        settings.put("schedule", new Integer(s));
    }

    public synchronized boolean isSpoolEnable()
    {
        return bSpoolEnable;
    }

    public synchronized void setSpoolEnable(boolean bEnable)
    {
        bSpoolEnable = bEnable;
        settings.put("spoolenable", (bEnable) ? Boolean.TRUE : Boolean.FALSE);
    }

    public synchronized boolean isAllowDownload()
    {
        return allowDownload;
    }

    public synchronized void setAllowDownload(boolean bEnable)
    {
        allowDownload = bEnable;
        settings.put("allowdownload", (bEnable) ? Boolean.TRUE : Boolean.FALSE);
    }

    public synchronized boolean isMaintainConnections()
    {
        return bMaintainConnections;
    }

    public synchronized void setMaintainConnections(boolean bEnable)
    {
        // don't allow Java to set this if license doesn't allow it
        if (bFastMail)
        {
            bMaintainConnections = bEnable;
            settings.put("maintainconnections", (bEnable) ? Boolean.TRUE : Boolean.FALSE);
        }
    }

    public synchronized int getMaxDeliveryThreads()
    {
        return maxDeliveryThreads;
    }

    public synchronized void setMaxDeliveryThreads(int count)
    {
        maxDeliveryThreads = count;
        settings.put("maxthreads", new Integer(count));
    }

    public synchronized void setMaxDeliveryThreads(double count)
    {
        maxDeliveryThreads = (new Double(count)).intValue();
        settings.put("maxthreads", new Integer(maxDeliveryThreads));
    }

    public synchronized boolean isSpoolToMemory()
    {
        return spoolToMemory;
    }

    public synchronized void setSpoolToMemory(boolean bEnable)
    {
        // don't allow Java to set this if license doesn't allow it
        if (bFastMail)
        {
            spoolToMemory = bEnable;
            settings.put("spooltomemory", (bEnable) ? Boolean.TRUE : Boolean.FALSE);
        }
    }

    public synchronized int getSpoolMessagesLimit()
    {
        return spoolMessagesLimit;
    }

    public synchronized void setSpoolMessagesLimit(int count)
    {
        spoolMessagesLimit = count;
        settings.put("spoolmessageslimit", new Integer(count));
    }

    public synchronized void setSpoolMessagesLimit(double count)
    {
        spoolMessagesLimit = (new Double(count)).intValue();
        settings.put("spoolmessageslimit", new Integer(spoolMessagesLimit));
    }

    public void setUseTLS(boolean bEnable)
    {
        useTLS = Boolean.valueOf(bEnable);
        settings.put("usetls", useTLS);
    }

    public boolean isUseTLS()
    {
    	return useTLS.booleanValue();
    }

    public void setUseSSL(boolean bEnable)
    {
        useSSL =Boolean.valueOf(bEnable);
        settings.put("usessl", useSSL);
    }

    public boolean isUseSSL()
    {
    	return useSSL.booleanValue();
    }

    /**
     * Used for testing purposes ONLY.
     * Force reading mail from spool files, not from cache.
     */
    public void setBypassSoftCache(boolean bEnable)
    {
        bypassSoftCache = bEnable;
    }

    public void setSign(boolean sign)
    {
        this.sign = Boolean.valueOf(sign);
        settings.put("sign", this.sign);
    }

    public boolean isSign()
    {
        return sign.booleanValue();
    }

    public String getKeystore()
    {
        return keystore;
    }

    public void setKeystore(String keystore)
    {
        this.keystore = keystore;
        settings.put("keystore", keystore);
    }

    public String getKeyAlias()
    {
        return keyAlias;
    }

    public void setKeyAlias(String keyAlias)
    {
        this.keyAlias = keyAlias;
        settings.put("keyalias", keyAlias);
    }

    /**
     * Verify the connection to the server in the admin settings
     * Currently only checks the first server on the list against the default port.
     * @return true if we are able to connect.
     */
    public boolean verifyServer()
    {
        // getServer() is a csv list of mailserver and backup servers.
        HostImpl[] list = HostImpl.parseServer(getServer(), getPort(), getUsername(), decryptPassword(getPassword()), getTimeout(), useTLS, useSSL);
        if (list.length <1)
            return false;

        // Only test the first server
        HostImpl ms = list[0];
        try
        {
            Session s = ms.getSession();
            String protocol = s.getProperty("mail.transport.protocol");
            if (protocol == null) protocol = ms.getProtocol(); 
            Transport t = s.getTransport(protocol);
            t.connect(ms.getHost(), ms.getPort(), ms.getUsername(), ms.getPassword());
        }
        catch (Exception e)
        {
            return false;
        }

        return true;
    }

    /*  Never used - throws exception instead of returning true/false
	private void verifyServerInternal(HostImpl ms) throws MessagingException
	{
		Session s;
		try
		{
			s = ms.getSession();
		}
		catch (Exception e)
		{
			throw new MessagingException(e.getMessage());
		}
		try
		{
			Provider p = s.getProvider("smtp");
			Transport t = s.getTransport(p);
			t.connect(ms.getHost(), ms.getPort(), null, null);
		}
		catch (MessagingException e)
		{
			throw e;
		}
	}
    */

    /**
     * Sets global defaults in the mail object for server, port and timeout
     * if the tag hasn't already set them
     *
     * @param mailobj
     * @throws MailSessionException
     */
    public void validate(Object o) throws MailSessionException
    {
    	MailImpl mailobj = (MailImpl) o;
        boolean bSetSource = false;
        HostImpl[] servers = mailobj.getServers();

        // If no servers specified via the tag, get globals
        if (servers == null || servers.length == 0)
        {
            bSetSource = true;
            int port = mailobj.getPort();
            int timeout = mailobj.getTimeout();

            // Use tag attributes (if any) as defaults for the other stuff,
            // otherwise use the global settings.
            final String username = mailobj.getUsername();
            final String password = mailobj.getPassword();
            servers = HostImpl.parseServer(getServer(),
                                           (port > 0 ? port : getPort()),
                                           username != null ? username : getUsername(),
                                           password != null ? password : decryptPassword(getPassword()),
                                           (timeout > 0 ? timeout : getTimeout()),
                                           useTLS, useSSL);
            if (servers.length == 0)
            {
                logger.error("Mail: '" + mailobj.getSubject() + "' From: '" + mailobj.getSender() +
                             "' was missing server information.");
                throw new ServerMissingException();
            }
            
            // Bug 3583375 When SSL/TLS is enabled and spooling is enabled the mail written to file is not containing
            // whether SSL is enabled or not (when server is not specified in the tag)
            mailobj.setUseSSL(useSSL);
            mailobj.setUseTLS(useTLS);
        }
        else
        {
            // Fill in global port and timeout if needed.
            for (int i = 0; i < servers.length; i++)
            {
                HostImpl ms = servers[i];

                if (ms.getPort() == -1)
                {
                    int port = getPort();
                    ms.setPort(port);
                    bSetSource = true;
                }
                if (ms.getTimeout() == -1)
                {
                    int timeout = getTimeout();
                    ms.setTimeout(timeout);
                    bSetSource = true;
                }

            }
        }

        // If we changed anything, put the whole list back.
        if (bSetSource)
        {
            mailobj.setServers(servers);
        }

        // If details for mail signing is not specified in the tag, get the global.
        if(mailobj.getSign() == null)
            mailobj.setSign(sign);

        if(mailobj.getKeystoreFile() == null)
        {
            /*77655 Since the values for keystore,keystorePassword,keyPassword,keyAlias come from the xml as "" later it was considering their values
             (keyPassword) as "" which was causing the error.So doing this check and setting the values only if they are not null and not empty.

             */
            if(keystore != null && keystore.length() != 0)
            {
                mailobj.setKeystoreFile(keystore);
            }
            if(keystorePassword != null && keystorePassword.length() != 0)
            {
             mailobj.setKeystorePassword(decryptPassword(keystorePassword));
            }
            if(keyPassword != null && keyPassword.length() != 0)
            {
              mailobj.setKeyPassword(decryptPassword(keyPassword));
            }
            if(keyAlias != null && keyAlias.length() != 0)
            {
            mailobj.setKeyalias(keyAlias);
            }
        }
    }

    /**
     * Store or deliver mail, depending on spool settings.
     *
     * @param mailobj the message to handle
     * @throws MailSessionException if there is a problem spooling the message
     * @throws MailDeliveryException if there is a problem delivering the message
     */
    public void storeMail(Object o) throws MailSessionException, MailDeliveryException
    {
    	MailImpl mailobj = (MailImpl) o;
        // Check the tag, then the admin setting to see if we should spool this message
        boolean bSpoolIt;
        if (mailobj.getSpoolEnable() != null)
        {
            bSpoolIt = mailobj.getSpoolEnable().booleanValue();
        }
        else
        {
            bSpoolIt = isSpoolEnable();
        }

        if (!(bSpoolIt))
        {
            // send the message, no spool file
            deliver(mailobj, null);
            return;
        }

        // Put the message on the work queue, no spool file
        if (spoolToMemory)
        {
            // Check to make sure we aren't using all the memory
            if (spoolMessages.size() < spoolMessagesLimit)
            {
                postMessage(mailobj);
                return;
            }
        }

        // Spool the mail to a file
        String filename = null;
        try
        {
            /**
             * security model, adding this to grant access for the sandbox
             */
            if (System.getSecurityManager() == null)
            {
                filename = postSpoolMail(mailobj, spoolDir);
            }
            else
            {
                // running with sandbox security, we grant access permission to spool mail
                final MailImpl mObj = mailobj;
                filename = (String) AccessController.doPrivileged(new PrivilegedExceptionAction()
                {
                    public Object run() throws Exception
                    {
                        return postSpoolMail(mObj, spoolDir);
                    }
                });
            }

            // Add it to the filename->message cache
            mailStorage.put(filename, mailobj);

            // Why don't we add it to the current work list?
            // This will cause problems as most workers will exit, but
            // the last few will see this work and continue, reducing throughput
            //spoolFiles.add(filename);
        }
        catch (Exception e)
        {
            logger.error(e);
            if (filename != null)
            {
                postUndeliverMail(filename);
            }
            throw new MailSessionException(e);
        }

    }

    /**
     * Get the next spool file and send the message.
     * @return TRUE if we didn't find any more mail to send
     */
    private boolean sendMail()
    {
        String[] filename = new String[1];

        // Get the impl and the filename
        MailImpl impl = retrieveMail(filename);

        if (impl == null)
            return true;

        // deliver the message
        try
        {
            deliver(impl, filename[0]);
        }
        catch (MailDeliveryException e)
        {
            // This exception is already logged, so just ignore it
            // as we are in spool mode.
        }
        return false;
    }

    /**
     * Delivers mail immediately. This was introduced to support requirements for monitoring
     * where email needs to be sent without any delay.
     *
     * @throws MailDeliveryException
     */
    public void deliver(Object o) throws MailDeliveryException
    {
    	MailImpl impl = (MailImpl) o;
        deliver(impl, null);
    }

    /**
     * Deliver the message described by impl
     *
     * @param impl the message to deliver
     * @param filename the name of the spool file, null if none
     * @throws MailDeliveryException if an error delivering the message
     */
    private void deliver(MailImpl impl, String filename) throws MailDeliveryException
    {
        final long ONE_MINUTE = 1000 * 60;
        String deliveryHost;

        // Just in case
        if (impl == null)
            return;

        // Get a connection and send the message
        Transport connection = null;
        HostImpl server = null;
        boolean frompool = false;
        try
        {
            Session session = null;
            Exception lastException = null;

            // Get a connection, if the first doesn't work, try the next, etc.
            HostImpl[] serverList = impl.getServers();
            for (int i = 0; i < serverList.length; i++)
            {
                server = serverList[i];
                String serverKey = server.toString();
                // Check if this server was 'bad',
                // if this is the last server left, go ahead and try it anyway.
                if ( (i < serverList.length - 1) && badServers.containsKey(serverKey))
                {
                    long now = System.currentTimeMillis();
                    long lasttime = ((Date) badServers.get(serverKey)).getTime();
                    if (now < lasttime + ONE_MINUTE)
                        continue;   // try the next one
                }

                // Get a connection from the pool
                if (bMaintainConnections)
                {
                    connection = connectionPool.checkout(serverKey);
                    if (connection != null)
                    {
                        frompool = true;
                    }
                }

                // If we didn't get a connection from the pool, create one
                try
                    {
                        //83980  - whenever the connection was taken from pool, session value was null and in InternetAddress.java
                        //if session is null it uses InetAddress.getLocalHost().getHostName() even if mail.host is set in System Property
                        //now getting session in all cases even if connection is not null.
                        session = server.getSession();
                        if (connection == null)
                       {
                         session.setDebug(impl.isDebug());
                         connection = getConnection(server, session);
                         //lastException = null;  (already null)
                         // This server is OK now (if it ever was marked bad).
                         badServers.remove(server.toString());
                        }
                    }
                    catch (Exception e)
                    {
                        // Remember this server to avoid trying it again right away
                        badServers.put(server.toString(), new Date());
                        lastException = e;
                        continue;   // try the next server, if any
                    }


                // Get the message
                Message msg = impl.createMessage(session);

                // Send the message
                try
                {
                    sendMessage(connection, msg, msg.getAllRecipients());
                    if (bMaintainConnections)
                        connectionPool.checkin(serverKey, connection);
                    lastException = null;
                    break;  // done, break out of server loop
                }
                catch (SendFailedException sendex)
                {
                     /*
                     For Bug 72580 it was throwing exceptions because the settings on the exchange server for session size and number of messages sent per connection
                     where exceeding since we wont know the settings on the server whenever this exception occurs we are closing the current connection and creating a new one
                     and sending the mails which were undelivered.

                     */
                          if(sendex.getClass().getName().equals("com.sun.mail.smtp.SMTPSendFailedException") &&  ((SMTPSendFailedException)sendex).getReturnCode() == 421)
                          {

                               try
                              {
                                connection.close();
                              }
                              catch(MessagingException mex) { }

                                 try
                              {
                                session = server.getSession();
                                session.setDebug(impl.isDebug());
                                connection = getConnection(server, session);
                                // This server is OK now (if it ever was marked bad).
                                badServers.remove(server.toString());
                            }
                            catch (Exception e)
                            {
                                // Remember this server to avoid trying it again right away
                                badServers.put(server.toString(), new Date());
                                lastException = e;
                                continue;   // try the next server, if any
                            }

                            try
                            {
                                javax.mail.Address valid[] = sendex.getValidUnsentAddresses();
                                if(valid != null)
                                {
                                    sendMessage(connection, msg, valid);
                                }
                                if(bMaintainConnections)
                                connectionPool.checkin(serverKey, connection);
                                lastException = null;
                                break;
                            }
                            catch(SendFailedException sendexcep)
                            {

                                if(bMaintainConnections)
                                connectionPool.checkin(serverKey, connection);
                                throw sendexcep;
                            }
                            catch(MessagingException mex)
                            {
                                throw mex;
                            }
                        }
                        else
                        {

					      	// Addressing error, server is still OK
                    		if (bMaintainConnections)
                        	connectionPool.checkin(serverKey, connection);
                    		throw sendex;   // rethrow and break out of server loop
					     }
                }
                // If there was a connection problem, try again if we got this from the pool
                catch (MessagingException e)
                {
                    if (!frompool)
                        throw e;    // rethrow and break out of server loop

                    // if we got the connection from the pool
                    // it may not be any good and we should try again
                    session = server.getSession();
                    session.setDebug(impl.isDebug());
                    try
                    {   // Get a new connection
                        connection = getConnection(server, session);
                    }
                    catch (MessagingException mex)
                    {
                        badServers.put(server.toString(), new Date());
                        lastException = mex;
                        continue;   // try the next server, if any
                    }
                    try
                    {   // Send the message
                        sendMessage(connection, msg, msg.getAllRecipients());
                        if (bMaintainConnections)
                            connectionPool.checkin(serverKey, connection);
                        lastException = null;
                        break;  // done, break out of server loop
                    }
                    catch (SendFailedException sendex)
                    {   // Addressing error, server is still OK
                        if (bMaintainConnections)
                            connectionPool.checkin(serverKey, connection);
                        throw sendex;   // rethrow and break out of server loop
                    }
                    catch (MessagingException mex)
                    {
                        throw mex;  // rethrow and break out of server loop
                    }
                }
            }   // server loop

            // Something went wrong, throw the last error
            if (lastException != null)
                throw lastException;

            // If we get here, we delivered it OK.  Record the host.
            deliveryHost = server.getHost();
        }
        catch (Exception ex)
        {

            logger.error(ex.toString(),ex);
            if (filename != null) {
                postUndeliverMail(filename);
            }
            else {
                postUndeliverMessage(impl);
            }

            // rethrow
            throw new MailDeliveryException(ex);
        }
        finally
        {
            if (!bMaintainConnections && connection != null)
            {
                try { connection.close(); } catch (MessagingException mex) { /* ignore */ }
            }
        }

        // Delete any temporary attachments if user has set removeAttachments= true
        deleteAttachments(impl);

        // Remove the spool file if we are NOT doing immediate delivery
        if (filename != null)
            removeSpoolFile(filename);

        // Log it
        if (isMailSentLoggingEnable())
        {
            InternetAddress[] to = impl.getRecipient();
            String to_string = "";
            for (int i = 0; i < to.length; i++)
            {
                to_string += to[i].getAddress();
                if ((i + 1) < to.length)
                    to_string += ", ";
            }
            sentlogger.info("Mail: '" +
                            impl.getSubject() + "' From:'" + impl.getSender() + "' To:'" + to_string
                            + "' was successfully sent using " + deliveryHost);
        }
    }
    
    private void sendMessage(final Transport mailTransport, final Message msg, Address[] receipients) throws MessagingException {
    	if(IS_JVM_SECURITY_ENABLED) {
            try
            {
                AccessController.doPrivileged(new PrivilegedExceptionAction<Object>()
                {
                    @Override
                    public Object run() throws Exception
                    {Thread.currentThread().setContextClassLoader( getClass().getClassLoader() );
                        mailTransport.sendMessage(msg, receipients);            
                        return null;
                    }
                }, null, tempFolderPermission);
            }
            catch (PrivilegedActionException e)
            {
                if(e.getException() instanceof MessagingException)
                    throw (MessagingException)e.getException();
                throw new RuntimeException(e.getException());
            }
        } else
            mailTransport.sendMessage(msg, receipients);            
    }


    private void deleteAttachments(MailImpl impl)
    {
        try
        {
            //if attachments is not available, return
            if (impl.getAttachements() == null)
                return;
            Iterator itr = impl.getAttachements().keySet().iterator();
            while (itr.hasNext())
            {
                final String tempStr = (String) itr.next();
                try
                {
                    if (impl.isRemoveAttachment(tempStr))
                    {
                        /**
                         * security model, adding this to grant access for the sandbox
                         */
                        if (System.getSecurityManager() == null || !impl.isVarAttachment(tempStr))
                        {
                            deleteAttachmentsImpl(tempStr);
                        }
                        else
                        {
                            // running with sandbox security, we grant access permission to delete Attachments
                            AccessController.doPrivileged(new PrivilegedExceptionAction()
                            {
                                public Object run() throws Exception
                                {
                                    deleteAttachmentsImpl(tempStr);
                                    return null;
                                }
                            });
                        }
                    }
                }
                catch (Exception e)
                {
                    logger.error(e);
                }
            }
        } catch (MessagingException ex)
        {
        }
    }

    private void deleteAttachmentsImpl(String tempStr)
    {
        File tempAttach = VFSFileFactory.getFileObject(tempStr);
        // check if file exists and is to be removed.
        if (tempAttach.exists())
        {
            tempAttach.delete();
        }
    }

    /**
     * Return a connected Transport.
     *
     * @param svr info on the server to connect to.
     * @return a transport that has had connect() called on it.
     * @throws MessagingException
     */
    private Transport getConnection(HostImpl svr, Session session) throws MessagingException
    {
        // Get the JavaMail transport and do the connect
        Transport t = getTransport(session, svr.getProtocol());
        try
        {
            t.connect(svr.getHost(), svr.getPort(), svr.getUsername(), svr.getPassword());
        }
        catch (AuthenticationFailedException e)
        {
            // JavaMail sometimes throws this exception without a message
            String msg = e.getMessage();
            if (msg == null || msg.length() == 0)
                throw new AuthenticationFailedException("Authentication Failed");
            else
                throw e;    // rethrow
        }

        // return it
        return t;
    }
    
    private Transport getTransport(final Session session, final String protocol) throws MessagingException {
    	if(System.getSecurityManager() == null)
    		return session.getTransport(protocol);
    	else {
    		try {
				return AccessController.doPrivileged(new PrivilegedExceptionAction<Transport>() {

					@Override
					public Transport run() throws NoSuchProviderException {
						return session.getTransport(protocol);
					}
					
				}, null, ((RuntimeServiceImpl)ServiceFactory.getRuntimeService()).getWebInfClassesReadPermission());
			} catch (PrivilegedActionException e) {
				Exception exception = e.getException();
				if(exception instanceof MessagingException)
					throw (MessagingException) exception;
				throw new MessagingException(null, exception);
			}
    	}
    }


    /**
     * Routine called by the scheduler every N seconds to deliver pending mail
     */
    public void run()
    {
        if (mailSpoolerStarted) {
            try
            {
                // Update the list of work to do
                refreshSpoolFiles();

                if (bFastMail)
                {
                    deliverFast();

                    // close mail server connections
                    if (bMaintainConnections)
                    {
                        logger.debug("Closing mail server connections.");
                        connectionPool.cleanup();
                    }
                }
                else    // ColdFusion Standard doesn't get "fast" mail
                {
                    deliverStandard();
                }
            }
            catch (Exception e)
            {
                // Don't let anything stop us from rescheduling ourselves
                logger.error(e);
            }
        }
        // schedule for next time look up: now + interval
        logger.debug("Next mail spool run in " + getSchedule() / 1000 + " seconds.");
        SchedulerService ss = ServiceFactory.getSchedulerService();
        ss.schedule(this, (System.currentTimeMillis() + getSchedule()));
    }

    /**
     * Deliver mail in reasonable size batches.
     */
    private void deliverStandard()
    {
        // Preserve ColdFusion MX batch sizes
        int batch_size = 35;
        int size = spoolFiles.size();
        if (size > 100)
        {
            batch_size = size / (int)Math.log ((double)size);
            if (batch_size > 1000) batch_size = 1000;
            if (batch_size < 35) batch_size = 35;
        }
        try
        {
            boolean done = false;
            int count = 0;
            while (!done && count < batch_size)
            {
                done = sendMail();
                count++;
            }
        }
        catch (Exception e)
        {
            // Something unexpected happened, log it and continue.
            logger.error(e);
        }
    }

    /**
     * Deliver all the mail that is currently queued.
     * Spawn multiple threads to perform parallel delivery.
     */
    private void deliverFast()
    {
        try
        {
            // Do we have any work?
            int messages = spoolFiles.size();
            synchronized (spoolMessages)
            {
                messages += spoolMessages.size();
            }
            if (messages == 0)
            {
                return; // done
            }

            // Don't spawn threads we don't need,
            // Try one thread per 10 messages, up to the max (maxDeliveryThreads)
            int thread_count = messages / 10;
            if (thread_count > maxDeliveryThreads) thread_count = maxDeliveryThreads;
            if (thread_count < 1) thread_count = 1;

            // Skip thread creation if only a single thread
            if (thread_count == 1)
            {
                logger.debug("Running delivery in this thread (" + spoolFiles.size() + " entries).");
                boolean done = false;
                // Work the queue till it's completely gone
                while (!done)
                {
                    done = sendMail();
                }
            }
            else    // thread_count > 1
            {
                logger.debug("Starting up " + thread_count + " worker threads to deliver "+ messages + " messages.");

                // Worker status object
                final WorkerStatus status = new WorkerStatus();

                // This is the code that each worker will execute
                Runnable work = new Runnable()
                {
                    public void run()
                    {
                        status.workerBegin();
                        // Work the queue till it's completely gone
                        boolean done = false;
                        while (!done)
                        {
                            try
                            {
                                done = sendMail();
                            }
                            catch (Exception e)
                            {
                                // Something unexpected happened, log it and continue.
                                logger.error(e);
                            }
                        }
                        status.workerEnd();
                    }
                };

                // Spawn worker threads to deliver mail
                for (int i=0; i < thread_count; i++)
                {
                    Thread t = new Thread(work, "mailWorker-" + i);
                    t.start();
                }

                // Make sure workers start
                status.waitBegin();
                logger.debug("Worker threads started (" + thread_count + "), waiting...");

                // Wait for all workers to complete
                status.waitDone();
                logger.debug("Worker threads finished (all "+ thread_count + " of them)");

            }
        }
        catch (Exception e)
        {
            // Something unexpected happened, log it and continue.
            logger.error(e);
        }
    }

    /**
     * Move a spool file to the underlivered directory.
     * Logs a message if MailSentLogging is true.
     *
     * @param filename the spool file to move
     */
    private void postUndeliverMail(String filename)
    {
        // Create temp file.
        try
        {
            File spoolFile = new File(emailSpoolDir + File.separator + filename);
            File undeliverMail = new File(emailUndeliverDir + File.separator + filename);
            // prevent a problem when the undelivered file exists
            if (undeliverMail.exists())
            {
                undeliverMail.delete();
            }
            spoolFile.renameTo(undeliverMail);
            if (isMailSentLoggingEnable())
            {
                sentlogger.info("Moved undelivered mail: " + filename + " to " +
                                emailUndeliverDir + " directory");
            }

        }
        catch (Exception ex)
        {
            logger.error("Unable to post to " +
                         emailUndeliverDir + " directory: " + ex.toString());
            throw new MailSessionException(ex);
        }
    }
    
    public boolean isMessagingException(Throwable th){
    	return th instanceof MessagingException;
    }
    
    public Throwable getNextMessagingException(Throwable th){
    	return ((MessagingException)th).getNextException();
    }

    /**
     * Write a message to the undelivered folder that was never spooled to disk.
     *
     * @param message the mail message to write
     */
    void postUndeliverMessage(MailImpl message)
    {
        try
        {
            // Write a new file in the underlivered directory
            String filename = postSpoolMail(message, undeliverDir);

            // log it
            if (isMailSentLoggingEnable())
            {
                sentlogger.info("Wrote undelivered mail: " + filename + " to " +
                                emailUndeliverDir + " directory");
            }
        }
        catch (Exception ex)
        {
            logger.error("Unable to post to " +
                         emailUndeliverDir + " directory: " + ex.toString());
        }
    }


    /**
     * Put this message on the work queue that is NOT backed by a spool
     * file on disk.  This trades speed for reliability.
     *
     * @param message The message to post
     */
    private void postMessage(MailImpl message)
    {
        synchronized (spoolMessages)
        {
            spoolMessages.addLast(message);
        }
    }


    /**
     * Create a spool file from the mail message provided.
     *
     * @param impl the message to spool
     * @param directory the directory to create the spool file
     * @return the filename of the spool file
     * @throws MailSessionException if there is a problem spooling the message
     */
    private String postSpoolMail(MailImpl impl, File directory) throws MailSessionException
    {
        boolean lockSpool = spoolDir.equals(directory);
        boolean lockAcquired = false;
        // Create temp file.
        try
        {
            String filename;
            // lock the spool directory
            if (lockSpool)
            {
                try
                {
                    // Don't create a file if spooler is reading the directory list.
                    // This prevents him from seeing a partially written spool file.
                    // Multiple threads can write spool files at the same time however ("Read" Lock).
                    spoolLock.requestReadLock(SPOOL_LOCK_TIMEOUT * 1000);
                    lockAcquired = true;
                }
                catch (InterruptedException e)
                {
                    throw new MailSessionException(new SpoolLockTimeoutException());
                }
            }
            File temp = File.createTempFile("Mail", ".cfmail", directory);
            filename = temp.getName();

            // Write to temp file
            BufferedWriter out = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(temp), "UTF-8"));

            // Bug#65050 EET#1497-- First entry should always be content-type in spooled file for correct message/from/to encoding
            // type field in spool file appears after subject. to and from entries come before type. So while encoding
            // from and to, encoding type comes as null which is causing the problem. Moved type entry to be the
            // first in the spool file.

            // mimetype field
            out.write("type" + separator + impl.getContentType());
            out.newLine();

            // server info per server - one line for host & port, then username and password if set
            // server: host:port
            // [server-username: un]
            // [server-password: pw]
            HostImpl[] servers = impl.getServers();
            if (servers.length > 0)
            {
                for (int j = 0; j < servers.length; j++)
                {
                    out.write("server" + separator);
                    HostImpl host = servers[j];
                    out.write(host.getHostString());
                    out.newLine();
                    // Username and password (password gets Base64 encoded)
                    final String username = host.getUsername();
                    final String password = host.getPassword();
                    if (username != null && password != null)
                    {
                        out.write("server-username" + separator + username);
                        out.newLine();
                        // encrypt this?
                        out.write("server-password" + separator + Base64Encoder.encode(password.getBytes()));
                        out.newLine();
                    }
                }
            }

            // Use TLS, if true
            Boolean useTLS = impl.getUseTLS();
            if (useTLS != null && useTLS.booleanValue())
            {
                out.write("usetls" + separator + "true");
                out.newLine();
            }

            // Use SSL, if true
            Boolean useSSL = impl.getUseSSL();
            if (useSSL != null && useSSL.booleanValue())
            {
                out.write("usessl" + separator + "true");
                out.newLine();
            }

            // Port, if set in tag
            int port = impl.getPort();
            if (port > 0)
            {
                out.write("port" + separator + port);
                out.newLine();
            }

            // Timeout, if set in tag
            int timeout = impl.getTimeout();
            if (timeout > 0)
            {
                out.write("timeout" + separator + timeout);
                out.newLine();
            }

            // from field
            InternetAddress sender = impl.getSender();
            if (sender != null)
            {
                out.write("from" + separator + sender.toString());
                out.newLine();
            }

            // to field(s)
            InternetAddress[] to = impl.getRecipient();
            writeAddress("to", to, out);

            // cc field(s)
            InternetAddress[] cc = impl.getCc();
            if (cc != null)
            {
                writeAddress("cc", cc, out);
            }

            // bcc field(s)
            InternetAddress[] bcc = impl.getBcc();
            if (bcc != null)
            {
                writeAddress("bcc", bcc, out);
            }

            // ReplyTo field
            InternetAddress[] replyto = impl.getReplyTo();
            if (replyto != null)
            {
                writeAddress("replyto", replyto, out);
            }

            // FailTo field
            if (impl.getFailTo() != null)
            {
                out.write("failto" + separator + impl.getFailTo());
                out.newLine();
            }

            // subject field
            // Bug 61098 - Prevent malicious subject from tricking our spool file
            String subject = impl.getSubject().replace('\n', ' ').replace('\r', ' ');
            out.write("subject" + separator + subject);
            out.newLine();

            // Mail headers (including mailerid)
            try
            {
                Header[] headers = impl.getHeaders();
                for (int j = 0; j < headers.length; j++)
                {
                    Header h = headers[j];
                    out.write(h.getName() + separator + h.getValue());
                    out.newLine();
                }
            }
            catch (Exception e)
            {
                logger.error(e);
                throw new MailSessionException(e);
            }

            // body text - Have to handle multiple body parts
            try
            {
                if (!impl.isMultipart())
                {
                    // Simple case, write body with no bodytype
                    writeBody(out, impl.getBody());
                }
                else
                {
                    String[] parts = impl.getBodyParts();
                    String[] types = impl.getBodyTypes();

                    if (parts != null)
                    {
                        for (int i = 0; i < parts.length; i++)
                        {
                            if (parts[i] != null)
                            {
                                // Write out body of message part
                                out.write("bodypart-start" + separator + types[i]);
                                out.newLine();
                                writeBody(out, parts[i]);
                                out.write("bodypart-end" + separator + types[i]);
                                out.newLine();
                            }
                        }
                    }
                }
            }
            catch (MessagingException e)
            {
                logger.error(e);
                throw new MailSessionException(e);
            }

            // attachements - special format:
            //   file: <filename>
            //   file-type: <Content-Type>
            //   file-disposition: inline|attachment
            //   file-id: <Content-ID>
            try
            {
                // Map returned is filename keys, MimeBodyParts
                Map attachements = impl.getAttachements();
                if (attachements != null)
                {
                    Iterator it = attachements.keySet().iterator();
                    while (it.hasNext())
                    {
                        String file = (String) it.next();
                        MimeBodyPart part = (MimeBodyPart) attachements.get(file);
                        // Get the type that may have been explicitly set on the body part
                        // Note: getContentType() will always return text/plain if not set
                        String type = part.getHeader("Content-Type", null);
                        String dispo = part.getDisposition();
                        String overrideFileName = part.getFileName();
                        String id = part.getHeader("Content-ID", null);

                        out.write("file" + separator + file);
                        out.newLine();
                        if(overrideFileName != null){
                            out.write("filename" + separator + overrideFileName);
                            out.newLine();
                        }
                        if (type != null)
                        {
                        	type = type.replaceAll("\r\n","");
                        	out.write("file-type" + separator + type);
                            out.newLine();
                        }
                        if (dispo != null)
                        {
                            out.write("file-disposition" + separator + dispo);
                            out.newLine();
                        }
                        if (id != null)
                        {
                            out.write("file-id" + separator + id);
                            out.newLine();
                        }
                        // write if the attachment was to be removed after being delivered
                        out.write("remove"+separator+ impl.isRemoveAttachment(file));
                        out.newLine();
                    }
                }
            }
            catch (MessagingException e)
            {
                logger.error(e);
                throw new MailSessionException(e);
            }

            out.close();
            return filename;

        }
        catch (IOException ex)
        {
            logger.error(ex);
            throw new MailSessionException(ex);
        }
        finally
        {
            if (lockSpool && lockAcquired)
                spoolLock.releaseReadLock();
        }
    }

    /**
     * Loop over the array of addresses and write them out comma separated.
     *
     * @param field the name of the field to write
     * @param to the array of addresses
     * @param out where to write
     * @throws IOException
     */
    private void writeAddress(String field, InternetAddress[] to, BufferedWriter out) throws IOException
    {
        out.write(field + separator);
        int i;
        for (i = 0; i < to.length; i++)
        {
            out.write(to[i].toString());
            if (i < (to.length - 1))
            {
                out.write(",");
            }
        }
        out.newLine();
    }

    /**
     * Write a single body part out
     *
     * @param out where to write the data
     * @param bodytext text to write to the file
     * @throws MailSessionException if there are problems writing to the spool file
     */
    private void writeBody(BufferedWriter out, String bodytext)
            throws MailSessionException
    {
        if (bodytext != null)
        {
            StringReader sr = new StringReader(bodytext);
            BufferedReader br = new BufferedReader(sr);
            try
            {
                String line = br.readLine();
                while (line != null)
                {
                    // Write body lines
                    out.write("body" + separator + line);
                    out.newLine();
                    line = br.readLine();
                }
            }
            catch (IOException e)
            {
                logger.error(e);
                throw new MailSessionException(e);
            }
        }
    }

    /**
     * Get the next message to deliver.
     * First checks the message list, then spool files from the mailStorage map.
     *
     * @param filename Ouptut parameter - the spool file name we just retrieved
     * @return the mail message to deliver, or null if nothing to do
     */
    private MailImpl retrieveMail(String[] filename)
    {
        MailImpl retMail = null;

        // Check for spooled messages
        synchronized(spoolMessages)
        {
            if (spoolMessages.size() > 0)
            {
                retMail = (MailImpl) spoolMessages.removeFirst();
                return retMail;
            }
        }

        // Try and retrive one of the spool files
        // loop until we get one or run out
        while (spoolFiles.size() > 0)
        {
            MailImpl impl;

            // Grab the first filename from the list
            // and remove it from our cached list of spool filenames.
            String key = (String) spoolFiles.remove(0);

            // Even though mailStorage is a synchronized class,
            // we lock to make the get AND remove an atomic operation.
            synchronized (mailStorage)
            {
                if (bypassSoftCache)
                {
                    // For testing ONLY - bypass the SoftCache, get message from disk
                    impl = (MailImpl) mailStorage.fetch(key);
                }
                else
                {
                    // mailStorage is a SoftCache that will retrieve
                    // mail from the disk file if it isn't in the map
                    impl = (MailImpl) mailStorage.get(key);
                }
                mailStorage.remove(key);
            }

            if (impl != null)
            {
                // Set return values
                retMail = impl;
                filename[0] = key;

                // All done looping
                break;
            }
        }

        // return Message found, if any
        return retMail;
    }

    /**
     * Utility method to examine the spool directory and
     * update the SpoolFiles global with the current work list.
     *
     * This routine is not thread-safe and should not be called by anyone other
     * than the spooler thread (i.e. no workers should call this).
     * <p>
     * WARNING: this will <b>clear</b> the work list before updating it, so don't
     * depend on this to add new work to an existing list. i.e. you might want
     * to check if the work list is empty before calling this.
     *
     * @throws MailDeliveryException if we timeout waiting to read the spool directory
     */
    private void refreshSpoolFiles() throws MailDeliveryException
    {
        // refresh list
        String[] fileList = null;
        try
        {
            // Dont read the spool if someone is writing a file
            // This prevents us from seeing a partially written spool file
            spoolLock.requestWriteLock(SPOOL_LOCK_TIMEOUT * 1000); // 60 seconds
            fileList = spoolDir.list();
            spoolLock.releaseWriteLock();
        }
        catch (InterruptedException e)
        {
            throw new MailDeliveryException(new SpoolLockTimeoutException());
        }

        // Make sure the clear and add are atomic as there could
        // be a request thread trying to spool mail at the same time.
        synchronized (spoolFiles)
        {
            // Clear any existing items from the list
            spoolFiles.clear();

            // Copy the filenames to our work list
            if (fileList != null)
                spoolFiles.addAll(Arrays.asList(fileList));
        }
    }

    /**
     * Remove file from the spool directory.
     * @param filename the file to remove
     * @return success if the file is removed.
     */
    private boolean removeSpoolFile(String filename)
    {
        // remove file with key as the name;
        File spoolFile = new File(emailSpoolDir + File.separator + filename);
        return spoolFile.delete();
    }

    /**
     * Read a spool file from disk and create a mail message (MailImpl).
     *
     * @param spoolFile the name of the file to process
     * @return a MailImpl object created from the spool file.
     *
     * @throws InvalidSpoolFileException if the file is invalid for any reason
     * @throws IOException if there are problems reading a spool file
     * @throws MessagingException if there is a problem creating the mail object
     */
    private MailImpl retrieveSpoolMail(String spoolFile)
            throws InvalidSpoolFileException, IOException, MessagingException
    {
        MailImpl mailimpl = new MailImpl();
        String filepath = spoolDir.getPath() + File.separator + spoolFile;
        BufferedReader in = new BufferedReader(new InputStreamReader(new FileInputStream(filepath), "UTF-8"));

        try
        {
            processFile(in, mailimpl);
        }
        finally
        {
            // Close the spool file
            in.close();
        }

        // return the Mail message
        return mailimpl;
    }

    /**
     * Process a spool file and create a MailImpl object from it.
     *
     * @param in a reader for the spool file
     * @param mailimpl the mail message to populate
     *
     * @throws InvalidSpoolFileException if the file is badly formatted.
     * @throws IOException if there is a problem reading the file
     * @throws MessagingException if there is a problem creating the MailImpl
     */
    private void processFile(BufferedReader in, MailImpl mailimpl)
            throws InvalidSpoolFileException, IOException, MessagingException
    {
        StringBuilder bf = new StringBuilder();
        String bodyType = null;
        List hostList = new ArrayList();
        HostImpl currentServer = null;

        String line = in.readLine();
        if (line == null || line.length() == 0)
        {
            throw new InvalidSpoolFileException();
        }

        while (line != null)
        {
            // Ignore empty lines
            if (line.length() == 0)
            {
                line = in.readLine();
                continue;
            }

            int indx = line.indexOf(separator);
            if (indx == -1) // bad file
                throw new InvalidSpoolFileException();

            String name = line.substring(0, indx);
            String value = line.substring(indx + separator.length());

            if (name.length() != 0 && value != null)
            {
                if (name.equals("body"))
                {
                    bf.append(value).append("\r\n");
                }
                else if (name.equals("bodypart-start"))
                {
                    // Start a new part
                    bodyType = value;
                }
                else if (name.equals("bodypart-end"))
                {
                    // Add the part
                    mailimpl.addPart(bf.toString(), bodyType, null, -1);
                    // clear the body buffer and bodyType
                    bf.delete(0, bf.length());
                    bodyType = null;
                }
                else if (name.equals("server"))
                {
                    currentServer = new HostImpl();
                    int colon = value.indexOf(':');
                    if (colon != -1)
                    {
                        int port = Integer.parseInt(value.substring(colon + 1));
                        String host = value.substring(0, colon);
                        currentServer.setHost(host);
                        currentServer.setPort(port);
                    }
                    else
                    {
                        currentServer.setHost(value);
                    }
                    if ( !bFastMail && hostList.size() > 0)
                    {
                        // Only allowed a single server, ignore this server
                        logger.warn(RB.getString(MailTag.class, "MailTag.noFailover"));
                    }
                    else
                    {
                        hostList.add(currentServer);
                    }
                }
                else if (name.equals("server-username"))
                {
                    if (currentServer != null)
                        currentServer.setUsername(value);
                }
                else if (name.equals("server-password"))
                {
                    if (currentServer != null)
                        currentServer.setPassword(new String(Base64Encoder.decode(value)));
                }
                else if (name.equals("port"))
                {
                    mailimpl.setPort((new Integer(value)).intValue());
                }
                else if (name.equals("timeout"))
                {
                    mailimpl.setTimeout((new Integer(value)).intValue());
                }
                else if (name.equals("from"))
                {
                    InternetAddress sender = new InternetAddress(value);
                    mailimpl.setSender(sender);
                }
                else if (name.equals("to"))
                {
                    mailimpl.setRecipient(MailImpl.setInternetAddress(value));
                }
                else if (name.equals("cc"))
                {
                    mailimpl.setCc(MailImpl.setInternetAddress(value));
                }
                else if (name.equals("bcc"))
                {
                    mailimpl.setBcc(MailImpl.setInternetAddress(value));
                }
                else if (name.equals("replyto"))
                {
                    mailimpl.setReplyTo(MailImpl.setInternetAddress(value));
                }
                else if (name.equals("failto"))
                {
                    mailimpl.setFailTo(value);
                }
                else if (name.equals("subject"))
                {
                    mailimpl.setSubject(value);
                }
                else if (name.equals("type"))
                {
                    mailimpl.setContentType(value, null);
                }
                else if (name.equals("mimeattach")) // todo: not sure if we add this in spooled file else need to get remove value too
                {
                    mailimpl.setAttachment(VFSFileFactory.getFileObject(value), Boolean.FALSE);
                }
                else if (name.equals("mailerid"))
                {
                    mailimpl.setHeader("X-Mailer", value);
                }
                else if (name.equals("file"))
                {
                    String type = null;
                    String dispo = null;
                    String id = null;
                    Boolean remove= null;
                    String fileName = null;
                    boolean readnext = false;
                    // attachements format:
                    //   file: <filename>
                    //   file-type: <Content-Type>
                    //   file-disposition: inline|attachment
                    //   file-id: <Content-ID>

                    // Support CFMX 6.1 format too:
                    //   file: <filename>[;<type>]
                    int semi = value.indexOf(';');
                    String filename;
                    if (semi != -1)
                    {
                        filename = value.substring(0, semi);
                        type = value.substring(semi + 1);
                    }
                    else
                    {
                        filename = value;
                    }

                    if (filename == null || filename.length() == 0)
                    {
                        throw new InvalidSpoolFileException();
                    }

                    if (type == null)   // not old style type
                    {
                        // check for other file attributes
                        line = in.readLine();
                        readnext = true;
                        while (line != null)
                        {
                            indx = line.indexOf(separator);
                            // Ignore empty lines
                            if (line.length() == 0)
                            {
                                line = in.readLine();
                                continue;
                            }
                            if (indx == -1) // bad file
                                throw new InvalidSpoolFileException();

                            name = line.substring(0, indx);
                            value = line.substring(indx + separator.length());

                            if (name.length() != 0 && value != null)
                            {
                                if (name.equals("file-type"))
                                    type = value;
                                else if (name.equals("file-disposition"))
                                    dispo = value;
                                else if (name.equals("file-id"))
                                    id = value;
                                else if (name.equals("remove"))
                                    remove=Boolean.valueOf(value);
                                else if (name.equals("filename"))
                                    fileName = value;
                                else
                                    break;
                            }
                            // next line
                            line = in.readLine();
                        }
                    }

                    // check if the filename is a URL
                    URL url_obj = resolveUrl(filename);
                    if (url_obj != null)
                    {
                        mailimpl.setAttachment(url_obj, type, dispo, id, fileName);
                    }
                    else
                    {
                        File file_obj = VFSFileFactory.getFileObject(filename);
                        mailimpl.setAttachment(file_obj, type, dispo, id, remove, fileName);
                    }
                    // if we already read the next line, continue the outer loop
                    if (readnext)
                        continue;
                }
                else if (name.equals("file-type") || name.equals("file-disposition") || name.equals("file-id"))
                {
                    // we should never see these items
                    throw new InvalidSpoolFileException();
                }
                else if (name.equals("usetls") && "true".equals(value))
                {
                    mailimpl.setUseTLS(true);
                }
                else if (name.equals("usessl") && "true".equals(value))
                {
                    mailimpl.setUseSSL(true);
                }
                else    // everything else is treated as a mail header
                {
                    mailimpl.setHeader(name, value);
                }
            } //end if

            // read the next line
            line = in.readLine();
        } // end while

        // Fill in hosts in the message
        HostImpl[] hArray = new HostImpl[hostList.size()];
        hostList.toArray(hArray);
        mailimpl.setServers(hArray);

        // Fill in server info with tag defaults
        mailimpl.validateServers();

        // Fill in server info with global defaults
        validate(mailimpl);

        // If we have body content, we are doing a single part message
        if (bf.length() > 0)
        {
            mailimpl.addPart(bf.toString(), null, null, -1);
        }
    }

    public void validateEmail(String email){
    	 try{
             email = IDN.toASCII(email);
         } catch(Exception e){}
         try{                
             new InternetAddress(email).validate();
         }catch (AddressException e){
             throw new InvalidEmailTypeException();
         }catch (NullPointerException e){
             throw new InvalidEmailTypeException();
         }
    }
    
    public void containsOutputTag(Tag parent){
    	if (parent instanceof MailTag) {
   		 ((MailTag)parent).ContainsOutputTag(true);
   		}
    }
    
    public Date getMailDateFormat(String date){
    	try {
			return new javax.mail.internet.MailDateFormat().parse(date);
		} catch (java.text.ParseException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
    	return null;
    }
    
	public String[] getContentType(String ct) {
		ContentType contentType = null;
		;
		try {
			contentType = new ContentType(ct);
		} catch (ParseException e) {
			return new String[] { null, null };
		}
		String contentString = contentType.getBaseType();
		String cs = contentType.getParameter("charset");
		return new String[] { contentString, cs };
	}
    public DataSource getByteArrayDataSource(byte[] obj, String o){
    	return new ByteArrayDataSource( ( byte [] ) obj, o );
    }
    /**
     * Resolve a string to a URL, will lookup JNDI names too.
     * @param url string to convert
     * @return a URL
     */
    protected URL resolveUrl(String url)
    {
        // FIXME: need to find out if the logic here applies to the mail spooler.

        URL u;
        if (url.startsWith("http://") || url.startsWith("https://") || url.startsWith("ftp://"))
        {
            try
            {
                u = new URL(url);
            }
            catch (Exception ex)
            {
                return null;
            }
        }
        else if (url.startsWith("java:comp/env/url"))
        {
            try
            {
                InitialContext ctx = new InitialContext();
                u = (URL) ctx.lookup(url);
            }
            catch (Exception ex)
            {
                return null;
            }
        }
        else
        {
            try
            {
                u = new URL(url);
            }
            catch (Exception ex)
            {
                return null;
            }
        }
        return u;
    }

    /**
     * Writes a message to the mail spoolers log file.
     * Non-errors are logged only if MailSentLogging is true.
     *
     * @param message the message to write
     * @param bError if true, uses the mail log otherwise uses the mailsent log.
     */
    public void writeToLog(String message, boolean bError)
    {
        if (bError)
            logger.error(message);
        else
        {
            if (isMailSentLoggingEnable())
                sentlogger.info(message);
        }
    }


    

    /**
     * return the map of configuration settings
     */
    public Map getResourceBundle()
    {
        if (rb == null)
        {
            rb = new HashMap();
            rb.put("configuration.keys",
                    "severity,logpath,spooldir,undeliverdir,path,server,port,timeout,schedule,mailsentloggingenable,spoolenable,maintainconnections,maxthreads,spooltomemory,spoolmessageslimit,usetls,usessl,username,password,sign,keystore,keystorepassword,keyalias,keypassword,allowdownload");
            rb.put("configuration.types",
                    "java.lang.String,java.lang.String,java.lang.String,java.lang.String,java.lang.String,java.lang.String,java.lang.Number,java.lang.Number,java.lang.Number,java.lang.Boolean,java.lang.Boolean,java.lang.Boolean,java.lang.Number,java.lang.Boolean,java.lang.Number,java.lang.Boolean,java.lang.Boolean,java.lang.String,java.lang.String,java.lang.Boolean,java.lang.String,java.lang.String,java.lang.String,java.lang.String,java.lang.Boolean");
            rb.put("configuration.formats",
                    "coldfusion.server.StringFormatter,coldfusion.server.StringFormatter,coldfusion.server.StringFormatter,coldfusion.server.StringFormatter,coldfusion.server.StringFormatter,coldfusion.server.StringFormatter,coldfusion.server.NumberFormatter,coldfusion.server.NumberFormatter,coldfusion.server.NumberFormatter,coldfusion.server.BooleanFormatter,coldfusion.server.BooleanFormatter,coldfusion.server.BooleanFormatter,coldfusion.server.NumberFormatter,coldfusion.server.BooleanFormatter,coldfusion.server.NumberFormatter,coldfusion.server.BooleanFormatter,coldfusion.server.BooleanFormatter,coldfusion.server.StringFormatter,coldfusion.server.StringFormatter,coldfusion.server.BooleanFormatter,coldfusion.server.StringFormatter,coldfusion.server.StringFormatter,coldfusion.server.StringFormatter,coldfusion.server.StringFormatter,coldfusion.server.BooleanFormatter");
        }
        return rb;
    }

    /**
     * Mail spooler storage of mail messages.  Implements soft references
     * so mail messages can be garbaged collected by the VM if it needs memory.
     * @author Tom Jordahl
     */
    public class SpoolerSoftCache extends SoftCache
    {
        //private int misses = 0;

        /**
         * Override the fetch method to read a mail spool file from disk
         *
         * @param key a String, the filename of the spool file
         * @return a MailImpl object or null if we couldn't read the file
         */
        protected Object fetch(Object key)
        {
            //misses += 1;
            String filename = (String) key;
            MailImpl impl = null;
            try
            {
                // Go read the file from disk
                impl = retrieveSpoolMail(filename);
            }
            catch (InvalidSpoolFileException spex)
            {
                // Something was wrong with this file.
                spex.setFilename(filename);
                logger.error(spex);
                postUndeliverMail(filename);
            }
            catch (MessagingException msgex)
            {
                logger.error(msgex);
                postUndeliverMail(filename);
            }
            catch (IOException ioex)
            {
                logger.error(ioex);
                postUndeliverMail(filename);
            }
            return impl;
        }

    }

    /**
     * Utility class that keeps track of worker status
     */
    protected class WorkerStatus
    {
        /**
         * The number of Worker object threads that are currently working
         * on something.
         */
        private int active = 0;

        /**
         * This boolean keeps track of if the very first thread has started
         * or not. This prevents this object from falsely reporting that the ThreadPool
         * is done, just because the first thread has not yet started.
         */
        private boolean started = false;

        /**
         * This method can be called to block the current thread
         * until the ThreadPool is done.
         */
        synchronized public void waitDone()
        {
            try
            {
                while (active > 0)
                {
                    wait();
                }
            }
            catch (InterruptedException e)
            {
                // ignore
            }
        }
        /**

         * Called to wait for the first thread to start. Once this method returns the
         * process has begun.
         */
        synchronized public void waitBegin()
        {
            try
            {
                while (!started)
                {
                    wait();
                }
            }
            catch (InterruptedException e)
            {
                // ignore
            }
        }


        /**
         * Called by a Worker object
         * to indicate that it has begun
         * working on a workload.
         */
        synchronized public void workerBegin()
        {
            active++;
            started = true;
            logger.debug(Thread.currentThread().getName() + " started up.");
            notify();
        }

        /**
         * Called by a Worker object to
         * indicate that it has completed a
         * workload.
         */
        synchronized public void workerEnd()
        {
            active--;
            logger.debug(Thread.currentThread().getName() + " shutting down (" + active + " still active).");
            notify();
        }

    }

	public String BCrypt(byte[] input, byte[] salt, int rounds){
        return new String(BCrypt.generate(input, salt, rounds));
    }

	public String SCrypt(byte[] input, byte[] salt, int cpuCost, int blockSize, int parallelizationParameter,
            int keylength){
        return  new String(  SCrypt.generate(input, salt, cpuCost, blockSize, parallelizationParameter, keylength));
    }

}

