/*
 * Copyright (c) 1997-2019 IDRsolutions (https://www.idrsolutions.com)
 */
function FDFXFAENCODING() {

    function BigEnd(d) {
        var p = 0;
        this.len = d.length;
        this.u8 = function () {
            return d[p++] & 0xff;
        };
        this.u16 = function () {
            return ((d[p++] & 0xff) << 8) | (d[p++] & 0xff);
        };
        this.u24 = function () {
            return ((d[p++] & 0xff) << 16) | ((d[p++] & 0xff) << 8) | (d[p++] & 0xff);
        };
        this.u32 = function () {
            return ((d[p++] & 0xff) << 24) | ((d[p++] & 0xff) << 16)
                    | ((d[p++] & 0xff) << 8) | (d[p++] & 0xff);
        };
        this.read = function (copyTo) {
            for (var i = 0, ii = copyTo.length; i < ii; i++) {
                copyTo[i] = d[p++];
            }
        };
        this.skip = function (n) {
            p += n;
        };
        this.pos = function () {
            return p;
        };
        this.moveTo = function (n) {
            p = n;
        }
    }
    ;

    function LittleEnd(d) {
        var p = 0;
        this.len = d.length;
        this.u8 = function () {
            return d[p++] & 0xff;
        };
        this.u16 = function () {
            return this.u8() | this.u8() << 8;
        };
        this.u24 = function () {
            return this.u8() | this.u8() << 8 | this.u8() << 16;
        };
        this.u32 = function () {
            return this.u8() | this.u8() << 8 | this.u8() << 16 | this.u8() << 24;
        };
        this.read = function (copyTo) {
            for (var i = 0, ii = copyTo.length; i < ii; i++) {
                copyTo[i] = d[p++];
            }
        };
        this.skip = function (n) {
            p += n;
        };
        this.pos = function (n) {
            return p;
        };
        this.moveTo = function (n) {
            p = n;
        };
    }

    function BitReader(d) {
        var p = 0, bufferSize = 0, buffer = 0;
        this.readBits = function (lenToRead) {
            while (bufferSize < lenToRead) {
                buffer = (buffer << 8) | d[p++] & 0xff;
                bufferSize += 8;
            }
            bufferSize -= lenToRead;
            return (buffer >>> bufferSize) & ((1 << lenToRead) - 1);
        };
    }

    this.decodeLZW = function (input, exp) {
        var output = new Array(exp);
        var bitsToGet = 9, bp = 0, tp = 258, op = 0, putBuffer = 0, putBits = 0;
        var codes = new Array(4096);
        for (var i = 0; i < 256; i++) {
            codes[i] = [i];
        }

        var code = 0, oldCode = 0;
        var chars;

        while (((code = findNext()) !== 257) && op < output.length) {

            if (code === 256) {
                codes = new Array(4096);
                for (var i = 0; i < 256; i++) {
                    codes[i] = [i];
                }
                tp = 258;
                bitsToGet = 9;

                code = findNext();
                if (code === 257) {
                    break;
                }
                addCodes(codes[code]);
                oldCode = code;
            } else {
                if (code < tp) {
                    chars = codes[code];
                    addCodes(chars);
                    addCodeToCodes(codes[oldCode], chars[0]);
                    oldCode = code;
                } else {
                    chars = codes[oldCode];
                    chars = generateCodeArray(chars, chars[0]);
                    addCodes(chars);
                    addCodeArrToCodes(chars);
                    oldCode = code;
                }
            }
        }

        return output;

        function addCodes(cc) {
            for (var i = 0, ii = cc.length; i < ii; i++) {
                output[op++] = cc[i];
            }
        }

        function addCodeToCodes(oldCodes, c) {
            var length = oldCodes.length;
            var string = new Array(length + 1);
            for (var i = 0; i < length; i++) {
                string[i] = oldCodes[i];
            }
            string[length] = c;
            codes[tp++] = string;
            switch (tp) {
                case 511:
                    bitsToGet = 10;
                    break;
                case 1023:
                    bitsToGet = 11;
                    break;
                case 2047:
                    bitsToGet = 12;
                    break;
                default:
                    break;
            }
        }

        function addCodeArrToCodes(codeArr) {
            codes[tp++] = codeArr;
            switch (tp) {
                case 511:
                    bitsToGet = 10;
                    break;
                case 1023:
                    bitsToGet = 11;
                    break;
                case 2047:
                    bitsToGet = 12;
                    break;
                default:
                    break;
            }
        }

        function generateCodeArray(oldCodes, newString) {
            var length = oldCodes.length;
            var string = new Array(length + 1);
            for (var i = 0; i < length; i++) {
                string[i] = oldCodes[i];
            }
            string[length] = newString;
            return string;
        }

        function findNext() {
            var comb = [511, 1023, 2047, 4095];
            putBuffer = (putBuffer << 8) | (input[bp++] & 0xff);
            putBits += 8;
            if (putBits < bitsToGet) {
                putBuffer = (putBuffer << 8) | (input[bp++] & 0xff);
                putBits += 8;
            }
            var code = (putBuffer >> (putBits - bitsToGet)) & comb[bitsToGet - 9];
            putBits -= bitsToGet;
            return code;
        }
    };

    this.decodePackBits = function (input, expected) {
        var total = 0;
        var bb = new Array();
        var i = 0, cc, b;
        while (total < expected) {
            var n = input[i++];
            if ((n >= 0) && (n <= 127)) {
                cc = n + 1;
                total += cc;
                for (var j = 0; j < cc; j++) {
                    bb.push(input[i++]);
                }
            } else if (n > 127) {
                b = input[i++];
                cc = -(n - 256) + 1;
                total += cc;
                for (var j = 0; j < cc; j++) {
                    bb.push(b);
                }
            }
        }
        return bb;
    };

    this.b64ToBytes = function (str) {
        var HK = [
            255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, //   0..15
            255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, //  16..31
            255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 62, 255, 255, 255, 63, //  32..47
            52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 255, 255, 255, 254, 255, 255, //  48..63
            255, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, //  64..79
            15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 255, 255, 255, 255, 255, //  80..95
            255, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, //  96..111
            41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 255, 255, 255, 255, 255, // 112..127
            255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, // 128..143
            255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
            255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
            255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
            255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
            255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
            255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
            255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255
        ];
        var res = new Array();
        var c1, c2, c3, e1, e2, e3, e4, i = 0;
        var input = str.replace(/[^A-Za-z0-9\+\/\=]/g, "");
        var ii = input.length;
        while (i < ii) {
            e1 = HK[input.charCodeAt(i++)];
            e2 = HK[input.charCodeAt(i++)];
            e3 = HK[input.charCodeAt(i++)];
            e4 = HK[input.charCodeAt(i++)];
            c1 = (e1 << 2) | (e2 >> 4);
            c2 = ((e2 & 15) << 4) | (e3 >> 2);
            c3 = ((e3 & 3) << 6) | e4;
            res.push(c1);
            if (e3 !== 64) {
                res.push(c2);
            }
            if (e4 !== 64) {
                res.push(c3);
            }
        }
        return res;
    };
    this.toUTF8 = function (array) {
        var char2, char3;
        var out = new Array();
        var len = array.length;
        var i = 0;
        var c;
        while (i < len) {
            c = array[i++];
            switch (c >> 4) {
                case 0:
                case 1:
                case 2:
                case 3:
                case 4:
                case 5:
                case 6:
                case 7:
                    // 0xxxxxxx
                    out.push(String.fromCharCode(c));
                    break;
                case 12:
                case 13:
                    // 110x xxxx   10xx xxxx
                    char2 = array[i++];
                    out.push(String.fromCharCode(((c & 0x1F) << 6) | (char2 & 0x3F)));
                    break;
                case 14:
                    // 1110 xxxx  10xx xxxx  10xx xxxx
                    char2 = array[i++];
                    char3 = array[i++];
                    out.push(String.fromCharCode(((c & 0x0F) << 12) |
                            ((char2 & 0x3F) << 6) |
                            ((char3 & 0x3F) << 0)));
                    break;
            }
        }
        return out.join('');
    };
    this.bytesToB64 = function (data) {
        var HK = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
        var out = new Array(data.length);
        var chr1, chr2, chr3, enc1, enc2, enc3, enc4;
        var i = 0;

        while (i < data.length) {
            chr1 = data[i++];
            chr2 = data[i++];
            chr3 = data[i++];

            enc1 = chr1 >> 2;
            enc2 = ((chr1 & 3) << 4) | (chr2 >> 4);
            enc3 = ((chr2 & 15) << 2) | (chr3 >> 6);
            enc4 = chr3 & 63;

            if (isNaN(chr2)) {
                enc3 = enc4 = 64;
            } else if (isNaN(chr3)) {
                enc4 = 64;
            }
            out.push(HK.charAt(enc1) + HK.charAt(enc2) + HK.charAt(enc3) + HK.charAt(enc4));
        }
        return out.join('');
    };

    this.tiffToJpeg64 = function (data) {

        var Tags = {
            NewSubfileType: 0xfe, SubfileType: 0xff, ImageWidth: 0x100,
            ImageHeight: 0x101, BitsPerSample: 0x102, Compression: 0x103,
            Uncompressed: 1, CCITT_ID: 2, Group_3_Fax: 3, Group_4_Fax: 4,
            LZW: 5, JPEG: 6, JPEG_TechNote: 7, ADOBEDEFLATE: 8,
            PackBits: 0x8005, Deflate: 0x80b2, SGI_RLE: 0x8774,
            PhotometricInterpolation: 0x106,
            WhiteIsZero: 0, BlackIsZero: 1, RGB: 2, RGB_Palette: 3,
            Transparency_Mask: 4, CMYK: 5, YCbCr: 6, CIELab: 8,
            Threshholding: 0x107, CellWidth: 0x108, CellLength: 0x109,
            FillOrder: 0x10a, DocumentName: 0x10D, ImageDescription: 0x10e,
            Make: 0x10f, Model: 0x110, StripOffsets: 0x111,
            Orientation: 0x112, SamplesPerPixel: 0x115, RowsPerStrip: 0x116,
            StripByteCounts: 0x117, MinSampleValue: 0x118, MaxSampleValue: 0x119,
            Xresolution: 0x11a, Yresolution: 0x11b, PlanarConfiguration: 0x11c,
            PageName: 0x11d, Xposition: 0x11e, Yposition: 0x11f,
            FreeOffsets: 0x120, FreeByteCounts: 0x121, GrayResponseUnit: 0x122,
            GrayResponseCurve: 0x123, T4Options: 0x124, T6Options: 0x125,
            ResolutionUnit: 0x128, PageNumber: 0x129, TransferFunction: 0x12D,
            Software: 0x131, DateTime: 0x132, Artist: 0x13b,
            HostComputer: 0x13c, Predictor: 0x13d, WhitePoint: 0x13e,
            PrimaryChromaticities: 0x13f, ColorMap: 0x140, HalftoneHints: 0x141,
            TileWidth: 0x142, TileLength: 0x143, TileOffsets: 0x144,
            TileByteCounts: 0x145, BadFaxLines: 0x146, CleanFaxData: 0x147,
            ConsecutiveBadFaxLines: 0x148, SubIFDs: 0x14a, InkSet: 0x14c,
            InkNames: 0x14d, NumberOfInks: 0x14e, DotRange: 0x150,
            TargetPrinter: 0x151, ExtraSamples: 0x152, SampleFormat: 0x153,
            SMinSampleValue: 0x154, SMaxSampleValue: 0x155, TransferRange: 0x156,
            ClipPath: 0x157, XClipPathUnits: 0x158, YClipPathUnits: 0x159,
            Indexed: 0x15a, JPEGTables: 0x15b,
            JPEGProc: 0x200, JPEGInterchangeFormat: 0x201,
            JPEGInterchangeFormatLength: 0x202,
            JPEGRestartInterval: 0x203, JPEGLosslessPredictors: 0x205,
            JPEGPointTransforms: 0x206,
            JPEGQTables: 0x207, JPEGDCTables: 0x208, JPEGACTables: 0x209,
            YCbCrCoefficients: 0x211, YCbCrSubSampling: 0x212,
            YCbCrPositioning: 0x213,
            ReferenceBlackWhite: 0x214, StripRowCounts: 0x22f, XMP: 0x2bc,
            ImageID: 0x800d, Copyright: 0x8298,
            ICC: 0x8773, Exif_IFD: 0x8769, ExifVersion: 0x9000,
            DateTimeOriginal: 0x9003, DateTimeDigitized: 0x9004,
            ComponentConfiguration: 0x9101, CompressedBitsPerPixel: 0x9102,
            ApertureValue: 0x9202, ImageNumber: 0x9211, ImageHistory: 0x9213,
            ColorSpace: 0xa0001, PixelXDimension: 0xa002, PixelYDimension: 0xa003
        };

        function getValue(reader, ft) {
            var value;
            if (ft === 3) {
                value = reader.u16();
                reader.u16(); // read and ignore;
            } else {
                value = reader.u32();
            }
            return value;
        }

        function getValues(reader, ft, nv) {
            var res;
            if (nv === 1) {
                res = [reader.u32()];
            } else {
                var offset = reader.u32();
                var cur = reader.pos();
                reader.moveTo(offset);
                res = new Array(nv);
                for (var i = 0; i < nv; i++) {
                    res[i] = ft === 3 ? reader.u16() : reader.u32();
                }
                reader.moveTo(cur);
            }
            return res;
        }

        function readBPS(reader, nv) {
            var res = new Array(nv);
            switch (nv) {
                case 1:
                    res[0] = reader.u16();
                    reader.skip(2);
                    break;
                case 2:
                    res[0] = reader.u16();
                    res[1] = reader.u16();
                    break;
                default:
                    var offset = reader.u32();
                    var cur = reader.pos();
                    reader.moveTo(offset);
                    for (var i = 0; i < nv; i++) {
                        res[i] = reader.u16();
                    }
                    reader.moveTo(cur);
            }
            return res;
        }

        function readColorMap(reader, nv) {
            var offset = reader.u32();
            var cur = reader.pos();
            reader.moveTo(offset);
            var nColors = nv / 3;
            var res = new Array(3);
            var j = 0;
            while (j < 3) {
                res[j] = new Array(nColors);
                for (var i = 0; i < nColors; i++) {
                    res[j][i] = (reader.u16() >> 8) & 0xff;
                }
                j++;
            }
            reader.moveTo(cur);
            return res;
        }

        function doPredictor(output, iw, sh, nComp) {
            var c;
            for (var j = 0; j < sh; j++) {
                c = nComp * (j * iw + 1);
                for (var k = nComp, kk = iw * nComp; k < kk; k++) {
                    output[c] += output[c - nComp];
                    c++;
                }
            }
        }

        function uncompressData(input, exp, compType) {
            var output = [];
            var enc = new FDFXFAENCODING();
            switch (compType) {
                case Tags.Uncompressed:
                    output = input;
                    break;
                case Tags.LZW:
                    output = enc.decodeLZW(input, exp);
                    break;
                case Tags.PackBits:
                    output = enc.decodePackBits(input, exp);
                    break;
                case Tags.ADOBEDEFLATE:
                case Tags.Deflate:
                    output = enc.decodeFlate(input);
                    break;
            }
            return output;
        }

        var a = data[0];
        var b = data[1];
        var reader;
        if (a === 77 && b === 77) {
            reader = new BigEnd(data);
        } else if (a === 73 && b === 73) {
            reader = new LittleEnd(data);
        }
        reader.skip(2);
        reader.skip(2);//magic number;
        var ifd = reader.u32();
        //reading ifd;
        reader.moveTo(ifd);
        var nEntry = reader.u16();

        var iw, ih, tw, th, spp, bps = [1], compType, pMetric, rps, fill, colorMap,
                pred = 0, sFormat, planar, stripOffs, stripLens, tOffs, tLens;
        var fn, ft, nv;
        for (var i = 0; i < nEntry; i++) {
            fn = reader.u16();
            ft = reader.u16();
            nv = reader.u32();
//            console.log(fn + " " + nv);
            switch (fn) {
                case Tags.ImageWidth:
                    iw = getValue(reader, ft);
                    if (iw > 0xffffff) {
                        iw &= 0xffff;
                    }
                    break;
                case Tags.ImageHeight:
                    ih = getValue(reader, ft);
                    if (ih > 0xffffff) {
                        ih &= 0xffff;
                    }
                    break;
                case Tags.BitsPerSample:
                    bps = readBPS(reader, nv);
                    break;
                case Tags.Compression:
                    compType = reader.u16();
                    reader.skip(2);
                    break;
                case Tags.PhotometricInterpolation:
                    pMetric = reader.u16();
                    reader.skip(2);
                    break;
                case Tags.RowsPerStrip:
                    rps = getValue(reader, ft);
                    break;
                case Tags.StripOffsets:
                    stripOffs = getValues(reader, ft, nv);
                    break;
                case Tags.StripByteCounts:
                    stripLens = getValues(reader, ft, nv);
                    break;
                case Tags.SamplesPerPixel:
                    spp = reader.u16();
                    reader.skip(2);
                    break;
                case Tags.PlanarConfiguration:
                    planar = reader.u16();
                    reader.skip(2);
                    break;
                case Tags.ColorMap:
                    colorMap = readColorMap(reader, nv);
                    break;
                case Tags.TileWidth:
                    tw = getValue(reader, ft);
                    break;
                case Tags.TileLength:
                    th = getValue(reader, ft);
                    break;
                case Tags.TileOffsets:
                    tOffs = getValues(reader, ft, nv);
                    break;
                case Tags.TileByteCounts:
                    tLens = getValues(reader, ft, nv);
                    break;
                case Tags.FillOrder:
                    fill = reader.u32();
                    break;
                case Tags.Predictor:
                    pred = reader.u32();
                    break;
                case Tags.SampleFormat:
                    sFormat = getValues(reader, ft, nv);
                    break;
                default:
                    reader.u32();
            }
        }

        var iData = new Array(iw * ih * 4);
        var sampleLen = 0;
        var nComp = planar === 2 ? 1 : bps.length;
        if (planar === 2) {
            sampleLen = bps[0];
        } else {
            for (var i = 0, ii = bps.length; i < ii; i++) {
                sampleLen += bps[i];
            }
        }
        var bpsv = bps[0];

        if (tw > 0) {
            tOffs = tOffs.length === 0 ? stripOffs : tOffs;
            tLens = tLens.length === 0 ? stripLens : tLens;
            var xTiles = ((iw + (tw - 1)) / tw) | 0;
            var nTiles = tOffs.length;
            var exp = ((tw * th * sampleLen + 7) >> 3) | 0;
            var output = [];
            var tx, ty;
            for (var i = 0, ii = nTiles; i < ii; i++) {
                var tt = planar === 2 ? i % ((ii / spp) | 0) : i;
                tx = ((tt % xTiles) | 0) * tw;
                ty = ((tt / xTiles) | 0) * th;
                reader.moveTo(tOffs[i]);
                var tData = new Array(tLens[i]);
                reader.read(tData);
                var output = uncompressData(tData, exp, compType);
                if (pred === 2) {
                    doPredictor(output, iw, th, nComp);
                }
                var n = 0;
                if (pMetric === Tags.WhiteIsZero) {
                    for (var j = 0, jj = output.length; j < jj; j++) {
                        output[j] = ((output[n++] & 0xff) ^ 0xff);
                    }
                }
                var iw8 = 8 - (((iw * bps[0] * nComp) % 8) | 0);
                var scale = bpsv > 8 ? 1 : 255 / (Math.pow(2, bpsv) - 1);
                var shift = bpsv > 8 ? bpsv - 8 : 0;
                var p = 0, t, r, g, b, a, ix, iy, pp;
                var br = new BitReader(output);
                switch (spp) {
                    case 0:
                    case 1:
                        for (var h = 0; h < th; h++) {
                            iy = ty + h;
                            for (var w = 0; w < tw; w++) {
                                if (colorMap) {
                                    t = br.readBits(bpsv);
                                    r = colorMap[0][t];
                                    g = colorMap[1][t];
                                    b = colorMap[2][t];
                                } else {
                                    t = ((br.readBits(bpsv) >> shift) * scale) | 0;
                                    r = g = b = t;
                                }
                                ix = tx + w;
                                if (ix < iw && iy < ih) {
                                    p = (iy * iw + ix) << 2;
                                    iData[p++] = r;
                                    iData[p++] = g;
                                    iData[p++] = b;
                                    iData[p++] = 255;
                                }
                            }
                        }
                        break;
                    case 2:
                        break;
                    case 3:
                    case 4:
                        if (planar === 2) {
                            pp = (i / ((ii / spp) | 0)) | 0;
                            for (var h = 0; h < th; h++) {
                                iy = ty + h;
                                for (var w = 0; w < tw; w++) {
                                    r = ((br.readBits(bpsv) >> shift) * scale) | 0;
                                    ix = tx + w;
                                    if (ix < iw && iy < ih) {
                                        p = (iy * iw + ix) << 2;
                                        iData[p + pp] = r;
                                        if (spp === 3) {
                                            iData[p + 3] = 255;
                                        }
                                    }
                                }
                            }
                        } else {
                            for (var h = 0; h < th; h++) {
                                iy = ty + h;
                                for (var w = 0; w < tw; w++) {
                                    r = ((br.readBits(bpsv) >> shift) * scale) | 0;
                                    g = ((br.readBits(bpsv) >> shift) * scale) | 0;
                                    b = ((br.readBits(bpsv) >> shift) * scale) | 0;
                                    a = spp === 3 ? 255 : (br.readBits(bpsv) * scale) | 0;
                                    ix = tx + w;
                                    if (ix < iw && iy < ih) {
                                        p = (iy * iw + ix) << 2;
                                        iData[p++] = r;
                                        iData[p++] = g;
                                        iData[p++] = b;
                                        iData[p++] = a;
                                    }
                                }
                            }
                        }
                        break;
                }
            }
        } else {
            var boss = new Array();
            for (var i = 0, ii = stripOffs.length; i < ii; i++) {
                reader.moveTo(stripOffs[i]);
                var strips = new Array(stripLens[i]);
                reader.read(strips);
                var tt = planar === 2 ? i % ((ii / spp) | 0) : i;
                var balance = ih - (rps * tt);
                var sh = balance < rps ? balance : rps;
                var exp = (iw * sh * sampleLen + 7) >> 3;
                var output = uncompressData(strips, exp, compType);
                if (pred === 2) {
                    doPredictor(output, iw, sh, nComp);
                }
                var n = 0;
                if (pMetric === Tags.WhiteIsZero) {
                    for (var j = 0, jj = output.length; j < jj; j++) {
                        output[j] = ((output[n++] & 0xff) ^ 0xff);
                    }
                }
                for (var j = 0, jj = output.length; j < jj; j++) {
                    boss.push(output[j] & 0xff);
                }
            }
            var iw8 = 8 - (((iw * bpsv * nComp) % 8) | 0);
            var scale = bpsv > 8 ? 1 : 255 / (Math.pow(2, bpsv) - 1);
            var shift = bpsv > 8 ? bpsv - 8 : 0;
            var p = 0, t, r, g, b, a;
            var br = new BitReader(boss);
            switch (spp) {
                case 0:
                case 1:
                    for (var h = 0; h < ih; h++) {
                        for (var w = 0; w < iw; w++) {
                            if (colorMap) {
                                t = br.readBits(bpsv);
                                r = colorMap[0][t];
                                g = colorMap[1][t];
                                b = colorMap[2][t];
                            } else {
                                t = ((br.readBits(bpsv) >> shift) * scale) | 0;
                                r = g = b = t;
                            }
                            iData[p++] = r;
                            iData[p++] = g;
                            iData[p++] = b;
                            iData[p++] = 255;
                        }
                        if (iw8 !== 8) {
                            br.readBits(iw8);
                        }
                    }
                    break;
                case 2:
                    break;
                case 3:
                case 4:
                    if (planar === 2) {
                        for (var n = 0, nn = bps.length; n < nn; n++) {
                            p = n;
                            for (var h = 0; h < ih; h++) {
                                for (var w = 0; w < iw; w++) {
                                    iData[p] = (br.readBits(bpsv) * scale) | 0;
                                    p += 4;
                                }
                            }
                            if (iw8 !== 8) {
                                br.readBits(iw8);
                            }
                        }
                        if (spp === 3) {
                            for (var n = 3, nn = ih * iw * 4; n < nn; n += 4) {
                                iData[n] = 255;
                            }
                        }
                    } else {
                        for (var h = 0; h < ih; h++) {
                            for (var w = 0; w < iw; w++) {
                                iData[p++] = ((br.readBits(bpsv) >> shift) * scale) | 0;
                                iData[p++] = ((br.readBits(bpsv) >> shift) * scale) | 0;
                                iData[p++] = ((br.readBits(bpsv) >> shift) * scale) | 0;
                                iData[p++] = spp === 3 ? 255 : ((br.readBits(bpsv) >> shift) * scale) | 0;
                            }
                            if (iw8 !== 8) {
                                br.readBits(iw8);
                            }
                        }
                    }
                    break;
            }
        }
        if (pMetric === Tags.CMYK) {
            var r, g, b, a;
            for (var i = 0, ii = iw * ih * 4; i < ii; i += 4) {
                r = iData.data[i];
                g = iData.data[i + 1];
                b = iData.data[i + 2];
                a = iData.data[i + 3];
                if (pMetric === Tags.CMYK) {
                    r = (255 * (1 - r / 255.0) * (1 - a / 255)) | 0;
                    g = (255 * (1 - g / 255.0) * (1 - a / 255)) | 0;
                    b = (255 * (1 - b / 255.0) * (1 - a / 255)) | 0;
                }
                iData.data[i] = r;
                iData.data[i + 1] = g;
                iData.data[i + 2] = b;
                iData.data[i + 3] = 0xff;
            }
        }
        var compressed = this.encodeJPEG(iData, iw, ih);
        return this.bytesToB64(compressed);

    };

    this.decodeFlate = function (data) {
        var MASK_BITS = [
            0x0000, 0x0001, 0x0003, 0x0007, 0x000f, 0x001f, 0x003f, 0x007f, 0x00ff,
            0x01ff, 0x03ff, 0x07ff, 0x0fff, 0x1fff, 0x3fff, 0x7fff, 0xffff
        ];
        var CODE_LENGTHS = [
            3, 4, 5, 6, 7, 8, 9, 10, 11, 13, 15, 17, 19, 23, 27, 31, 35, 43, 51, 59,
            67, 83, 99, 115, 131, 163, 195, 227, 258, 0, 0
        ];
        var CODE_DISTANCES = [
            1, 2, 3, 4, 5, 7, 9, 13, 17, 25, 33, 49, 65, 97, 129, 193, 257, 385, 513,
            769, 1025, 1537, 2049, 3073, 4097, 6145, 8193, 12289, 16385, 24577
        ];
        var FLATE_MARGINS = [
            16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15
        ];
        var zip_cplext = [
            0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4,
            5, 5, 5, 5, 0, 99, 99
        ];
        var zip_cpdext = [
            0, 0, 0, 0, 1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9, 10,
            10, 11, 11, 12, 12, 13, 13
        ];

        var WINDOW_SIZE = 32768, STORED_BLOCK = 0;
        var windowSlides = new Array(WINDOW_SIZE << 1);
        var flateType = -1;
        var flatePos = 0;
        var bitsBuffer = 0;
        var bitsLength = 0;
        var isEOF = false;
        var zip_wp = 0;
        var zip_fixed_tl = null;
        var zip_fixed_td, zip_fixed_bl, zip_fixed_bd;
        var zip_copy_leng = 0;
        var zip_copy_dist = 0;
        var zip_lbits = 9;
        var zip_dbits = 6;
        var zip_tl = null, zip_td;
        var zip_bl, zip_bd;


        var flateData = new Array(data.length - 2);
        for (var i = 2, ii = data.length; i < ii; i++) {
            flateData[i - 2] = data[i];
        }
        var res = [], k, j, buffLen = 1024;

        var buff = new Array(buffLen);
        while ((k = inflateChunks(buff, 0, buffLen)) > 0) {
            for (j = 0; j < k; j++) {
                res.push(buff[j]);
            }
        }

        flateData = null;
        return res;

        function readByte() {
            if (flateData.length === flatePos) {
                return -1;
            }
            return flateData[flatePos++] & 0xff;
        }

        function readBits(x) {
            return bitsBuffer & MASK_BITS[x];
        }

        function HuffmanTableList() {
            this.next = null;
            this.list = null;
        }

        function HuffmanTableNode() {
            this.e = 0;
            this.b = 0;
            this.n = 0;
            this.t = null;
        }

        function HuffmanTableBlock(b, n, s, d, e, mm) {
            this.BMAX = 16;
            this.N_MAX = 288;
            this.status = 0;
            this.root = null;
            this.m = 0;
            {
                var a, c = new Array(this.BMAX + 1);
                var el, f, g, h, i, j, k, lx = new Array(this.BMAX + 1);
                var p, pidx, q;
                var r = new HuffmanTableNode();
                var u = new Array(this.BMAX);
                var v = new Array(this.N_MAX);
                var x = new Array(this.BMAX + 1);
                var w, xp, y, z, o, tail;

                tail = this.root = null;
                for (i = 0; i < c.length; i++) {
                    c[i] = 0;
                }
                for (i = 0; i < lx.length; i++) {
                    lx[i] = 0;
                }
                for (i = 0; i < u.length; i++) {
                    u[i] = null;
                }
                for (i = 0; i < v.length; i++) {
                    v[i] = 0;
                }
                for (i = 0; i < x.length; i++) {
                    x[i] = 0;
                }

                el = n > 256 ? b[256] : this.BMAX;
                p = b;
                pidx = 0;
                i = n;
                do {
                    c[p[pidx]]++;
                    pidx++;
                } while (--i > 0);
                if (c[0] === n) {
                    this.root = null;
                    this.m = 0;
                    this.status = 0;
                    return;
                }
                for (j = 1; j <= this.BMAX; j++) {
                    if (c[j] !== 0) {
                        break;
                    }
                }
                k = j;
                if (mm < j) {
                    mm = j;
                }
                for (i = this.BMAX; i !== 0; i--) {
                    if (c[i] !== 0) {
                        break;
                    }
                }
                g = i;
                if (mm > i) {
                    mm = i;
                }
                for (y = 1 << j; j < i; j++, y <<= 1) {
                    if ((y -= c[j]) < 0) {
                        this.status = 2;
                        this.m = mm;
                        return;
                    }
                }
                if ((y -= c[i]) < 0) {
                    this.status = 2;
                    this.m = mm;
                    return;
                }
                c[i] += y;

                x[1] = j = 0;
                p = c;
                pidx = 1;
                xp = 2;
                while (--i > 0) {
                    x[xp++] = (j += p[pidx++]);
                }

                p = b;
                pidx = 0;
                i = 0;
                do {
                    if ((j = p[pidx++]) !== 0) {
                        v[x[j]++] = i;
                    }
                } while (++i < n);
                n = x[g];
                x[0] = i = 0;
                p = v;
                pidx = 0;
                h = -1;
                w = lx[0] = 0;
                q = null;
                z = 0;

                for (; k <= g; k++) {
                    a = c[k];
                    while (a-- > 0) {
                        while (k > w + lx[1 + h]) {
                            w += lx[1 + h];
                            h++;
                            z = (z = g - w) > mm ? mm : z;
                            if ((f = 1 << (j = k - w)) > a + 1) {
                                f -= a + 1;
                                xp = k;
                                while (++j < z) {
                                    if ((f <<= 1) <= c[++xp])
                                        break;
                                    f -= c[xp];
                                }
                            }
                            if (w + j > el && w < el)
                                j = el - w;
                            z = 1 << j;
                            lx[1 + h] = j;

                            q = new Array(z);
                            for (o = 0; o < z; o++) {
                                q[o] = new HuffmanTableNode();
                            }

                            if (tail) {
                                tail = tail.next = new HuffmanTableList();
                            } else {
                                tail = this.root = new HuffmanTableList();
                            }
                            tail.next = null;
                            tail.list = q;
                            u[h] = q;
                            if (h > 0) {
                                x[h] = i;
                                r.b = lx[h];
                                r.e = 16 + j;
                                r.t = q;
                                j = (i & ((1 << w) - 1)) >> (w - lx[h]);
                                u[h - 1][j].e = r.e;
                                u[h - 1][j].b = r.b;
                                u[h - 1][j].n = r.n;
                                u[h - 1][j].t = r.t;
                            }
                        }

                        r.b = k - w;
                        if (pidx >= n) {
                            r.e = 99;
                        } else if (p[pidx] < s) {
                            r.e = (p[pidx] < 256 ? 16 : 15);
                            r.n = p[pidx++];
                        } else {
                            r.e = e[p[pidx] - s];
                            r.n = d[p[pidx++] - s];
                        }

                        f = 1 << (k - w);
                        for (j = i >> w; j < z; j += f) {
                            q[j].e = r.e;
                            q[j].b = r.b;
                            q[j].n = r.n;
                            q[j].t = r.t;
                        }

                        for (j = 1 << (k - 1); (i & j) !== 0; j >>= 1) {
                            i ^= j;
                        }
                        i ^= j;

                        while ((i & ((1 << w) - 1)) !== x[h]) {
                            w -= lx[h];
                            h--;
                        }
                    }
                }

                this.m = lx[1];
                this.status = ((y !== 0 && g !== 1) ? 1 : 0);
            }
        }

        function zip_NEEDBITS(n) {
            while (bitsLength < n) {
                bitsBuffer |= readByte() << bitsLength;
                bitsLength += 8;
            }
        }

        function ignoreBits(n) {
            bitsBuffer >>= n;
            bitsLength -= n;
        }

        function decodeHFC(buff, off, size) {

            if (size === 0) {
                return 0;
            }
            var e, t, n = 0;
            while (true) {
                zip_NEEDBITS(zip_bl);
                t = zip_tl.list[readBits(zip_bl)];
                e = t.e;
                while (e > 16) {
                    if (e === 99) {
                        return -1;
                    }
                    ignoreBits(t.b);
                    e -= 16;
                    zip_NEEDBITS(e);
                    t = t.t[readBits(e)];
                    e = t.e;
                }
                ignoreBits(t.b);

                if (e === 16) {
                    zip_wp &= WINDOW_SIZE - 1;
                    buff[off + n++] = windowSlides[zip_wp++] = t.n;
                    if (n === size) {
                        return size;
                    }
                    continue;
                }

                if (e === 15) {
                    break;
                }
                zip_NEEDBITS(e);
                zip_copy_leng = t.n + readBits(e);
                ignoreBits(e);

                zip_NEEDBITS(zip_bd);
                t = zip_td.list[readBits(zip_bd)];
                e = t.e;

                while (e > 16) {
                    if (e === 99) {
                        return -1;
                    }
                    ignoreBits(t.b);
                    e -= 16;
                    zip_NEEDBITS(e);
                    t = t.t[readBits(e)];
                    e = t.e;
                }
                ignoreBits(t.b);
                zip_NEEDBITS(e);
                zip_copy_dist = zip_wp - t.n - readBits(e);
                ignoreBits(e);

                while (zip_copy_leng > 0 && n < size) {
                    zip_copy_leng--;
                    zip_copy_dist &= WINDOW_SIZE - 1;
                    zip_wp &= WINDOW_SIZE - 1;
                    buff[off + n++] = windowSlides[zip_wp++] = windowSlides[zip_copy_dist++];
                }
                if (n === size) {
                    return size;
                }
            }

            flateType = -1;
            return n;
        }

        function decodeHFCS(buff, off, size) {
            var n;
            n = bitsLength & 7;
            ignoreBits(n);
            zip_NEEDBITS(16);
            n = readBits(16);
            ignoreBits(16);
            zip_NEEDBITS(16);
            if (n !== ((~bitsBuffer) & 0xffff))
                return -1;
            ignoreBits(16);

            zip_copy_leng = n;

            n = 0;
            while (zip_copy_leng > 0 && n < size) {
                zip_copy_leng--;
                zip_wp &= WINDOW_SIZE - 1;
                zip_NEEDBITS(8);
                buff[off + n++] = windowSlides[zip_wp++] = readBits(8);
                ignoreBits(8);
            }

            if (zip_copy_leng === 0) {
                flateType = -1;
            }
            return n;
        }

        function decodeHFCF(buff, off, size) {
            if (zip_fixed_tl === null) {
                var i;
                var l = new Array(288);
                var h;

                for (i = 0; i < 144; i++) {
                    l[i] = 8;
                }
                for (; i < 256; i++) {
                    l[i] = 9;
                }
                for (; i < 280; i++) {
                    l[i] = 7;
                }
                for (; i < 288; i++) {
                    l[i] = 8;
                }
                zip_fixed_bl = 7;

                h = new HuffmanTableBlock(l, 288, 257, CODE_LENGTHS, zip_cplext, zip_fixed_bl);
                if (h.status !== 0) {
                    throw("EcmaFlateDecodeError : Huffman Status " + h.status);
                    return -1;
                }
                zip_fixed_tl = h.root;
                zip_fixed_bl = h.m;

                for (i = 0; i < 30; i++) {
                    l[i] = 5;
                }
                zip_fixed_bd = 5;

                h = new HuffmanTableBlock(l, 30, 0, CODE_DISTANCES, zip_cpdext, zip_fixed_bd);
                if (h.status > 1) {
                    zip_fixed_tl = null;
                    throw("EcmaFlateDecodeError : Huffman Status" + h.status);
                    return -1;
                }
                zip_fixed_td = h.root;
                zip_fixed_bd = h.m;
            }

            zip_tl = zip_fixed_tl;
            zip_td = zip_fixed_td;
            zip_bl = zip_fixed_bl;
            zip_bd = zip_fixed_bd;
            return decodeHFC(buff, off, size);
        }

        function decodeHFCD(buff, off, size) {
            var i, j, l, n, t, nb, nl, nd, h;
            var ll = new Array(286 + 30);

            for (i = 0; i < ll.length; i++) {
                ll[i] = 0;
            }

            zip_NEEDBITS(5);
            nl = 257 + readBits(5);
            ignoreBits(5);
            zip_NEEDBITS(5);
            nd = 1 + readBits(5);
            ignoreBits(5);
            zip_NEEDBITS(4);
            nb = 4 + readBits(4);
            ignoreBits(4);
            if (nl > 286 || nd > 30) {
                return -1;
            }
            for (j = 0; j < nb; j++) {
                zip_NEEDBITS(3);
                ll[FLATE_MARGINS[j]] = readBits(3);
                ignoreBits(3);
            }
            for (; j < 19; j++) {
                ll[FLATE_MARGINS[j]] = 0;
            }

            zip_bl = 7;
            h = new HuffmanTableBlock(ll, 19, 19, null, null, zip_bl);
            if (h.status !== 0) {
                return -1;
            }
            zip_tl = h.root;
            zip_bl = h.m;
            n = nl + nd;
            i = l = 0;
            while (i < n) {
                zip_NEEDBITS(zip_bl);
                t = zip_tl.list[readBits(zip_bl)];
                j = t.b;
                ignoreBits(j);
                j = t.n;
                if (j < 16)
                    ll[i++] = l = j;
                else if (j === 16) {
                    zip_NEEDBITS(2);
                    j = 3 + readBits(2);
                    ignoreBits(2);
                    if (i + j > n)
                        return -1;
                    while (j-- > 0)
                        ll[i++] = l;
                } else if (j === 17) {
                    zip_NEEDBITS(3);
                    j = 3 + readBits(3);
                    ignoreBits(3);
                    if (i + j > n)
                        return -1;
                    while (j-- > 0)
                        ll[i++] = 0;
                    l = 0;
                } else {
                    zip_NEEDBITS(7);
                    j = 11 + readBits(7);
                    ignoreBits(7);
                    if (i + j > n)
                        return -1;
                    while (j-- > 0)
                        ll[i++] = 0;
                    l = 0;
                }
            }

            zip_bl = zip_lbits;
            h = new HuffmanTableBlock(ll, nl, 257, CODE_LENGTHS, zip_cplext, zip_bl);
            if (zip_bl === 0) {
                h.status = 1;
            }
            if (h.status !== 0) {
                return -1;
            }
            zip_tl = h.root;
            zip_bl = h.m;

            for (i = 0; i < nd; i++)
                ll[i] = ll[i + nl];
            zip_bd = zip_dbits;
            h = new HuffmanTableBlock(ll, nd, 0, CODE_DISTANCES, zip_cpdext, zip_bd);
            zip_td = h.root;
            zip_bd = h.m;

            if (zip_bd === 0 && nl > 257 || h.status !== 0) {
                return -1;
            }
            return decodeHFC(buff, off, size);
        }

        function inflateChunks(buff, off, size) {
            var n = 0, i;
            while (n < size) {
                if (isEOF && flateType === -1) {
                    return n;
                }
                if (zip_copy_leng > 0) {
                    if (flateType !== STORED_BLOCK) {
                        while (zip_copy_leng > 0 && n < size) {
                            zip_copy_leng--;
                            zip_copy_dist &= WINDOW_SIZE - 1;
                            zip_wp &= WINDOW_SIZE - 1;
                            buff[off + n++] = windowSlides[zip_wp++] =
                                    windowSlides[zip_copy_dist++];
                        }
                    } else {
                        while (zip_copy_leng > 0 && n < size) {
                            zip_copy_leng--;
                            zip_wp &= WINDOW_SIZE - 1;
                            zip_NEEDBITS(8);
                            buff[off + n++] = windowSlides[zip_wp++] = readBits(8);
                            ignoreBits(8);
                        }
                        if (zip_copy_leng === 0)
                            flateType = -1;
                    }
                    if (n === size)
                        return n;
                }

                if (flateType === -1) {
                    if (isEOF) {
                        break;
                    }
                    zip_NEEDBITS(1);
                    if (readBits(1) !== 0) {
                        isEOF = true;
                    }
                    ignoreBits(1);
                    zip_NEEDBITS(2);
                    flateType = readBits(2);
                    ignoreBits(2);
                    zip_tl = null;
                    zip_copy_leng = 0;
                }
                switch (flateType) {
                    case 0:
                        i = decodeHFCS(buff, off + n, size - n);
                        break;
                    case 1:
                        if (zip_tl) {
                            i = decodeHFC(buff, off + n, size - n);
                        } else {
                            i = decodeHFCF(buff, off + n, size - n);
                        }
                        break;
                    case 2:
                        if (zip_tl) {
                            i = decodeHFC(buff, off + n, size - n);
                        } else {
                            i = decodeHFCD(buff, off + n, size - n);
                        }
                        break;
                    default:
                        i = -1;
                        break;
                }
                if (i === -1) {
                    return (isEOF) ? 0 : -1;
                }
                n += i;
            }
            return n;
        }
    };

    this.encodeJPEG = function (imageData, iw, ih) {

        var quality = 50, bp = 0;
        var stream;
        var QL = [
            16, 11, 10, 16, 24, 40, 51, 61, 12, 12, 14, 19, 26, 58, 60, 55,
            14, 13, 16, 24, 40, 57, 69, 56, 14, 17, 22, 29, 51, 87, 80, 62,
            18, 22, 37, 56, 68, 109, 103, 77, 24, 35, 55, 64, 81, 104, 113, 92,
            49, 64, 78, 87, 103, 121, 120, 101, 72, 92, 95, 98, 112, 100, 103, 99
        ];
        var QC = [
            17, 18, 24, 47, 99, 99, 99, 99, 18, 21, 26, 66, 99, 99, 99, 99,
            24, 26, 56, 99, 99, 99, 99, 99, 47, 66, 99, 99, 99, 99, 99, 99,
            99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99,
            99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99
        ];
        var ZIGZAG = [
            0, 1, 5, 6, 14, 15, 27, 28, 2, 4, 7, 13, 16, 26, 29, 42,
            3, 8, 12, 17, 25, 30, 41, 43, 9, 11, 18, 24, 31, 40, 44, 53,
            10, 19, 23, 32, 39, 45, 52, 54, 20, 22, 33, 38, 46, 51, 55, 60,
            21, 34, 37, 47, 50, 56, 59, 61, 35, 36, 48, 49, 57, 58, 62, 63
        ];

        var CODESDCL = [0, 0, 1, 5, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0];
        var SYMBOLSDCL = [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11];
        var CODESACL = [0, 0, 2, 1, 3, 3, 2, 4, 3, 5, 5, 4, 4, 0, 0, 1, 0x7d];
        var SYMBOLSACL = [
            0x01, 0x02, 0x03, 0x00, 0x04, 0x11, 0x05, 0x12, 0x21, 0x31, 0x41, 0x06,
            0x13, 0x51, 0x61, 0x07, 0x22, 0x71, 0x14, 0x32, 0x81, 0x91, 0xa1, 0x08,
            0x23, 0x42, 0xb1, 0xc1, 0x15, 0x52, 0xd1, 0xf0, 0x24, 0x33, 0x62, 0x72,
            0x82, 0x09, 0x0a, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x25, 0x26, 0x27, 0x28,
            0x29, 0x2a, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x43, 0x44, 0x45,
            0x46, 0x47, 0x48, 0x49, 0x4a, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59,
            0x5a, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x73, 0x74, 0x75,
            0x76, 0x77, 0x78, 0x79, 0x7a, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89,
            0x8a, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0xa2, 0xa3,
            0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6,
            0xb7, 0xb8, 0xb9, 0xba, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9,
            0xca, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xe1, 0xe2,
            0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xf1, 0xf2, 0xf3, 0xf4,
            0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa
        ];

        var CODESDCC = [0, 0, 3, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0];
        var SYMBOLSDCC = [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11];
        var CODESACC = [0, 0, 2, 1, 2, 4, 4, 3, 4, 7, 5, 4, 4, 0, 1, 2, 0x77];
        var SYMBOLSACC = [
            0x00, 0x01, 0x02, 0x03, 0x11, 0x04, 0x05, 0x21, 0x31, 0x06, 0x12, 0x41,
            0x51, 0x07, 0x61, 0x71, 0x13, 0x22, 0x32, 0x81, 0x08, 0x14, 0x42, 0x91,
            0xa1, 0xb1, 0xc1, 0x09, 0x23, 0x33, 0x52, 0xf0, 0x15, 0x62, 0x72, 0xd1,
            0x0a, 0x16, 0x24, 0x34, 0xe1, 0x25, 0xf1, 0x17, 0x18, 0x19, 0x1a, 0x26,
            0x27, 0x28, 0x29, 0x2a, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x43, 0x44,
            0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
            0x59, 0x5a, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x73, 0x74,
            0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
            0x88, 0x89, 0x8a, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a,
            0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xb2, 0xb3, 0xb4,
            0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
            0xc8, 0xc9, 0xca, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda,
            0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xf2, 0xf3, 0xf4,
            0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa
        ];

        var TL = new Array(), TC = new Array(), FDL = new Array(), FDC = new Array();
        var DU = new Array();

        var DCHTL = new Array(256), DCHTC = new Array(256), ACHTL = new Array(256);
        var ACHTC = new Array(256);

        var codes = new Array(65536), items = new Array(65536);

        var buffer = 0, pointer = 7;

        function generateQTables(qq) {
            for (var i = 0; i < 64; i++) {
                var j = parseInt((QL[i] * qq + 50) / 100);
                TL[ZIGZAG[i]] = j < 1 ? 1 : j > 255 ? 255 : j;
            }
            for (var i = 0; i < 64; i++) {
                var j = parseInt((QC[i] * qq + 50) / 100);
                TC[ZIGZAG[i]] = j < 1 ? 1 : j > 255 ? 255 : j;
            }
            var multipliyer = [
                1.0, 1.387039845, 1.306562965, 1.175875602, 1.0, 0.785694958,
                0.541196100, 0.275899379
            ];
            var k = 0;
            for (var row = 0; row < 8; row++) {
                for (var col = 0; col < 8; col++) {
                    FDL[k] = (1.0 / (TL[ZIGZAG[k]] * multipliyer[row] * multipliyer[col] * 8.0));
                    FDC[k] = (1.0 / (TC[ZIGZAG[k]] * multipliyer[row] * multipliyer[col] * 8.0));
                    k++;
                }
            }
        }
        function generateHuffmanMapping(huffman, codes, lookup) {
            var value = 0, index = 0, t, max = 1;
            for (var i = 1; i <= 16; i++) {
                for (var j = 1; j <= codes[i]; j++) {
                    t = lookup[index];
                    huffman[t] = (value << 16) | i;
                    index++;
                    value++;
                    max = Math.max(max, value);
                }
                value *= 2;
            }
        }
        function generateHuffmanItems() {
            var low = 1;
            var high = 2;
            for (var i = 1; i <= 15; i++) {
                for (var j = low; j < high; j++) {
                    items[32767 + j] = i;
                    codes[32767 + j] = (j << 16) | i;
                }
                for (var nrneg = -(high - 1); nrneg <= -low; nrneg++) {
                    items[32767 + nrneg] = i;
                    codes[32767 + nrneg] = (high - 1 + nrneg) << 16 | i;
                }
                low <<= 1;
                high <<= 1;
            }
        }
        function putHuffBits(bs) {
            var value = bs >> 16;
            var pp = (bs & 0xffff) - 1;
            while (pp >= 0) {
                if ((value & (1 << pp)) !== 0) {
                    buffer |= (1 << pointer);
                }
                pp--;
                pointer--;
                if (pointer < 0) {
                    if (buffer === 0xFF) {
                        putByte(0xFF);
                        putByte(0);
                    } else {
                        putByte(buffer);
                    }
                    pointer = 7;
                    buffer = 0;
                }
            }
        }
        function putByte(v) {
            stream[bp++] = v;
        }
        function putChar(v) {
            putByte((v >> 8) & 0xFF);
            putByte(v & 0xFF);
        }
        function forwardDCT(inp) {
            var d0, d1, d2, d3, d4, d5, d6, d7, t0, t1, t2, t3, t4, t5, t6, t7, t10, t11, t12, t13;
            var p;
            for (p = 0; p < 64; p += 8) {
                d0 = inp[p];
                d1 = inp[p + 1];
                d2 = inp[p + 2];
                d3 = inp[p + 3];
                d4 = inp[p + 4];
                d5 = inp[p + 5];
                d6 = inp[p + 6];
                d7 = inp[p + 7];

                t0 = d0 + d7;
                t7 = d0 - d7;
                t1 = d1 + d6;
                t6 = d1 - d6;
                t2 = d2 + d5;
                t5 = d2 - d5;
                t3 = d3 + d4;
                t4 = d3 - d4;

                t10 = t0 + t3;
                t13 = t0 - t3;
                t11 = t1 + t2;
                t12 = t1 - t2;

                inp[p] = t10 + t11;
                inp[p + 4] = t10 - t11;
                d1 = ((t12 + t13) * 2896) / 4096;
                inp[p + 2] = t13 + d1;
                inp[p + 6] = t13 - d1;
                t10 = t4 + t5;
                t12 = t6 + t7;

                d5 = ((t10 - t12) * 1567) / 4096;
                d2 = d5 + (t10 * 2216) / 4096;
                d4 = d5 + (t12 * 5351) / 4096;
                d3 = ((t5 + t6) * 2896) / 4096;
                d6 = t7 + d3;
                d7 = t7 - d3;
                inp[p + 5] = d7 + d2;
                inp[p + 3] = d7 - d2;
                inp[p + 1] = d6 + d4;
                inp[p + 7] = d6 - d4;
            }
            for (p = 0; p < 8; p++) {
                d0 = inp[p];
                d1 = inp[p + 8];
                d2 = inp[p + 16];
                d3 = inp[p + 24];
                d4 = inp[p + 32];
                d5 = inp[p + 40];
                d6 = inp[p + 48];
                d7 = inp[p + 56];

                t0 = d0 + d7;
                t7 = d0 - d7;
                t1 = d1 + d6;
                t6 = d1 - d6;
                t2 = d2 + d5;
                t5 = d2 - d5;
                t3 = d3 + d4;
                t4 = d3 - d4;

                t10 = t0 + t3;
                t13 = t0 - t3;
                t11 = t1 + t2;
                t12 = t1 - t2;

                inp[p] = t10 + t11;
                inp[p + 32] = t10 - t11;
                d1 = ((t12 + t13) * 2896) / 4096;
                t10 = t4 + t5;
                t12 = t6 + t7;
                d5 = ((t10 - t12) * 1567) / 4096;
                d2 = d5 + (t10 * 2216) / 4096;
                d4 = d5 + (t12 * 5351) / 4096;
                d3 = ((t5 + t6) * 2896) / 4096;
                d6 = t7 + d3;
                d7 = t7 - d3;

                inp[p + 16] = t13 + d1;
                inp[p + 48] = t13 - d1;
                inp[p + 40] = d7 + d2;
                inp[p + 24] = d7 - d2;
                inp[p + 8] = d6 + d4;
                inp[p + 56] = d6 - d4;
            }
        }
        function writeAPP0() {
            putChar(0xFFE0);
            putChar(16);
            putByte(0x4A);
            putByte(0x46);
            putByte(0x49);
            putByte(0x46);
            putByte(0);
            putByte(1);
            putByte(1);
            putByte(0);
            putChar(1);
            putChar(1);
            putByte(0);
            putByte(0);
        }
        function writeSOF0(width, height) {
            putChar(0xFFC0);
            putChar(17);   // length, truecolor YUV JPG
            putByte(8);    // precision
            putChar(height);
            putChar(width);
            putByte(3);    // nrofcomponents
            putByte(1);
            putByte(0x11);
            putByte(0);
            putByte(2);
            putByte(0x11);
            putByte(1);
            putByte(3);
            putByte(0x11);
            putByte(1);
        }
        function writeSOS() {
            putChar(0xFFDA); // marker
            putChar(12); // length
            putByte(3); // nrofcomponents
            putByte(1); // IdY
            putByte(0); // HTY
            putByte(2); // IdU
            putByte(0x11); // HTU
            putByte(3); // IdV
            putByte(0x11); // HTV
            putByte(0); // Ss
            putByte(0x3f); // Se
            putByte(0); // Bf
        }
        function writeDQT() {
            putChar(0xFFDB);
            putChar(132);
            putByte(0);
            for (var i = 0; i < 64; i++) {
                putByte(TL[i]);
            }
            putByte(1);
            for (var j = 0; j < 64; j++) {
                putByte(TC[j]);
            }
        }
        function writeDHT() {
            putChar(0xFFC4);
            putChar(0x01A2);

            putByte(0);
            var i;
            for (i = 0; i < 16; i++) {
                putByte(CODESDCL[i + 1]);
            }
            for (i = 0; i <= 11; i++) {
                putByte(SYMBOLSDCL[i]);
            }
            putByte(0x10);
            for (i = 0; i < 16; i++) {
                putByte(CODESACL[i + 1]);
            }
            for (i = 0; i <= 161; i++) {
                putByte(SYMBOLSACL[i]);
            }
            putByte(1);
            for (i = 0; i < 16; i++) {
                putByte(CODESDCC[i + 1]);
            }
            for (i = 0; i <= 11; i++) {
                putByte(SYMBOLSDCC[i]);
            }
            putByte(0x11);
            for (var o = 0; o < 16; o++) {
                putByte(CODESACC[o + 1]);
            }
            for (var p = 0; p <= 161; p++) {
                putByte(SYMBOLSACC[p]);
            }
        }
        function compress(CDU, fdtbl, DC, huffDC, huffAC) {
            var EOB = huffAC[0];
            var M16zeroes = huffAC[0xF0];
            var pos;
            var fq;
            forwardDCT(CDU);
            //quantization step
            for (var j = 0; j < 64; ++j) {
                fq = CDU[j] * fdtbl[j];
                DU[ZIGZAG[j]] = ((fq > 0.0) ? (fq + 0.5) : (fq - 0.5)) | 0;
            }
            var Diff = DU[0] - DC;
            DC = DU[0];
            if (Diff === 0) {
                putHuffBits(huffDC[0]);
            } else {
                pos = 32767 + Diff;
                putHuffBits(huffDC[items[pos]]);
                putHuffBits(codes[pos]);
            }
            var end0pos = 63;
            while ((end0pos > 0) && (DU[end0pos] === 0)) {
                end0pos--;
            }
            if (end0pos === 0) {
                putHuffBits(EOB);
                return DC;
            }
            var i = 1;
            var lng;
            while (i <= end0pos) {
                var startpos = i;
                for (; (DU[i] === 0) && (i <= end0pos); ++i) {
                }
                var nrzeroes = i - startpos;
                if (nrzeroes >= 16) {
                    lng = nrzeroes >> 4;
                    for (var nrmarker = 1; nrmarker <= lng; ++nrmarker) {
                        putHuffBits(M16zeroes);
                    }
                    nrzeroes &= 0xF;
                }
                pos = 32767 + DU[i];
                putHuffBits(huffAC[(nrzeroes << 4) + items[pos]]);
                putHuffBits(codes[pos]);
                i++;
            }
            if (end0pos != 63) {
                putHuffBits(EOB);
            }
            return DC;
        }
        function encode(data, width, height) {
            stream = new Array();
            bp = 0;

            var qq = quality < 50 ? 5000 / quality : (200 - (quality << 1));
            generateQTables(qq);

            generateHuffmanMapping(DCHTL, CODESDCL, SYMBOLSDCL);
            generateHuffmanMapping(DCHTC, CODESDCC, SYMBOLSDCC);
            generateHuffmanMapping(ACHTL, CODESACL, SYMBOLSACL);
            generateHuffmanMapping(ACHTC, CODESACC, SYMBOLSACC);

            generateHuffmanItems();

            buffer = 0;
            pointer = 7;

            putChar(0xFFD8); // SOI
            writeAPP0();
            writeDQT();
            writeSOF0(width, height);
            writeDHT();
            writeSOS();

            var DCY = 0, DCU = 0, DCV = 0;

            buffer = 0;
            pointer = 7;

            var unitY = new Array(64);
            var unitU = new Array(64);
            var unitV = new Array(64);

            var x, y = 0, r, g, b;
            var start, p, col, row, pos;

            var maxLen = width * height;
            var pix;

            while (y < height) {
                x = 0;
                while (x < width) {
                    start = width * y + x;
                    for (pos = 0; pos < 64; pos++) {
                        row = pos >> 3;
                        col = pos & 7;
                        p = start + (row * width) + col;

                        if (y + row >= height) {
                            p -= (width * (y + 1 + row - height));
                        }
                        if (x + col >= width) {
                            p -= ((x + col) - width + 4);
                        }
                        if (p > maxLen || p < 0) {
                            continue;
                        }
                        pix = p * 4;
                        r = data[pix];
                        g = data[pix + 1];
                        b = data[pix + 2];

                        unitY[pos] = ((128 + 76 * r + 150 * g + 29 * b) >> 8) - 128;
                        unitU[pos] = (128 + 127 * b - 84 * g - 43 * r) >> 8;
                        unitV[pos] = (128 + 127 * r - 106 * g - 21 * b) >> 8;
                    }
                    DCY = compress(unitY, FDL, DCY, DCHTL, ACHTL);
                    DCU = compress(unitU, FDC, DCU, DCHTC, ACHTC);
                    DCV = compress(unitV, FDC, DCV, DCHTC, ACHTC);
                    x += 8;
                }
                y += 8;
            }

            if (pointer >= 0) {
                var n = pointer + 1;
                var m = (1 << (pointer + 1)) - 1;
                putHuffBits((m << 16) | n);
            }
            putChar(0xFFD9);
            return stream;
        }
        return encode(imageData, iw, ih);
    };

}

