/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 2009 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and may be covered by U.S. and Foreign Patents,
* patents in process, and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
package coldfusion.mail;

import coldfusion.runtime.ApplicationException;
import coldfusion.runtime.QueryFunction;
import coldfusion.runtime.Struct;
import coldfusion.sql.QueryTable;
import coldfusion.sql.Table;
import coldfusion.util.CFDumpable;

import com.sun.mail.imap.IMAPFolder;

import java.util.ArrayList;

import javax.mail.Flags;
import javax.mail.Folder;
import javax.mail.FolderNotFoundException;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Provider;
import javax.mail.Session;
import javax.mail.Store;
import javax.mail.search.MessageNumberTerm;
import javax.mail.search.OrTerm;
import javax.mail.search.SearchTerm;

public class IMapImpl  implements CFDumpable
{
    protected HostImpl source;
    protected Session session = null;

    protected String username = null;
    protected String password = null;

    protected Store store = null;
    protected Folder folder = null;
    private String accept = null;

    private String[] HEADER_FIELDS = {"sentdate","rxddate","size","lines","answered","deleted","draft","flagged","recent","seen", "from", "messagenumber", "replyto", "subject", "cc", "to", "header", "messageid", "uid"};
    private String[] BODY_FIELDS  = {"body", "textbody", "htmlbody", "sentdate","rxddate","size","lines","answered","deleted","draft","flagged","recent","seen", "from", "messagenumber", "replyto", "subject", "cc", "to", "header", "attachments", "attachmentfiles", "cids","messageid", "uid"};
    protected String[] fields = HEADER_FIELDS;

    protected String attach_path = null;
    protected boolean unique_filenames = false;

    protected boolean debug = false;
    private String[] uids = null;
    private String[] msgnums = null;
    private int startmsg = 1;
    private int maxrows = 999999;
    private String inbox_folder = "INBOX";
    private boolean useSSL = false;
    private SearchTerm searchTerm = null;
    static final String TITLE_FULLNAME = "FULLNAME";
    static final String TITLE_NAME = "NAME";
    static final String TITLE_TOTALMESSAGES = "TOTALMESSAGES";
    static final String TITLE_UNREAD = "UNREAD";
    static final String TITLE_NEW = "NEW";
    static final String columnNamesforlistDir[] =
{
        TITLE_FULLNAME,
        TITLE_NAME,
        TITLE_TOTALMESSAGES,
        TITLE_UNREAD,
        TITLE_NEW
};
    static final String columnTypesforlistDir[] =
{
        "CF_SQL_VARCHAR", //FULLNAME
        "CF_SQL_VARCHAR", //NAME
        "CF_SQL_VARCHAR", //TOTALMESSAGES
        "CF_SQL_VARCHAR", //UNREAD
        "CF_SQL_VARCHAR", //NEW
};

    /**
     * 
     * @param o
     */
    public void setAccept(String o)
    {
        accept = o;
    }

    /**
     * 
     * @return
     */
    public String getAccept()
    {
        return accept;
    }

    /**
     * @return the searchTerm
     */
    public SearchTerm getSearchTerm()
    {
        return searchTerm;
    }

    /**
     * @param searchTerm
     *            the searchTerm to set
     */
    public void setSearchTerm(SearchTerm searchTerm)
    {
        this.searchTerm = searchTerm;
    }

    public Store getStore()
    {
        return store;
    }

    public void setStore(Store store)
    {
        this.store = store;
    }

    public Folder getFolder()
    {
        return folder;
    }

    public void setFolder(Folder folder)
    {
        this.folder = folder;
    }

    public void setUseSSL(boolean useSSL)
    {
        this.useSSL = useSSL;
    }

    /**
     * Set a mail source used for retrieving mail
     * @param s
     */
    public void setMailSource(HostImpl s)
    {
        source = s;
    }

    public void setUsername(String s)
    {
        username = s;
    }

    public void setPassword(String p)
    {
        password = p;
    }

    public void setPath(String p)
    {
        attach_path = p;
    }

    public void setUniqueFilenames(boolean b)
    {
        unique_filenames = b;
    }

    public void setDebug(boolean d)
    {
        debug = d;
    }

    public void setUids(String[] uids)
    {
        this.uids = uids;
    }

    public void setMsgs(String[] m)
    {
        msgnums = m;
    }

    public void setStartMsg(int start)
    {
        startmsg = start;
    }

    public void setMaxRows(int max)
    {
        maxrows = max;
    }

    public void setBody(boolean b)
    {
        if (b)
            fields = BODY_FIELDS;
        else
            fields = HEADER_FIELDS;
    }

    public String getLabel()
    {
        return "IMAP Connection";
    }

    public Object getMetadata()
    {
        Struct temp = new Struct();
        temp.put("SERVER",source.getHost());
        temp.put("PORT",source.getPort());
        temp.put("SSL",useSSL);
        temp.put("CLOSED",folder==null);
        return temp;
    }

    protected void finalize() throws Throwable
    {
        clear(); // to ensure in case of uncaught exceptions or when explicit close is not called
        source=null;
        super.finalize();
    }

    public void clearOperationVar()
    {
        attach_path = null;
        unique_filenames = false;
        debug = false;
        startmsg = 1;
        maxrows = 999999;
        msgnums = null;
        uids = null;

    }

    /**
     * Reset member elements to null, close any open folder or store.
     */
    public void clear()
    {
//        source = null;
        session = null;

        username = null;
        password = null;

        fields = HEADER_FIELDS;

        attach_path = null;
        unique_filenames = false;
        debug = false;
        startmsg = 1;
        maxrows = 999999;
        msgnums = null;
        uids = null;

        try
        {
            folder.close(false);
        }
        catch (Exception ex)
        {
        }
        finally
        {
            folder = null;
        }

        try
        {
            store.close();
        }
        catch (Exception ex)
        {
        }
        finally
        {
            store = null;
        }
    }


    /**
     * Allocates a JavaMail session object for the class
     * and connects to the mail store.
     *
     * @throws MessagingException for any errors
     */
    public void validate_folder() throws MessagingException
    {
        if (session == null)
        {
            try
            {
                session = source.getSession();
            }
            catch (Exception ex)
            {
                throw new MessagingException(ex.getMessage());
            }
        }

        session.setDebug(debug);

        if(useSSL)
        {
            try
            {
                session.getProvider("imaps");
            } catch (Exception ex)
            {
                session.addProvider(new Provider(session.getProvider("imap").getType(), "imaps", "com.sun.mail.imap.IMAPSSLStore", "Sun Microsystems, Inc", null));
            }
            store = session.getStore("imaps");
        }
        else
        {
            store = session.getStore("imap");
        }
        String host = source.getHost();
        int port = source.getPort();

        if (port == -1)
        {
            store.connect(host, username, password);
        }
        else
        {
            store.connect(host, port, username, password);
        }

        if ((folder = store.getDefaultFolder()) == null)
        {
            throw new FolderNotFoundException();
        }
    }


    /**
     * Performs the GET and GETALL operations on messages
     * specified by the previous executed statements.
     *
     * @return a SQL table of messages
     * @throws MessagingException
     */
    public Table getMails(String strFolder) throws MessagingException
    {
        if(strFolder==null)
            strFolder=inbox_folder;
        Folder tempFolder=null;
        try
        {
            tempFolder = folder.getFolder(strFolder);
            if (tempFolder == null)
            {
                throw new FolderNotFoundException(strFolder, null);
            }

            tempFolder.open(Folder.READ_ONLY);
            Message[] msgs = getMessages(true, tempFolder);

            // put messages into a query table
            EmailTable table = new EmailTable();
            table.populate(tempFolder, fields, msgs, attach_path, unique_filenames, accept);
            return table;
        }
        finally
        {
            if (tempFolder != null)
            {
                tempFolder.close(false);
            }
        }
    }

    /**
     * List all folders in the user account
     *
     * @return a SQL table of messages
     * @throws MessagingException
     */
    public Table listAllFolders(String strFolder, boolean recurse) throws MessagingException
    {
        // todo: to we need to take care of sub folders
        QueryTable resultTable = new QueryTable(0, columnNamesforlistDir, columnTypesforlistDir);
        Folder tempFolder = folder;
        if(strFolder != null)
            tempFolder = folder.getFolder(strFolder);
        if (tempFolder == null)
        {
            throw new FolderNotFoundException(strFolder, null);
        }
        listFolders(tempFolder, resultTable, recurse);
        return resultTable;
    }

    private void listFolders(Folder tempFolder, QueryTable resultTable, boolean recurse) throws MessagingException
    {
        Folder[] folderList = tempFolder.list();
        if (folderList == null || folderList.length == 0)
        {
            return;
        }
        for (Folder aFolderList : folderList)
        {
            try
            {
                String name = aFolderList.getName();
                String fullName = aFolderList.getFullName();
                int newMessageCount = aFolderList.getNewMessageCount();
                int messageCount = aFolderList.getMessageCount();
                int unreadMessageCount = aFolderList.getUnreadMessageCount();
                // there can be issues with getting the above values on invalid folders. ignore them
                QueryFunction.QueryAddRow(resultTable);
                QueryFunction.QuerySetCell(resultTable, TITLE_UNREAD, unreadMessageCount);
                QueryFunction.QuerySetCell(resultTable, TITLE_TOTALMESSAGES, messageCount);
                QueryFunction.QuerySetCell(resultTable, TITLE_NEW, newMessageCount);
                QueryFunction.QuerySetCell(resultTable, TITLE_FULLNAME, fullName);
                QueryFunction.QuerySetCell(resultTable, TITLE_NAME, name);
                if(recurse)
                    listFolders(aFolderList, resultTable,recurse);

            } catch (Exception e)
            {
              //absorb it
            }

        }
    }

    /**
     * Deletes a folder
     *
     * @throws MessagingException
     */
    public void deleteFolder(String strFolder) throws MessagingException
    {
        String restrictedFolder = "INBOX, OUTBOX";
        if (restrictedFolder.indexOf(strFolder.toUpperCase())!=-1)
            throw new DeleteFolderExistsException(strFolder);
        Folder tempFolder = folder.getFolder(strFolder);
        if (tempFolder == null)
        {
            throw new FolderNotFoundException(strFolder, null);
        }
        tempFolder.delete(true);
    }

    /**
     * Rename a folder
     *
     * @throws MessagingException
     */
    public void renameFolder(String strFolder, String strNewFolder) throws MessagingException
    {
        // todo: to we need to take care of sub folders
        String restrictedFolder = "INBOX, OUTBOX";
        if (restrictedFolder.indexOf(strFolder.toUpperCase())!=-1 ||
                restrictedFolder.indexOf(strNewFolder.toUpperCase())!=-1)
            throw new RenameFolderDeleteException(strFolder,strNewFolder);
        Folder tempFolder = folder.getFolder(strFolder);
        if (tempFolder == null)
        {
            throw new FolderNotFoundException(strFolder, null);
        }
        IMAPFolder newFolder = (IMAPFolder) store.getFolder(strNewFolder);
        if (newFolder.exists())
            throw new RenameFolderExistsException(strFolder, strNewFolder);
        //newFolder.create(Folder.HOLDS_MESSAGES);
        boolean success = tempFolder.renameTo(newFolder);
        if(!success)
        {
            // need to delete new folder, bring back all messages and throw an exception
            Message[] msgs = getMessages(false, tempFolder);
            newFolder.copyMessages(msgs, tempFolder);
            newFolder.delete(true);
            throw new RenameFolderDeleteException(strFolder,strNewFolder);
        }
    }

    /**
     * create a new folder
     *
     * @throws MessagingException
     */
    public void createFolder(String strFolder) throws MessagingException
    {
        IMAPFolder newFolder = (IMAPFolder) store.getFolder(strFolder);
        if (newFolder.exists())
            throw new CreateFolderExistsException(strFolder);
        newFolder.create(Folder.HOLDS_MESSAGES);
    }

    /**
     * Deletes messages as specified by the previous executed statements
     *
     * @throws MessagingException
     */
    public void moveMails(String strFolder,String strNewFolder) throws MessagingException
    {
        // todo: to we need to take care of sub folders
        IMAPFolder toFolder = null;
        if(strFolder==null)
            strFolder=inbox_folder;
        Folder tempFolder=null;
        try
        {
            toFolder=(IMAPFolder)folder.getFolder(strNewFolder);
            tempFolder = folder.getFolder(strFolder);
            if (tempFolder == null)
            {
                throw new FolderNotFoundException(strFolder, null);
            }
            if (toFolder == null)
            {
                throw new FolderNotFoundException(strNewFolder, null);
            }

            tempFolder.open(Folder.READ_WRITE);
            toFolder.open(Folder.READ_WRITE);
            Message[] msgs = getMessages(false, tempFolder);
            tempFolder.copyMessages(msgs, toFolder);
            tempFolder.setFlags( msgs, new Flags(Flags.Flag.DELETED), true);
        }
        finally
        {
            if (tempFolder != null)
            {
                tempFolder.close(true);
            }
            if (toFolder != null)
            {
                toFolder.close(true);
            }
        }
    }

    /**
     * Mark messages seen as specified by the previous executed statements
     *
     * @throws MessagingException
     */
    public void markRead(String strFolder) throws MessagingException
    {
        // CW: maybe it's a good idea to return the update count...
        if(strFolder==null)
            strFolder=inbox_folder;
        Folder tempFolder=null;
        try
        {
            tempFolder = folder.getFolder(strFolder);
            if (tempFolder == null)
            {
                throw new FolderNotFoundException(strFolder, null);
            }

            tempFolder.open(Folder.READ_WRITE);
            Message[] msgs = getMessages(false, tempFolder);
            tempFolder.setFlags( msgs, new Flags(Flags.Flag.SEEN), true);
        }
        finally
        {
            if (tempFolder != null)
            {
                tempFolder.close(true);
            }
        }
    }


    /**
     * Deletes messages as specified by the previous executed statements
     *
     * @throws MessagingException
     */
    public void deleteMails(String strFolder) throws MessagingException
    {
        // CW: maybe it's a good idea to return the update count...
        if(strFolder==null)
            strFolder=inbox_folder;
        Folder tempFolder=null;
        try
        {
            tempFolder = folder.getFolder(strFolder);
            if (tempFolder == null)
            {
                throw new FolderNotFoundException(strFolder, null);
            }

            tempFolder.open(Folder.READ_WRITE);
            Message[] msgs = getMessages(false, tempFolder);
            tempFolder.setFlags( msgs, new Flags(Flags.Flag.DELETED), true);
        }
        finally
        {
            if (tempFolder != null)
            {
                tempFolder.close(true);
            }
        }
    }

    /**
     * Return an array of messages that we should process
     *
     * @throws MessagingException
     */
    private Message[] getMessages(boolean isGet, Folder folder) throws MessagingException
    {
        Message[] msgs;
        int count = folder.getMessageCount();

        if (count == 0)
            return new Message[0];

        // if we have message numbers, use them
        if (msgnums != null)
        {
            // populate message number search Term
            SearchTerm searchTerm = null;
            for (int i = 0; i < msgnums.length; i++)
            {
                try
                {
                    Integer num = Integer.valueOf(msgnums[i]);
                    if (num.intValue() > 0 && num.intValue() <= count)
                    {
                        if (searchTerm == null)
                        {
                            searchTerm = new MessageNumberTerm(num);
                        }
                        else
                        {
                            searchTerm = new OrTerm(searchTerm, new MessageNumberTerm(num));
                        }
                    }
                }
                catch (NumberFormatException e)
                {
                    // ignore, and skip this value
                }
            }

            // get specified messages
            msgs = folder.search(searchTerm);

        }
        // If we have a list of UIDs, use them, For IMapFolder object getMessagesByUID is faster than older approach.
        else if (uids != null)
        {
            ArrayList<Long> uidsArrayList = new ArrayList<Long>();
            for (int i = 0; i < uids.length; i++)
            {
                try
                {
                    Long num = Long.valueOf(uids[i]);
                    long longValue = num.longValue();
                    if (longValue > 0)
                    {
                        uidsArrayList.add(longValue);
                    }
                }
                catch (NumberFormatException e)
                {
                    // ignore, and skip this value
                }
            }

            long[] uidsArray = new long[uidsArrayList.size()];
            int i = 0;
            for (Long long1 : uidsArrayList)
            {
                uidsArray[i++] = long1;
            }
            IMAPFolder iFolder = (IMAPFolder) folder;
            msgs = iFolder.getMessagesByUID(uidsArray);
        }
        else if (searchTerm != null)
        {
            // get specified messages
            msgs = folder.search(searchTerm);
        }
        else // use limits if no uids either
        {
            if (startmsg > count)
                throw new ArrayIndexOutOfBoundsException(Integer.toString(count));
            int end = startmsg + (maxrows - 1);
            if (end > count) end = count;
            msgs = folder.getMessages(startmsg, end);
        }
        return msgs;
    }

    /**
	 * Folder {oldFolder} could not be renamed to {newFolder}.
     * {oldFolder} exists. Try using a different name.
	 */
	public class RenameFolderExistsException extends ApplicationException {
        private static final long serialVersionUID = 1L;
        public String oldFolder;
        public String newFolder;
	    RenameFolderExistsException(String oldFolder, String newFolder) {
	        super();
            this.oldFolder = oldFolder;
            this.newFolder = newFolder;
	    }
	}

    /**
     * Operation failed: {oldFolder} cannot be deleted or renamed to {newFolder}.
	 */
	public class RenameFolderDeleteException extends ApplicationException {
        private static final long serialVersionUID = 1L;
        public String oldFolder;
        public String newFolder;
	    RenameFolderDeleteException(String oldFolder, String newFolder) {
	        super();
            this.oldFolder = oldFolder;
            this.newFolder = newFolder;
	    }
	}

    /**
	 * Folder {oldFolder} could not be created.
     * {oldFolder} already exists. Delete and retry.
	 */
	public class CreateFolderExistsException extends ApplicationException {
        private static final long serialVersionUID = 1L;
        public String oldFolder;
	    CreateFolderExistsException(String oldFolder) {
	        super();
            this.oldFolder = oldFolder;
	    }
	}

     /**
	 * Folder {oldFolder} could not be deleted.
     * Check if it is read-only.
	 */
	public class DeleteFolderExistsException extends ApplicationException {
        private static final long serialVersionUID = 1L;
        public String oldFolder;
	    DeleteFolderExistsException(String oldFolder) {
	        super();
            this.oldFolder = oldFolder;
	    }
	}

}