/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2008 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package coldfusion.tagext.mail;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.security.Permission;
import java.util.Arrays;
import java.util.List;

import javax.activation.CommandMap;
import javax.activation.MailcapCommandMap;
import javax.mail.internet.InternetAddress;
import javax.mail.search.SearchTerm;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.tagext.BodyTag;

import coldfusion.Version;
import coldfusion.log.Logger;
import coldfusion.mail.HostImpl;
import coldfusion.mail.MailImpl;
import coldfusion.mail.core.MailAttachmentException;
import coldfusion.mail.core.MailExceptions.InvalidCharacterEncodingException;
import coldfusion.mail.core.MailExceptions.InvalidIDNAVersionException;
import coldfusion.mail.core.MailExceptions.InvalidSignAttributesException;
import coldfusion.mail.core.MailExceptions.KeyStoreNotFoundException;
import coldfusion.mail.core.MailExceptions.MultipartException;
import coldfusion.mail.core.MailHeaderException;
import coldfusion.mail.core.MailSessionException;
import coldfusion.monitor.beans.TagAttribute;
import coldfusion.runtime.CFOutput;
import coldfusion.runtime.RequestMonitor;
import coldfusion.runtime.RequestTimedOutException;
import coldfusion.runtime.RuntimeServiceImpl;
import coldfusion.server.LicenseService;
import coldfusion.server.MailSpoolService;
import coldfusion.server.ServiceFactory;
import coldfusion.tagext.GenericTagPermission;
import coldfusion.tagext.InvalidTagAttributeException;
import coldfusion.tagext.OutputException;
import coldfusion.tagext.ResourceNotFoundException;
import coldfusion.tagext.io.OutputTag;
import coldfusion.util.Utils;



/**
 * Sends e-mail messages by an SMTP server.
 * @tag mail
 *
 * @tagbody JSP
 * @author Clement Wong
 */
public class MailTag extends OutputTag implements BodyTag
{
    public static final String DEFAULT_MAILER_ID = "ColdFusion " + Version.getMajor() + " Application Server";

	private static final GenericTagPermission tp = new GenericTagPermission("cfmail");

    protected static final int INVALID_PORT = -1;
    protected static final int DEF_PORT = 25;
    protected static final int DEF_TIMEOUT = -1;
    protected static final String DEF_PRIORITY = "3";

    protected String host = null;
    protected int port = INVALID_PORT;
    protected String sender = null;
    protected String recipient = null;
    protected String cc = null;
    protected String bcc = null;
    protected String replyTo = null;
    protected String failTo = null;
    protected String subject = null;
    protected int timeout = DEF_TIMEOUT;
    protected String type;
    protected String attach;
    protected String mailerid = DEFAULT_MAILER_ID;
    protected String priority = DEF_PRIORITY;
    protected boolean debug = false;
    protected Boolean spoolEnable = null;
    protected boolean allowFailover = false;
    protected int wrapText = -1;
    protected String username = null;
    protected String password = null;
    protected String charset = null;
    private boolean containsOutputTag = false;
    private long startTime = 0;
    private boolean useTLS = false;
    private boolean useSSL = false;
    private Boolean removeAttachment= null;
    private Boolean sign = null;
    private Boolean encrypt = null;
    private String recipientCert = null;
    private String encryptionAlgorithm = null;
    private String keystore = null;
    private String keyalias = null;
    private String keystorePassword = null;
    private String keyPassword = null;
    
    public static final int DEFAULT_IDNA_VERSION = Integer.valueOf(System.getProperty("coldfusion.email.idn.version", "2008"));
    private int idnaVersion = DEFAULT_IDNA_VERSION;
    
    public Boolean getRemoveAttachment()
    {
        return removeAttachment;
    }

    public boolean isRemoveAttachment()
    {
        return removeAttachment!= null && removeAttachment.booleanValue();
    }
    /**
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setRemove(boolean b)
    {
        this.removeAttachment = Boolean.valueOf(b);
    }

    protected coldfusion.mail.MailImpl impl = null;

    /**
     * Clear class variables before tag is reused.
     * YOU MUST KEEP THIS IN SYNC WITH THE ABOVE LIST
     */
    public void release() {
        cleanup();
        super.release();
    }

    public void doFinally()
    {
        super.doFinally();
        cleanup();
    }

    private void cleanup()
    {
        host = null;
        port = INVALID_PORT;
        sender = null;
        recipient = null;
        cc = null;
        bcc = null;
        replyTo = null;
        failTo = null;
        subject = null;
        query = null;
        timeout = DEF_TIMEOUT;
        username = null;
        password = null;
        charset = null;
        spoolEnable = null;
        wrapText = -1;
        containsOutputTag = false;
        allowFailover = false;

        type = null;
        attach = null;
        mailerid = DEFAULT_MAILER_ID;
        priority = DEF_PRIORITY;
        debug = false;
        startTime = 0;
        useTLS = false;
        useSSL = false;
        removeAttachment=null;
        sign = null;
        encrypt = null;
        recipientCert = null;
        encryptionAlgorithm = null;
        keystore = null;
        keystorePassword = null;
        keyPassword = null;
        keyalias = null;
        impl = null;
        idnaVersion = DEFAULT_IDNA_VERSION;
        //impl.clear();
    }


    protected coldfusion.mail.MailImpl getMailImpl() {
        return impl;
    }

    public void ContainsOutputTag(boolean flag) {
     containsOutputTag = flag;
    }

	protected Permission getPermission()
	{
		return tp;
	}

	/**
	 * Optional.  The hostname or IP address of the SMTP server to use for
	 * sending messages.  Can be specified as a comma serparated list
     * which can include username, password and port information.
     * <P><code>
     * [username:password@]server[:port]
     * </code><P>
     * If no server is specified, the server name specified in the ColdFusion
	 * Administrator is used.
	 *
	 * @tagattribute
	 * @required false
	 * @rtexprvalue true
	 */
    public void setServer(String h) {
        host = h;
    }

    public String getServer() {
        return host;
    }

	/**
	 * The TCP/IP port on which the SMTP server listens for requests. Default is 25.
	 * @tagattribute
	 * @required false
	 * @rtexprvalue true
	 */
    public void setPort(int p) {
        port = p;
    }

    public int getPort() {
        return port;
    }

	/**
	 * Required. The sender of the e-mail message
	 * @tagattribute
	 * @required true
	 * @rtexprvalue true
	 */
    public void setFrom(String o) {
        // I guess this is trying to balance brackets...
        int index = o.lastIndexOf("<");
        if (index != -1) {
            int eIndex = o.lastIndexOf(">");
            if (eIndex == -1 || eIndex < index)
                o = o + ">";
        }
        sender = Utils.stripCRLF(o.replace(';', ','));
    }

    public String getFrom()
    {
        return sender;
    }

	/**
	 * The subject of the mail message.
	 * @tagattribute
	 * @required yes
	 * @rtexprvalue true
	 */
    public void setSubject(String s) {
        subject = Utils.stripCRLF(s.replaceAll("\n+", " ")); // replacing new line with spaces as gmail does.
    }

    public String getSubject() {
        return subject;
    }

	/**
	 * The number of seconds to wait before timing out the connection to the SMTP server.
	 * @tagattribute
	 * @required false
	 * @rtexprvalue true
	 */
    public void setTimeout(int t) {
        timeout = t;
    }

    public int getTimeout() {
        return timeout;
    }

	/**
	 * Required. The name of the e-mail message recipient(s).
	 *
	 * @tagattribute
	 * @required true
	 * @rtexprvalue true
	 */
    public void setTo(String addr) {
        recipient = Utils.stripCRLF(addr.replace(';', ','));
    }

    public String getTo() {
        return recipient;
    }

	/**
	 * Optional. Indicates addresses to copy the e-mail message to; "cc" stands for "carbon copy."
	 * @tagattribute
	 * @required false
	 * @rtexprvalue true
	 */
    public void setCc(String addr) {
        cc = Utils.stripCRLF(addr.replace(';', ','));
    }

    public String getCc() {
        return cc;
    }

	/**
	 * Optional. Indicates addresses to copy the e-mail message to,
	 * without listing them in the message header.  Bcc stands for "blind carbon copy".
	 *
	 * @tagattribute
	 * @required false
	 * @rtexprvalue true
	 */
    public void setBcc(String addr) {
        bcc = Utils.stripCRLF(addr.replace(';', ','));
    }

    public String getBcc() {
        return bcc;
    }

    /**
     * Optional. Indicates addresses to send replys to
     *
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setReplyTo(String addr) {
        replyTo = Utils.stripCRLF(addr.replace(';', ','));
    }

    /**
     * Optional. Indicates addresses to send delivery failure notifications to.
     * <P>
     * This sents the "Envelope" From address, which is different than the From address
     * in the message itself.  See RFC 821.
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setFailTo(String addr) {
        failTo = Utils.stripCRLF(addr);
    }


	/**
	 * Specifies extended type attributes for the message.
     * Special values are PLAIN or TEXT and HTML, otherwise sets the content-type of the message.
	 * @tagattribute
	 * @required false
	 * @rtexprvalue true
	 */
    public void setType(String s) {
      if(s.length() == 0 || s.equalsIgnoreCase("PLAIN") || s.equalsIgnoreCase("TEXT"))
        type = "text/plain";
      else if (s.equalsIgnoreCase("HTML"))
        type = "text/html";
      else
            type = Utils.stripCRLF(s);
    }

    public String getType() {

        return type;
    }
	/**
	 * Specifies the path of the file to be attached to the e-mail message.
	 * An attached file is MIME-encoded
	 * @tagattribute
	 * @required false
	 * @rtexprvalue true
	 */
    public void setMimeattach(String s) {
        attach = s;
    }


    public Object getMimeattach() {
        return attach;
    }
	/**
	 * Specifies a mailer ID to be passed in the X-Mailer SMTP header,
	 * which identifies the mailer application. The default is ColdFusion
	 * Application Server.
	 * @tagattribute
	 * @required false
	 * @rtexprvalue true
	 */
    public void setMailerid(String s) {
        mailerid = Utils.stripCRLF(s);
    }

    public String getMailerid() {
        return mailerid;
    }

    /**
	 * Optional. Indicates priority of the e-mail message.
	 * @tagattribute
	 * @required false
	 * @rtexprvalue true
	 */
    public void setPriority(String pri) {
        priority = pri;
    }

    public String getPriority() {
        return priority;
    }


    /**
     * Should we spool this message to disk or attempt immediate delivery?
     * Tag overrides admin setting
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setSpoolenable(boolean b) {
        spoolEnable = Boolean.valueOf(b);
    }

    /**
     * Should we wrap the mail text at 70 characters.
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setWrapText(int wrapText)
    {
        this.wrapText = wrapText;
    }

    public int getWrapText()
    {
        return wrapText;
    }

    /**
     * Username parameter for SMTP authentication
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setUsername(String u) {
        username = u;
    }

    public String getUsername() {
        return username;
    }

    /**
     * Password parameter for SMTP authentication
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setPassword(String p) {
        password = p;
    }

    /**
     * Use StartTLS (see RFC 2487 and RFC 3501) to
     * switch the connection to be secured by TLS.
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setUseTLS(boolean useTLS)
    {
        this.useTLS = useTLS;
    }

    /**
     * Use SSL to connect to the mail server
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setUseSSL(boolean useSSL)
    {
        this.useSSL = useSSL;
    }

    /**
	 * @tagattribute
	 * @required false
	 * @rtexprvalue true
	 */
    public void setDebug(boolean d) {
        debug = d;
    }

    public boolean isDebug() {
        return debug;
    }

    /**
     * Set the character set of the mail message
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setCharset(String charset)
    {
        this.charset = Utils.stripCRLF(charset);
    }

    public String getCharset()
    {
        return charset;
    }


    public long getElapsedTime()
    {
         return System.currentTimeMillis() - startTime;
    }

    /**
     * Decides whether the mail needs to be signed.
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setSign(boolean sign)
    {
        this.sign = Boolean.valueOf(sign);
    }

    /**
     * Decides whether the mail needs to be encrypted.
     * 
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setEncrypt(boolean encrypt)
    {
        this.encrypt = Boolean.valueOf(encrypt);
    }

    public String getRecipientCert()
    {
        return recipientCert;
    }

    /**
     * Set the encryptionAlgorithm to use for encryption.
     * 
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setEncryptionAlgorithm(String encryptionAlgorithm)
    {
        this.encryptionAlgorithm = encryptionAlgorithm;
    }

    public String getEncryptionAlgorithm()
    {
        return encryptionAlgorithm;
    }

    /**
     * Set the recipientCert location.
     * 
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setRecipientCert(String recipientCertStr)
    {
        if (!new File(recipientCertStr).exists())
            throw new KeyStoreNotFoundException(recipientCertStr);
        this.recipientCert = recipientCertStr;
    }

    public String getKeystore()
    {
        return keystore;
    }

    /**
     * Set the keystore location.
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setKeystore(String keystore)
    {
        if(! new File(keystore).exists())
            throw new KeyStoreNotFoundException(keystore);
        this.keystore = keystore;
    }

    public String getKeyalias()
    {
        return keyalias;
    }

    
    /**
     * Set the alias of the key to be used.
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setKeyalias(String keyalias)
    {
        this.keyalias = keyalias;
    }

    public String getKeystorePassword()
    {
        return keystorePassword;
    }

    /**
     * Set the keystore password
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setKeystorePassword(String keystorePassword)
    {
        this.keystorePassword = keystorePassword;
    }

    public String getKeyPassword()
    {
        return keyPassword;
    }

    /**
     * Set the key password to recover private key
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setKeyPassword(String keyPassword)
    {
        this.keyPassword = keyPassword;
    }

    protected void validate() throws JspException
    {
        MailSpoolService ms = ServiceFactory.getMailSpoolService();
        impl = new coldfusion.mail.MailImpl();

        // Make sure the MailImpl has the charset right away
        if (charset == null) {
            charset = RuntimeServiceImpl.getDefaultMailCharset();
        }
        // Check the charset
        try {
            "hi".getBytes(charset);
        } catch (UnsupportedEncodingException e) {
            throw new InvalidCharacterEncodingException(e.getMessage());
        }

        if (type == null) {
            impl.setContentType("text/plain", charset);
        } else {
            impl.setContentType(type, charset);
        }

        // Set server defaults in the message
        if(port > INVALID_PORT)
          impl.setPort(port);

        if (timeout != DEF_TIMEOUT)
            impl.setTimeout(timeout);

        if (username != null && password != null) {
            impl.setUsername(username);
            impl.setPassword(password);
        }

        // If we have a server, parse it for a possible list.
        // Use tag values as defaults for returned host objects
        if (host != null)
        {
            HostImpl[] hosts = HostImpl.parseServer(host,
                                                    port,
                                                    username,
                                                    password,
                                                    timeout,
                                                    Boolean.valueOf(useTLS),
                                                    Boolean.valueOf(useSSL));
            if (! allowFailover && hosts.length > 1)
            {
                // Only allowed a single server
                hosts = new HostImpl[] { hosts[0] };
                Logger logger = ServiceFactory.getLoggingService().getLogger("coldfusion.mail");
                //logger.warn(RB.getString(MailTag.class, "MailTag.noFailover"));
            }
            impl.setServers(hosts);
            // Only set these attributes if server specified in the tag
            impl.setUseTLS(useTLS);
            impl.setUseSSL(useSSL);
            //impl.validateServers();   Don't need to do this because we pass tag defaults above
        }

        if (subject != null) {
            impl.setSubject(subject);
        }

        if (sender != null) {
            InternetAddress[] fromAddress = MailImpl.setInternetAddress(sender, idnaVersion);
            if(fromAddress != null && sender.trim().length() > 0 ) {
                  impl.setSender(fromAddress[0]);
            } else {
                // Bug 37498 - allow just text (no address) like CF5 did.
                try {
                    InternetAddress addr = new InternetAddress("", sender, charset);
                    impl.setSender(addr);
                } catch (UnsupportedEncodingException e) {
                    // not much we can do about this
                    throw new JspException(e);
                }
            }
        }

        InternetAddress[] rs = null;
        if(recipient !=null)
        {
            rs = MailImpl.setInternetAddress(recipient, idnaVersion);
        }
            
        if(rs != null) {
          impl.setRecipient(rs);
        } else {
              InvalidTagAttributeException e = new InvalidTagAttributeException("CFMAIL", "to", recipient);
              ms.writeToLog(e.getMessage(), true);
              throw e;
        }
        	
        if (cc != null)
            impl.setCc(MailImpl.setInternetAddress(cc, idnaVersion));
        if (bcc != null)
            impl.setBcc(MailImpl.setInternetAddress(bcc, idnaVersion));
        if (replyTo != null)
            impl.setReplyTo(MailImpl.setInternetAddress(replyTo, idnaVersion));
        if (failTo != null) {
            impl.setFailTo(failTo);
        }

        // Only set the spool attribute if it is present in the tag
        if (spoolEnable != null)
            impl.setSpoolEnable(spoolEnable.booleanValue());
        impl.setWrapText(wrapText);
        impl.setUsername(username);
        impl.setPassword(password);

        impl.setDebug(debug);

        //set priority if other than normal
        if (!DEF_PRIORITY.equals(priority))
        {
            //Some mail clients (notably Eudora) only like numeric values 1-5 for the X-Priority
            //header. Outlook, Mozilla and other clients are happy with either, so we convert
            //the standard strings to 1-5 values
            if ("highest".equalsIgnoreCase(priority) || "urgent".equalsIgnoreCase(priority))
                priority = "1";
            else if ("high".equalsIgnoreCase(priority))
                priority = "2";
            else if ("normal".equalsIgnoreCase(priority))
                priority = "3";
            else if ("low".equalsIgnoreCase(priority))
                priority = "4";
            else if ("lowest".equalsIgnoreCase(priority) || "non-urgent".equalsIgnoreCase(priority))
                priority = "5";

            try
            {
                impl.setHeader("X-Priority", priority);
            }
            catch (Exception ex)
            {
                throw new MailHeaderException(ex);
            }
        }
        if(sign != null && sign.booleanValue() == true)
        {
            // If user has specified sign="true" and specified keystore, he must specify keyalias, keystorepassword and keypassword.
            if((keystore != null) && (keystorePassword == null || keystorePassword.length() == 0))
                throw new InvalidSignAttributesException();              
        }   
        
        impl.setSign(sign);
        impl.setEncrypt(encrypt);
        impl.setEncryptionAlgorithm(encryptionAlgorithm);
        impl.setRecipientCert(recipientCert);
        impl.setKeystoreFile(keystore);
        impl.setKeystorePassword(keystorePassword);
        impl.setKeyPassword(keyPassword);
        impl.setKeyalias(keyalias);
    }

    public int doStartTag() throws JspException
    {
    	MailcapCommandMap mc = (MailcapCommandMap) CommandMap.getDefaultCommandMap(); 
    	mc.addMailcap("text/html;; x-java-content-handler=com.sun.mail.handlers.text_html"); 
    	mc.addMailcap("text/xml;; x-java-content-handler=com.sun.mail.handlers.text_xml"); 
    	mc.addMailcap("text/plain;; x-java-content-handler=com.sun.mail.handlers.text_plain"); 
    	mc.addMailcap("multipart/*;; x-java-content-handler=com.sun.mail.handlers.multipart_mixed"); 
    	mc.addMailcap("message/rfc822;; x-java-content- handler=com.sun.mail.handlers.message_rfc822"); 
        startTime = System.currentTimeMillis();
        //Tag Start monitoring event. This exists with onTagEnd() which is invoked at the end of the tag processing.
        // If you remove this make sure you remove the onTagEnd() also
        onTagStart();
        LicenseService ls =  ServiceFactory.getLicenseService();
        allowFailover = ls.allowFastMail();

        if ( ! this.deferAttributeProcessing )
	    processAttributes();

        checkTimeout();

	    // make sure we return EVAL_BODY_TAG so the body will be buffered.
	    return super.doStartTag() == SKIP_BODY ? SKIP_BODY : BodyTag.EVAL_BODY_TAG;
    }

    public void doInitBody() throws JspException
    {
		// Turn off output supression since this output isn't
		// going to appear on the final page anyways.
		if( bodyContent instanceof CFOutput )
		{
			((CFOutput)bodyContent).cfoutput(true);
		}
    }

    /**
      *  Validate the tag's attributes and set parameters.
      *  This is logically a part of doStartTag() processing, but since
      *  the attributes can change with the current query row, we must
      *  process them within the tag body.
      */
	public void processAttributes() throws JspException {

		validate();
		checkTimeout();
		SearchTerm s = null;
		// get default server info if not set in tag
		try {
			MailSpoolService ms = ServiceFactory.getMailSpoolService();
			ms.validate(impl);
		} catch (MailSessionException e) {
			throw e;
		} catch (Exception ex) {
			throw new MailSessionException(ex);
		}
		// Process attachements
		if (attach != null) {
			File file = new File(attach);
			if (file.isFile()) {
				try {
					impl.setAttachment(file, removeAttachment);
				} catch (Exception ex) {
					throw new MailAttachmentException(ex);
				}
			} else {
				file = new File(pageContext.getServletContext().getRealPath(
						attach));
				if (file.isFile()) {
					try {
						impl.setAttachment(file, removeAttachment);
					} catch (Exception ex) {
						throw new MailAttachmentException(ex);
					}
				} else {
					throw new ResourceNotFoundException(attach);
				}
			}
		}

		// Set the Mailer header
		try {
			impl.setHeader("X-Mailer", mailerid);
		} catch (Exception ex) {
			throw new MailHeaderException(ex);
		}

	}

    public int doAfterBody() throws JspException {

        MailSpoolService ms = ServiceFactory.getMailSpoolService();
        // Ignore the CFMail body if parts are already specified
        if (! impl.isMultipart()) {

            String ct = impl.getContentType();
            if (ct.toLowerCase().startsWith("multipart/")) {
                // Try to handle the case where user is specifying multipart Content-type
                // but is try to do it themselves, probably to do inline images.
                throw new MultipartException();
            }
            
            String content = this.mobileContent;
            if (bodyContent != null)
                content = bodyContent.getString();
            if (content != null && content.length() > 0 || impl.getBodyPartsSize() == 0)
            {
                try
                {
                    impl.addPart(content, type, charset, wrapText);
                    if (bodyContent != null)
                        bodyContent.clearBody();

                }
                catch (Exception ex)
                {
                    throw new OutputException(ex);
                }
            }
        }

        // Send the mailImpl object to the Spool queue
        ms.storeMail(impl);

        checkTimeout();

        if (query != null && (group != null || ! containsOutputTag)) {
            return super.doAfterBody();
        }
        else {
            return SKIP_BODY;
        }
    }

    public int doEndTag() throws JspException {
		// Leaving tag... output needs to go back to original state.
		if( bodyContent instanceof CFOutput )
		{
			((CFOutput)bodyContent).cfoutput(false);
		}
        // invoke monitoring event handlers to capture this tag's monitor data. This exists with onTagStart() at the tagStart.
        // If you remove this make sure you remove the tagStart() invocation near the tag start
		String server = host;
		if(port != -1) {
		    server = host + ":" + Integer.toString(port);
		}
        List<TagAttribute> tagAttributes = captureAttributes(Arrays.asList("server", "attachment"), Arrays.asList(convertUpperCase(server), convertUpperCase(attach)));
        onTagEnd(tagAttributes);
        return EVAL_PAGE;
    }




   
    
    public void checkTimeout() throws RequestTimedOutException {
        if (RequestMonitor.isRequestTimedOut()) {
            throw new RequestTimedOutException(tagNameFromClass());
        }
    }

    public int getIdnaVersion()
    {
        return idnaVersion;
    }

    /**
     * @tagattribute
     * @required false
     * @rtexprvalue true
     */
    public void setIdnaVersion(int idnaVersion)
    {
        if(idnaVersion != 2003 && idnaVersion != 2008) {
            throw new InvalidIDNAVersionException();
        }
        this.idnaVersion = idnaVersion;
    }

/*    public void doFinally()
	{
		release();
	    super.doFinally();
    }
*/
}

