/*
Copyright (c) 2000 Allaire.  All Rights Reserved.

DO NOT REDISTRIBUTE THIS SOFTWARE IN ANY WAY WITHOUT THE EXPRESSED
WRITTEN PERMISSION OF ALLAIRE.
*/
package coldfusion.mail;

import coldfusion.runtime.JSONUtils;
import coldfusion.wddx.Base64Encoder;

import javax.mail.Session;
import java.security.AccessController;
import java.security.PrivilegedExceptionAction;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.ArrayList;

/**
 * Class to wrap the server hosts for mail.
 *
 * @author Clement Wong
 * @author Tom Jordahl
 */
public class HostImpl
{

    private static final String IMAP = "imap";
	private static final String POP3 = "pop3";
	private static final String SMTP = "smtp";

	/**
     * Create a description of a host.
     * The default protocol is "smtp".
     */
    public HostImpl()
    {
    }

    /**
     * Create a description of a host.
     *
     * @param protocol which overrides the default of "smtp"
     */
    public HostImpl(String protocol)
    {
        this.protocol = protocol;
    }

    protected String host = null;
    protected int port = -1;
    protected int timeout = -1;
    protected String username = null;
    protected String password = null;
    protected String protocol = SMTP;
    protected boolean useTLS = false;
    protected boolean useSSL = false;
    private final String SSL_FACTORY = "javax.net.ssl.SSLSocketFactory";

    protected Properties props = (Properties) System.getProperties().clone();
    protected Session s;

    /**
     * Set the timeout
     * @param t timeout in seconds
     */
    public void setTimeout(int t)
    {
        timeout = t;
    }

    /**
     * get the timeout value (in seconds)
     * @return timeout in seconds.
     */
    public int getTimeout()
    {
        return timeout;
    }

    public void setProtocol(String p)
    {
        protocol = p;
    }

    public String getProtocol()
    {
        return protocol;
    }

    public String getHost()
    {
        return host;
    }

    public void setHost(String host)
    {
        this.host = host;
    }

    public int getPort()
    {
        return port;
    }

    public void setPort(int port)
    {
        this.port = port;
    }

    public String getUsername()
    {
        return username;
    }

    public void setUsername(String username)
    {
        if (username != null && username.length() > 0)
            this.username = username;
    }

    public String getPassword()
    {
        return password;
    }

    public void setPassword(String password)
    {
        if (password != null && password.length() > 0)
            this.password = password;
    }

    public void setUseTLS(boolean useTLS)
    {
        this.useTLS = useTLS;
    }

    public void setUseSSL(boolean useSSL)
    {
        this.useSSL = useSSL;
        if (useSSL)
        {
            switch(protocol.toLowerCase()) {
            case SMTP:
            case IMAP:
            case POP3:
            	protocol += "s";
            	break;            	
            }            
        }
    }

    /**
     * Implement interface coldfusion.mail.MailSource.
     * Return a JavaMail session object with properties (timeout, etc) set
     *
     * @return a JavaMail Session object
     */
    public Session getSession() throws Exception
    {

        if (s == null)
        {
            // Set the mail.host here, it gets used for the Message-ID
            // but not when we actually connect
            String mailDotHost = "mail.host";
			String mailHost = System.getProperty(mailDotHost, host);
            props.put(mailDotHost, mailHost);
            String mailDot = "mail.";
			if (protocol != null)
            {
                String dotHost = ".host";
				props.put(mailDot + protocol + dotHost, host);
            }
            
            // Code inspection of Sun's JavaMail impl show that it looks
            // for mail.smtp.connectiontimeout when doing a connect and
            // mail.smtp.timeout while reading.
            if (timeout != -1 && protocol != null)
            {
                int miliseconds = timeout * 1000;
                String dotTimeout = ".timeout";
                String dotConnectionTimeout = ".connectiontimeout";
				props.put(mailDot + protocol + dotTimeout, Integer.toString(miliseconds));
				props.put(mailDot + protocol + dotConnectionTimeout, Integer.toString(miliseconds));
            }
            else if (timeout != -1)
            {
                int miliseconds = timeout * 1000;
                String mailTimeout = "mail.timeout";
                String mailConnectionTimeout = "mail.connectiontimeout";
				props.put(mailTimeout, Integer.toString(miliseconds));
				props.put(mailConnectionTimeout, Integer.toString(miliseconds));
            }
            // Must set mail.smtp.auth to "true" to use authentication
            String TRUE = "true";
			String SMTPS = "smtps";
			if (username != null && password != null)
            {
                String dotAuth = ".auth";
				props.put(mailDot + protocol + dotAuth, TRUE);
                String authProtocol = protocol;
                if(protocol.equalsIgnoreCase(SMTPS))
                    authProtocol = SMTP;
                String[] passwordParts = password.split("\\.");
                if(passwordParts.length == 3)
                {
                    String base64decoded = "";
                    boolean secondaryOauth = false;
                    try{
                        base64decoded = new String(Base64Encoder.decode(passwordParts[0]));
                        if(JSONUtils.isJSON(base64decoded))
                            secondaryOauth = true;
                    } catch (Exception e)
                    {
                        secondaryOauth = false;
                    }
                    if(secondaryOauth)
                    {
                        String authMechanisms = ".auth.mechanisms";
                        String authLoginDisable = ".auth.login.disable";
                        String authPlainDisable = ".auth.plain.disable";
                        String authNTLMDisable = ".auth.ntlm.disable";
    					String xOauth2 = "XOAUTH2";
    					props.put(mailDot + authProtocol + authMechanisms, xOauth2);
    					props.put(mailDot + authProtocol + authLoginDisable, TRUE);
    					props.put(mailDot + authProtocol + authPlainDisable, TRUE);
    					props.put(mailDot + authProtocol + authNTLMDisable, TRUE);
                        if(authProtocol.contains(POP3)) {
							String authTwoLineFormat = ".auth.xoauth2.two.line.authentication.format";
							props.put(mailDot + authProtocol + authTwoLineFormat, TRUE);
						}
                    }
                }

            }
            
            if (useTLS)
            {
                String mailStartTLSEnable = "mail.smtp.starttls.enable";
				props.put(mailStartTLSEnable, TRUE);
            }
            if (useSSL)
            {
                String mailTransportProtocol = "mail.transport.protocol";
                String socketFactoryClass = ".socketFactory.class";
                String socketFactoryPort = ".socketFactory.port";
				props.put(mailTransportProtocol, SMTPS);
				props.put(mailDot + protocol + socketFactoryClass, SSL_FACTORY);
				props.put(mailDot + protocol + socketFactoryPort, port);
            }

            String dotPort = ".port";
			props.setProperty( mailDot + protocol + dotPort,""+ port);
            String sendPartial = ".sendpartial";
			props.setProperty( mailDot + protocol + sendPartial, TRUE);

            if (System.getSecurityManager() == null)
            {
                s = Session.getInstance(props, null);
            }
            else
            {
                // running with sandbox security, we grant access permission to
                // getting the session object
                s = (Session) AccessController.doPrivileged(new PrivilegedExceptionAction()
                {
                    public Object run() throws Exception
                    {
                        return Session.getInstance(props, null);
                    }
                });
            }
        }
        return s;
    }

    /**
     * Returns this host in the format
     *   hostname[:port][:username]
     */
    public String toString()
    {
        StringBuilder builder = new StringBuilder(host);
        if (port != -1)
            builder.append(":").append(port);
        if(username != null)
            builder.append(":").append(username);
        return builder.toString();
    }
    
    public String getHostString()
    {
        StringBuilder builder = new StringBuilder(host);
        if (port != -1)
            builder.append(":").append(port);
        return builder.toString();
    }

    /**
     * Utility to parse a host string in separate HostImpl objects.
     * <P>
     * Input can be a comma separated list of hostnames with
     * an optional username, password or port specified.  i.e. [user:pass@]hostname[:port]
     * <P>
     * Defaults unspecified items to null (username, password) or -1 (port, timeout).
     *
     * @param input the string to parse
     * @return an array of HostImpl objects
     */
    public static HostImpl[] parseServer(String input)
    {
        return parseServer(input, -1, null, null, -1, null, null);
    }


    /**
     * Utility to parse a host string in separate HostImpl objects.
     * <p>
     * Input can be a comma separated list of hostnames with
     * an optional username, password or port specified.  i.e. [user:pass@]hostname[:port]
     *
     * @param input the string to parse
     * @param defaultPort port to use if not found
     * @param defaultUser username to use if not found
     * @param defaultPassword password to use if not found
     * @param timeout value to set in created hosts
     * @param useTLS use transport level security
     * @param useSSL use SSL to connect
     * @return an array of HostImpl objects
     */
    public static HostImpl[] parseServer(String input,
                                         int defaultPort,
                                         String defaultUser,
                                         String defaultPassword,
                                         int timeout,
                                         Boolean useTLS,
                                         Boolean useSSL)
    {
        StringTokenizer tok = new StringTokenizer(input, ",");
        int size = tok.countTokens();
        ArrayList hosts = new ArrayList(size);
        while (tok.hasMoreTokens())
        {
            String token = tok.nextToken().trim();
            if (token.length() == 0)
                continue;
            
            HostImpl himpl = new HostImpl();
            himpl.setPort(defaultPort);
            himpl.setUsername(defaultUser);
            himpl.setPassword(defaultPassword);
            himpl.setTimeout(timeout);
            if (useTLS != null)
                himpl.setUseTLS(useTLS.booleanValue());
            if (useSSL != null)
                himpl.setUseSSL(useSSL.booleanValue());

            // Look for username:password
            int at = token.lastIndexOf('@');
            if (at != -1)
            {
                String auth = token.substring(0, at);
                token = token.substring(at + 1);
                int colon = auth.indexOf(':');
                if (colon != -1)
                {
                    himpl.setUsername(auth.substring(0,colon));
                    himpl.setPassword(auth.substring(colon + 1));
                }
                else
                {
                    himpl.setUsername(auth);
                }
            }

            // Now look for host:port
            int colon = token.indexOf(':');
            if (colon != -1)
            {
                int port = Integer.parseInt(token.substring(colon + 1));
                String host = token.substring(0, colon);
                himpl.setHost(host);
                himpl.setPort(port);
            }
            else
            {
                himpl.setHost(token);
            }
            hosts.add(himpl);
        }

        HostImpl[] hArray = new HostImpl[hosts.size()];
        hosts.toArray(hArray);
        return hArray;
    }


}

