/*
 * Copyright (c) 1997-2018 IDRsolutions (https://www.idrsolutions.com)
 */
var XFA_BORDER_STYLE = {
	none: "none", solid: "solid", lowered: "inset", dotted: "dotted",
	dashed: "dashed", emboessed: "outset", raised: "outset", etched: "outset",
	dashdot: "dashed", dashdotdot: "dotted"
};

var XFA_NUMERIC_PATTERN = "^[+-]?\\\\d*(\\\\.\\\\d*)?$";
var XFA_INTEGER_PATTERN = "^[+-]?\\\\d*$";


var ALL_PAGEAREAS = new Array();
var processingPageArea = null;
var processingContentArea = null;
var processingDateEle = null;

var textAreaPrevious;

var isFireFox = false;//navigator.userAgent.toLowerCase().indexOf("firefox") > 0;
var isChrome = navigator.userAgent.toLowerCase().indexOf('chrome') > -1;
var isIE = navigator.userAgent.toLowerCase().indexOf('msie') > -1 || navigator.userAgent.toLowerCase().indexOf('trident') > -1;

//below are the templates for fast performance cloning

var TEMPLATE_DRAW = DCE("div");
TEMPLATE_DRAW.setAttribute("class", "draw");

var TEMPLATE_FIELD = DCE("div");
TEMPLATE_FIELD.setAttribute("class", "field");

var TEMPLATE_UI_BUTTON = DCE("input");
TEMPLATE_UI_BUTTON.setAttribute("type", "button");
TEMPLATE_UI_BUTTON.setAttribute("class", "ui");

var TEMPLATE_UI_TEXTEDIT = DCE("input");
TEMPLATE_UI_TEXTEDIT.setAttribute("type", "text");
TEMPLATE_UI_TEXTEDIT.setAttribute("class", "ui");

var TEMPLATE_UI_PASSWORDEDIT = DCE("input");
TEMPLATE_UI_PASSWORDEDIT.setAttribute("type", "PASSWORD");
TEMPLATE_UI_PASSWORDEDIT.setAttribute("class", "ui");

var TEMPLATE_UI_TEXTAREA = DCE("textarea");
TEMPLATE_UI_TEXTAREA.setAttribute("class", "ui");

//use this code below to hide datetimepicker
document.onclick = function (e) {
	var ecPicker = document.getElementById("ecmaDatePicker");
	var cur = e.target;
	while (cur && cur !== ecPicker && cur !== processingDateEle) {
		cur = cur.parentNode;
		if (cur === null) {
			hideEcmaDatePicker();
		}
	}
};

function DCE(tag) {
	return document.createElement(tag);
}

function IFIT(actualW, actualH, maxW, maxH) {
	var ratio = Math.min((maxW / actualW), (maxH / actualH));
	return {w: (actualW * ratio), h: (actualH * ratio)};
}

function toRGB(str) {
	return "rgb(" + str + ")";
}

function layoutPseudoModel() {
	this.ready = false;
}

layoutPseudoModel.prototype = inherit(new object());

Object.defineProperty(layoutPseudoModel.prototype, "absPage", {
	writable: true, enumerable: true, configurable: true,
	value: function (objParam) {
		console.log("absPage not implemented");
		return 0;
	}
});

Object.defineProperty(layoutPseudoModel.prototype, "absPageCount", {
	writable: true, enumerable: true, configurable: true,
	value: function () {
		return document.getElementsByClassName("pageArea").length;
	}
});

Object.defineProperty(layoutPseudoModel.prototype, "absPageCountInBatch", {
	writable: true, enumerable: true, configurable: true,
	value: function () {
		console.log("absPageCountInBatch not implemented");
		return 0;
	}
});

Object.defineProperty(layoutPseudoModel.prototype, "absPageInBatch", {
	writable: true, enumerable: true, configurable: true,
	value: function (objParam) {
		console.log("absPageInBatch not implemented");
		return 0;
	}
});

Object.defineProperty(layoutPseudoModel.prototype, "absPageSpan", {
	writable: true, enumerable: true, configurable: true,
	value: function (objParam) {
		console.log("absPageSpan not implemented");
		return 0;
	}
});

Object.defineProperty(layoutPseudoModel.prototype, "h", {
	writable: true, enumerable: true, configurable: true,
	value: function (objParam, strParam, intParam) {
		console.log("h not implemented");
		return 0;
	}
});

Object.defineProperty(layoutPseudoModel.prototype, "page", {
	writable: true, enumerable: true, configurable: true,
	value: function (objParam) {
		console.log("page not implemented");
	}
});

Object.defineProperty(layoutPseudoModel.prototype, "pageContent", {
	writable: true, enumerable: true, configurable: true,
	value: function (intParam, strParam, boolParam) {
		console.log("pageContent not implemented");
	}
});

Object.defineProperty(layoutPseudoModel.prototype, "pageCount", {
	writable: true, enumerable: true, configurable: true,
	value: function () {
		return document.getElementsByClassName("pageArea").length;
	}
});

Object.defineProperty(layoutPseudoModel.prototype, "pageSpan", {
	writable: true, enumerable: true, configurable: true,
	value: function (objParam) {
		console.log("pageSpan not implemented");
	}
});

Object.defineProperty(layoutPseudoModel.prototype, "relayout", {
	writable: true, enumerable: true, configurable: true,
	value: function () {
		reDraw();
	}
});

Object.defineProperty(layoutPseudoModel.prototype, "relayoutPageArea", {
	writable: true, enumerable: true, configurable: true,
	value: function (intParam) {
		console.log("relayoutPageArea not implemented");
	}
});

Object.defineProperty(layoutPseudoModel.prototype, "sheet", {
	writable: true, enumerable: true, configurable: true,
	value: function (objParam) {
		console.log("sheet not implemented");
		return 0;
	}
});

Object.defineProperty(layoutPseudoModel.prototype, "sheetCount", {
	writable: true, enumerable: true, configurable: true,
	value: function () {
		console.log("sheetCount not implemented");
		return 0;
	}
});

Object.defineProperty(layoutPseudoModel.prototype, "sheetCountInBatch", {
	writable: true, enumerable: true, configurable: true,
	value: function () {
		console.log("sheetCountInBatch not implemented");
		return 0;
	}
});

Object.defineProperty(layoutPseudoModel.prototype, "sheetInBatch", {
	writable: true, enumerable: true, configurable: true,
	value: function (objParam) {
		console.log("sheetInBatch not implemented");
		return 0;
	}
});

Object.defineProperty(layoutPseudoModel.prototype, "w", {
	writable: true, enumerable: true, configurable: true,
	value: function (objParam, strParam, intParam) {
		console.log("w not implemented");
		return 0;
	}
});

Object.defineProperty(layoutPseudoModel.prototype, "x", {
	writable: true, enumerable: true, configurable: true,
	value: function (objParam, strParam, intParam) {
		console.log("x not implemented");
		return 0;
	}
});

Object.defineProperty(layoutPseudoModel.prototype, "y", {
	writable: true, enumerable: true, configurable: true,
	value: function (objParam, strParam, intParam) {
		console.log("y not implemented");
		return 0;
	}
});

function hostPseudoModel() {
	this.className = "hostPseudoModel";
	this.appType = "";
	this.calculationsEnabled = true;
	this.currentPage = 0;
	this.language = "";
	this.numPages = 0;
	this.platform = "";
	this.title = "";
	this.validationsEnabled = false;
	this.variation = "";
	this.version = "";
}

hostPseudoModel.prototype = inherit(new object());

Object.defineProperty(hostPseudoModel.prototype, "beep", {
	writable: true, enumerable: true, configurable: true,
	value: function () {
		var snd = new Audio("data:audio/wav;base64,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");
		snd.play();
	}
});

Object.defineProperty(hostPseudoModel.prototype, "documentCountInBatch", {
	writable: true, enumerable: true, configurable: true,
	value: function () {
		console.log("documentCountInBatch not implemented");
		return 1;
	}
});

Object.defineProperty(hostPseudoModel.prototype, "documentInBatch", {
	writable: true, enumerable: true, configurable: true,
	value: function () {
		console.log("documentInBatch not implemented");
		return 0;
	}
});

Object.defineProperty(hostPseudoModel.prototype, "exportData", {
	writable: true, enumerable: true, configurable: true,
	value: function (strParam, boolParam) {
		console.log("exportData not implemented");
	}
});

Object.defineProperty(hostPseudoModel.prototype, "getFocus", {
	writable: true, enumerable: true, configurable: true,
	value: function () {
		console.log("getFocus not implemented");
		return {};
	}
});

Object.defineProperty(hostPseudoModel.prototype, "gotoURL", {
	writable: true, enumerable: true, configurable: true,
	value: function (strParam, boolParam) {
		console.log("gotourl not implemented");
	}
});

Object.defineProperty(hostPseudoModel.prototype, "importData", {
	writable: true, enumerable: true, configurable: true,
	value: function (strParam) {
		console.log("importData not implemented");
	}
});

Object.defineProperty(hostPseudoModel.prototype, "messageBox", {
	writable: true, enumerable: true, configurable: true,
	value: function (strParam1, strParam2, intParam1, intParam2) {
		alert(strParam1);
	}
});

Object.defineProperty(hostPseudoModel.prototype, "openList", {
	writable: true, enumerable: true, configurable: true,
	value: function (objParam) {
		console.log("openList not implemented");
	}
});

Object.defineProperty(hostPseudoModel.prototype, "pageDown", {
	writable: true, enumerable: true, configurable: true,
	value: function () {
		console.log("pageDown not implemented");
	}
});

Object.defineProperty(hostPseudoModel.prototype, "pageUp", {
	writable: true, enumerable: true, configurable: true,
	value: function () {
		console.log("pageDown not implemented");
	}
});

Object.defineProperty(hostPseudoModel.prototype, "print", {
	writable: true, enumerable: true, configurable: true,
	value: function (boolParam1, intParam2, intParam3, boolParam4, boolParam5, boolParam6, boolParam7, boolParam8) {
		window.print();
	}
});

Object.defineProperty(hostPseudoModel.prototype, "resetData", {
	writable: true, enumerable: true, configurable: true,
	value: function () {
		console.log("resetData not implemented");
	}
});

Object.defineProperty(hostPseudoModel.prototype, "response", {
	writable: true, enumerable: true, configurable: true,
	value: function (strParam1, strParam2, strParam3, boolParam) {
		console.log("response not implemented");
		return "";
	}
});

Object.defineProperty(hostPseudoModel.prototype, "setFocus", {
	writable: true, enumerable: true, configurable: true,
	value: function (objParam) {
		if (objParam.domElement) {
			objParam.domElement.focus();
		}
	}
});

function validateInput(e, patternStr) {
	var reg = new RegExp(patternStr, "g");
	if (!reg.test(e.value)) {
		var errSpan = DCE("span");
		var spans = e.parentElement.getElementsByTagName("span");
		if (spans.length === 0) {
			errSpan.setAttribute("class", "errorMsg");
			errSpan.textContent = "Invalid Input";
			e.parentElement.appendChild(errSpan);
			window.setTimeout(function () {
				var par = errSpan.parentElement;
				par.removeChild(errSpan);
			}, 1000);
		}
		e.value = "";
	}
}

function getDraw(draw) {
	var dEle = TEMPLATE_DRAW.cloneNode(true);
	var value = findChild(draw, T_VALUE);
	var paraT = findChild(draw, T_PARA);
	if (paraT !== null) {
		var lineH = XPTS(paraT.lineHeight);
		if (lineH > 0) {
			dEle.style.lineHeight = lineH + "pt";
		}
	}

	var minW = draw._minW;
	if (minW > 0) {
		dEle.style.minWidth = minW + "pt";
	}
	var minH = draw._minH;
	if (minH > 0) {
		dEle.style.minHeight = minH + "pt";
	}
	var w = draw._w;
	if (w > 0) {
		dEle.style.width = w + "pt";
	}
	var h = draw._h;
	if (h > 0) {
		dEle.style.height = h + "pt";
	}

	w = Math.max(minW, w);
	h = Math.max(minH, h);

	var childList = draw.nodes;
	for (var i = 0, ii = childList.length; i < ii; i++) {
		var child = childList.item(i);
		var cid = child.classID;
		if (cid === T_BORDER) {
			drawBorder(child, dEle, false);
		} else if (cid === T_FONT) {
			drawFont(child, dEle);
		} else if (cid === T_MARGIN) {
			var mTop = XPTS(child.topInset);
			var mRight = XPTS(child.rightInset);
			var mBottom = XPTS(child.bottomInset);
			var mLeft = XPTS(child.leftInset);
			dEle.style.padding = mTop + "pt " + mRight + "pt " + mBottom + "pt " + mLeft + "pt";
		}
	}

	if (value !== null) {
		var list = value.nodes;
		for (var i = 0, len = list.length; i < len; i++) {
			var dd = list.item(i);
			var cn = dd.className;
			if (cn === "rectangle") {
				drawBorder(dd, dEle, false);
			} else if (cn === "exData" || cn === "text") {
				dEle.style.display = "table";
				var label = DCE("label");
				label.innerHTML = dd.value;
				dEle.appendChild(label);
				var para = findChild(draw, T_PARA);
				if (para !== null) {
					label.style.verticalAlign = para.vAlign;
					label.style.textAlign = para.hAlign;
				}
			} else if (cn === "para") {
				var lineH = XPTS(para.lineHeight);
				if (lineH > 0) {
					console.log(lineH + " " + lineH);
					dEle.style.lineHeight = lineH + "pt";
				}
			} else if (cn === "image") {
				var img = DCE("img");
				if (dd.href.length > 0) {
					img.setAttribute("src", dd.href);
				} else {
					var data64 = dd.value;
					var cType = dd.contentType;
					img.setAttribute("src", "data: " + cType + ";charset=utf-8;base64," + data64);
				}
				var asp = dd.aspect;

				img.style.maxWidth = w + "pt";
				img.style.maxHeight = h + "pt";

				img.onload = function () {
					var iw = XPTS(this.width + "px");
					var ih = XPTS(this.height + "px");
					if (asp === "fit" || asp === "none" || asp === "actual") {
						var obj = IFIT(iw, ih, w, h);
						img.style.width = obj.w + "pt";
						img.style.height = obj.h + "pt";
					} else if (asp === "width") {
						img.style.width = w + "pt";
					} else if (asp === "height") {
						img.style.maxHeight = h + "pt";
					}
				};

				dEle.appendChild(img);
			}
		}
	}

	var relevant = draw.getAttribute("relevant");
	if (relevant.length > 0) {
		if (relevant === "-print") {
			dEle.className += " printMinus";
		} else if (relevant === "+print") {
			dEle.className += " printPlus";
		}
	}

	draw.domElement = dEle;
	dEle.somElement = draw;

	if (draw.rotate !== "0") {
		var rr = draw.rotate;
		dEle.style.WebkitTransformOrigin = "0 0";
		dEle.style.msTransformOrigin = "0 0";
		dEle.style.transformOrigin = "0 0";
		dEle.style.WebkitTransform = "rotate(-" + rr + "deg)";
		dEle.style.msTransform = "rotate(-" + rr + "deg)";
		dEle.style.transform = "rotate(-" + rr + "deg)";
		console.log(dEle.style.transformOrigin);
	}

	return dEle;
}

function getField(field) {

	var minW = field._minW;
	var minH = field._minH;
	var maxW = field._maxW;
	var maxH = field._maxH;
	var w = field._w;
	var h = field._h;

	var fSet = TEMPLATE_FIELD.cloneNode(true);

	var fStyle = fSet.style;

	if (w === 0) {
		fStyle.minWidth = minW + "pt";
	} else {
		fStyle.width = w + "pt";
	}

	if (h === 0) {
		fStyle.minHeight = minH + "pt";
	} else {
		fStyle.height = h + "pt";
	}

	if (maxH > 0) {
		fStyle.maxWidth = maxW + "pt";
	}
	if (maxH > 0) {
		fStyle.maxHeight = maxH + "pt";
	}

	var fInput = getInput(field);
	var isButton = field.getSubType() === TYPE_BUTTON;

	var isCheck = field.getSubType() === TYPE_CHECKBUTTON;
	if (isCheck) {
		fInput.style.width = fInput.style.height = "10pt";
	}

	var caption = null;
	var curFontSize = 10;
	var curReserve = 0;
	var curWidth = Math.max(w, minW);

	var mRight = 0;
	var mLeft = 0;
	var mBottom = 0;
	var mTop = 0;

	var childList = field.nodes;
	for (var i = 0, ii = childList.length; i < ii; i++) {
		var child = childList.item(i);
		var cid = child.classID;
		if (cid === T_BORDER) {
			drawBorder(child, fSet, false);
		} else if (cid === T_CAPTION) {
			caption = child;
		} else if (cid === T_FONT) {
			drawFont(child, fSet);
			curFontSize = XPTS(child.size);
		} else if (cid === T_MARGIN) {
			mTop = XPTS(child.topInset);
			mRight = XPTS(child.rightInset);
			mBottom = XPTS(child.bottomInset);
			mLeft = XPTS(child.leftInset);
			fStyle.padding = mTop + "pt " + mRight + "pt " + mBottom + "pt " + mLeft + "pt";
		} else if (cid === T_UI) {
			var arr = new Array();
			findDescendant(child, T_BORDER, arr);
			//disable internal content border at the moment
//            if (arr.length > 0) {
//                drawBorder(arr[0], fInput, false);
//            }
		} else if (cid === T_PARA) {
			fInput.style.verticalAlign = child.vAlign;
			fInput.style.textAlign = child.hAlign;
		}
	}

	if (caption !== null) {

		var fCap = DCE("div");
		fCap.setAttribute("class", "caption");

		var para = findChild(caption, T_PARA);
		var ff = findChild(caption, T_FONT);
		if (ff !== null) {
			if (isButton) {
				drawFont(ff, fInput);
			} else {
				drawFont(ff, fCap);
			}
		}

		var place = caption.placement;
		var reserve = 0;
		if ("reserve" in caption) {
			reserve = XPTS(caption.reserve);
		}

		var value = findChild(caption, T_VALUE);

		if (isButton && value !== null) {
			//button does not like innerHTML stuff;
			var tSpan = DCE("span");
			tSpan.innerHTML = value.hashtagText;
			fInput.value = tSpan.textContent;
			fSet.appendChild(fInput);

		} else {

			var td0 = DCE("div");
			td0.setAttribute("class", "box");
			var td1 = DCE("div");
			td1.setAttribute("class", "box");
			fSet.appendChild(td0);
			fSet.appendChild(td1);

			if (value !== null) {
				var t = findChild(value, T_TEXT);
				if (t !== null) {
					fCap.innerHTML = t.value;
				} else {
					var ex = findChild(value, T_EXDATA);
					if (ex !== null) {
						fCap.innerHTML = ex.value;
					}
				}
			}
			if (place === "left") {
				if (reserve !== 0) {
					td0.style.width = reserve + "pt";
				}
				td0.appendChild(fCap);
				td1.appendChild(fInput);
				if (para !== null) {
					td0.style.verticalAlign = para.vAlign;
					td0.style.textAlign = para.hAlign;
				}
				td1.style.height = "100%";
				td1.style.verticalAlign = "middle";
				curReserve = reserve;
			} else if (place === "right") {
				if (reserve !== 0) {
					td1.style.width = reserve + "pt";
				}
				td0.appendChild(fInput);
				td1.appendChild(fCap);
				if (para !== null) {
					td1.style.verticalAlign = para.vAlign;
					td1.style.textAlign = para.hAlign;
				}
				td0.style.verticalAlign = "middle";
				td0.style.height = "100%";
				curReserve = reserve;
			} else if (place === "top") {
				td0.style.height = reserve + "pt";
				fCap.style.display = "table-cell";
				td0.appendChild(fCap);
				td1.appendChild(fInput);
				td1.style.verticalAlign = "middle";
				td0.style.display = "table-row";
				td1.style.height = "100%";
				if (para !== null) {
					fCap.style.verticalAlign = para.vAlign;
				}
			} else if (place === "bottom") {
				fCap.style.height = reserve + "pt";
				fCap.style.display = "table-cell";
				td0.appendChild(fInput);
				td1.appendChild(fCap);
				td0.style.height = "100%";
				td0.style.verticalAlign = "middle";
				td1.style.display = "table-row";
				if (para !== null) {
					fCap.style.verticalAlign = para.vAlign;
				}
			}
		}

	} else {
		fSet.appendChild(fInput);
	}

	field.domElement = fSet;
	fSet.somElement = field;

	var relevant = field.getAttribute("relevant");
	if (relevant.length > 0) {
		if (relevant === "-print") {
			fSet.className += " printMinus";
		} else if (relevant === "+print") {
			fSet.className += " printPlus";
		}
	}


	if (field.presence === "hidden") {
		fStyle.display = "none";
	} else if (field.presence === "invisible") {
		fStyle.visibility = "hidden";
	} else {
		var combArr = new Array();
		findDescendant(field, T_COMB, combArr);
		if (combArr.length > 0) {
			var nCell = parseInt(combArr[0].numberOfCells);
			if (nCell !== 0) {
				fInput.style.fontFamily = "monospace";
				fInput.maxLength = nCell;
				var holderStr = " ";
				for (var i = 1; i < nCell; i++) {
					holderStr += "|";
				}
				fInput.placeholder = holderStr;
				var gap = 0;

				if (place === "left" || place === "right") {
					gap = ((curWidth - curReserve - mLeft - mRight) - (curFontSize * 0.55 * nCell)) / (nCell + 1);
				} else {
					gap = ((curWidth - mLeft - mRight) - (curFontSize * 0.55 * nCell)) / (nCell + 1);
				}
				if (gap > 0) {
					fInput.style.letterSpacing = gap + "pt";
				}
			}
		}
	}

	if (field.rotate !== "0") {
		var rr = field.rotate;
		fSet.style.WebkitTransformOrigin = "0 0";
		fSet.style.msTransformOrigin = "0 0";
		fSet.style.transformOrigin = "0 0";
		fSet.style.WebkitTransform = "rotate(-" + rr + "deg)";
		fSet.style.msTransform = "rotate(-" + rr + "deg)";
		fSet.style.transform = "rotate(-" + rr + "deg)";
	}

	return fSet;
}

function expandTextArea(ele) {
	var mh = ele.somElement._minH;
	if (mh > 0) {
		var capt = findChild(ele.somElement, T_CAPTION);
		var marg = findChild(ele.somElement, T_MARGIN);
		if (marg !== null) {
			mh -= (XPTS(marg.topInset) + XPTS(marg.bottomInset));
		}
		if (capt !== null && (capt.placement === "top" || capt.placement === "bottom") && capt.reserve.length > 0) {
			mh -= XPTS(capt.reserve);
		}
		ele.style.height = "auto";
		var curHeight = XPTS(ele.scrollHeight + "px");
		if (curHeight > mh && ele.value.length > 0) {
			ele.style.height = curHeight + "pt";
		} else {
			ele.style.height = mh + "pt";
		}
	}
}

function isRawFloat(n) {
	return n === Number(n) && n % 1 !== 0;
}

function updateHTMLFields() {
	var fieldArr = document.getElementsByClassName("ui");
	for (var a = 0, aa = fieldArr.length; a < aa; a++) {
		var input = fieldArr[a];
		var field = input.somElement;
		if (!field) {
			continue;
		}
		var tagName = input.tagName.toUpperCase();
		if (tagName === "SELECT") {
			input.textContent = "";
			var optArr = new Array();
			var items = findChildren(field, T_ITEMS);
			if (items.length === 2) {
				var texts0 = findChildren(items[0], T_TEXT);
				var texts1 = findChildren(items[1], T_TEXT);
				var texts = [texts0, texts1];
				var textsLen = texts0.length;

				for (var i = 0; i < textsLen; i++) {
					var opt = DCE("option");
					optArr.push(opt);
				}

				for (var z = 0; z < items.length; z++) {
					var it = items[z];
					var isValue = it.save === "1"; //do string check not number
					if (isValue) {
						textsLen = texts[z].length;
						for (var i = 0; i < textsLen; i++) {
							var textStr = texts[z][i].value;
							optArr[i].value = textStr;
						}
					} else {
						for (var i = 0; i < textsLen; i++) {
							var textStr = texts[z][i].value;
							optArr[i].textContent = textStr;
						}
					}
				}
			} else if (items.length === 1) {
				var texts = findChildren(items[0], T_TEXT);
				for (var i = 0; i < texts.length; i++) {
					var opt = DCE("option");
					var textStr = texts[i].value;
					opt.textContent = textStr;
					opt.value = textStr;
					optArr.push(opt);
				}
			}
			if (optArr.length > 0 && optArr[0].value.trim() !== "") {
				var opt = DCE("option");
				input.appendChild(opt);
			}

			for (var z = 0, zz = optArr.length; z < zz; z++) {
				input.appendChild(optArr[z]);
			}
			input.value = field.rawValue;

		} else if (tagName === "TEXTAREA") {
			input.value = field.rawValue;
		} else if (tagName === "INPUT") {
			var type = input.type.toUpperCase();
			if (type === "TEXT" || type === "PASSWORD") {
				var raw = field.rawValue;
				if (raw && field.getSubType() === TYPE_NUMERICEDIT) {
					var rr = "" + raw;
					if (rr.indexOf(".") !== -1) {
						input.value = parseFloat(raw).toFixed(2);
					} else {
						input.value = parseInt(raw);
					}

				} else {
					input.value = field.rawValue;
				}
			} else if (type === "CHECKBOX") {
				if (field.rawValue === "1") {
					input.checked = true;
				}
			} else if (type === "radio") {

			}
		}
	}
}

function fireEvent(hEle, activity, isText) {

	var hType = hEle.type.toUpperCase();

	if (activity === "change") {
		if (hType === "CHECKBOX") {
			if (hEle.checked) {
				hEle.somElement.rawValue = xfa.event.newText = "1";
			} else {
				hEle.somElement.rawValue = xfa.event.newText = "0";
			}
		} else {
			hEle.somElement.rawValue = xfa.event.newText = hEle.value;
		}
	}

	var field = hEle.somElement;

	var arr = new Array();
	iterateAndFireEvent(field, activity, arr);

	var needUpdate = arr.length > 0;
	if ((activity === "change" || activity === "click") && !needUpdate && field.parent.className === "exclGroup") {
		var eventsArr = findChildren(field.parent, T_EVENT);
		for (var z = 0, zz = eventsArr.length; z < zz; z++) {
			if (eventsArr[z].activity === activity) {
				var scr = findChild(eventsArr[z], T_SCRIPT);
				if (scr !== null) {
					needUpdate = true;
					processingEle = field;
					field.parent.fireEvent(scr.value);
				}
			}
		}
	}

	if (activity !== "enter") {
		xfa.form.recalculate();
		updateHTMLFields();
	}

	if (needUpdate && !isText) {
		reDraw();
	} else if (hType === "TEXTAREA") {
		if (activity === "enter") {
			textAreaPrevious = hEle.value;
		} else if ((activity === "change") && hEle.value !== textAreaPrevious) {
			textAreaPrevious = hEle.value;
			reDraw(hEle);
		}
	}
}

function getInput(field) {

	var st = field.getSubType();
	var input = null;

	var isText = false;
	var isArea = false;

	switch (st) {
		case TYPE_TEXTEDIT:
			if (field.multiLine === "1") { //this is string check not number
				input = TEMPLATE_UI_TEXTAREA.cloneNode(true);
			} else {
				input = TEMPLATE_UI_TEXTEDIT.cloneNode(true);
				isArea = field._minH > 0;
			}
			input.value = field.rawValue;
			isText = true;
			break;
		case TYPE_BUTTON:
			input = TEMPLATE_UI_BUTTON.cloneNode(true);
			break;
		case TYPE_PASSWORDEDIT:
			input = TEMPLATE_UI_PASSWORDEDIT.cloneNode(true);
			input.value = field.rawValue;
			isText = true;
			break;
		case TYPE_NUMERICEDIT:
			input = TEMPLATE_UI_TEXTEDIT.cloneNode(true);
			input.setAttribute("onkeyup", "validateInput(this,\"" + XFA_NUMERIC_PATTERN + "\")");
			var raw = field.rawValue;
			if (raw) {
				var rr = "" + raw;
				if (rr.indexOf(".") !== -1) {
					input.value = parseFloat(raw).toFixed(2);
				} else {
					input.value = parseInt(raw);
				}
			} else {
				input.value = field.rawValue;
			}
			isText = true;
			break;
		case TYPE_DATETIMEEDIT:
			input = TEMPLATE_UI_TEXTEDIT.cloneNode(true);
			input.value = field.rawValue;
			isText = true;
			break;
		case TYPE_CHECKBUTTON:
			input = DCE("input");
			if (field.parent.className === "exclGroup") {
				var nn = field.parent.somExpression;
				input.setAttribute("type", "radio");
				input.setAttribute("name", nn);
				var itm = findChild(field, T_ITEMS);
				if (itm !== null) {
					var ing = findChild(itm, T_INTEGER);
					var txt = findChild(itm, T_TEXT);
					if (ing !== null) {
						input.value = ing.value;
					} else if (txt !== null) {
						input.value = txt.value;
					}
				}
				if (input.value === field.parent.rawValue) {
					input.checked = true;
				}

			} else {
				input.setAttribute("type", "checkbox");
				if (field.rawValue === "1") {
					input.checked = true;
				}
			}
			break;
		case TYPE_CHOICELIST:
			input = DCE("select");
			var items = findChildren(field, T_ITEMS);
			var optArr = new Array();
			if (items.length === 2) {
				var texts0 = findChildren(items[0], T_TEXT);
				var texts1 = findChildren(items[1], T_TEXT);
				var texts = [texts0, texts1];
				var textsLen = texts0.length;

				for (var i = 0; i < textsLen; i++) {
					var opt = DCE("option");
					optArr.push(opt);
				}

				for (var z = 0; z < items.length; z++) {
					var it = items[z];
					var isValue = it.save === "1"; //do string check not number
					if (isValue) {
						textsLen = texts[z].length;
						for (var i = 0; i < textsLen; i++) {
							var textStr = texts[z][i].value;
							optArr[i].value = textStr;
						}
					} else {
						for (var i = 0; i < textsLen; i++) {
							var textStr = texts[z][i].value;
							optArr[i].textContent = textStr;
						}
					}
				}
			} else if (items.length === 1) {
				var texts = findChildren(items[0], T_TEXT);
				for (var i = 0; i < texts.length; i++) {
					var opt = DCE("option");
					var textStr = texts[i].value;
					opt.textContent = textStr;
					opt.value = textStr;
					optArr.push(opt);
				}
			}
			if (optArr.length > 0 && optArr[0].value.trim() !== "") {
				var opt = DCE("option");
				input.appendChild(opt);
			}
			for (var z = 0, zz = optArr.length; z < zz; z++) {
				input.appendChild(optArr[z]);
			}
			input.selectedIndex = -1;
			input.value = field.rawValue;
			break;
		case TYPE_IMAGEEDIT:
			input = DCE("input");
			input.setAttribute("type", "file");
			break;
		case TYPE_SIGNATURE:
			input = DCE("div");
			input.style.width = input.style.height = "100%";
			input.style.backgroundColor = "rgba(222,227,255,0.9)";
			break;
	}

	if (field.access === "readOnly") {
		input.readOnly = true;
	}

	setInputEvents(input, field, isText, isArea);

	input.somElement = field;
	return input;

}

function setInputEvents(input, field, isText, isArea) {

	var subType = field.getSubType();

	var focusStr = "";
	var blurStr = "";
	var clickStr = "";
	var changeStr = "";

	var changeAdded = false;
	var eventsArr = findChildren(field, T_EVENT);
	for (var z = 0, zz = eventsArr.length; z < zz; z++) {
		var activity = eventsArr[z].activity;
		if (activity === "enter") {
			focusStr += "fireEvent(this,'enter'," + isText + ");";
		} else if (activity === "exit") {
			blurStr += "fireEvent(this,'exit'," + isText + ");";
		} else if (activity === "change") {
			if (isText) {
				changeStr += "fireEvent(this,'change'," + isText + ");";
			} else if (subType === TYPE_CHECKBUTTON) {
				clickStr += "fireEvent(this,'change',false);";
			} else if (subType === TYPE_CHOICELIST) {
				changeStr += "fireEvent(this,'change',false);";
				changeAdded = true;
			}
		} else if (activity === "mouseUp" || activity === "mouseDown" || activity === "click") {
			if (subType === TYPE_BUTTON || subType === TYPE_CHECKBUTTON) {
				clickStr += "fireEvent(this,'" + activity + "',false);";
			} else if (subType === TYPE_CHOICELIST) {
				changeStr += "fireEvent(this,'" + activity + "',false);";
			}
		}
	}

	if (isText && changeStr.length === 0) {
		blurStr += "fireEvent(this,'change'," + isText + ");";
	} else if (subType === TYPE_CHOICELIST) {
		changeStr += "fireEvent(this,'change'," + isText + ");";
	} else if (subType === TYPE_CHECKBUTTON) {
		if (!changeAdded) {
			clickStr += "fireEvent(this,'change'," + isText + ");";
		}
		clickStr += "fireEvent(this,'click'," + isText + ");";
	}

	if (focusStr.length > 0) {
		input.setAttribute("onfocus", focusStr);
	} else if (isArea) {
		input.setAttribute("onfocus", "fireEvent(this,'enter',true);");
	}


	if (blurStr.length > 0) {
		input.setAttribute("onblur", blurStr);
	}

	if (clickStr.length > 0) {
		input.setAttribute("onclick", clickStr);
	}
	if (changeStr.length > 0) {
		input.setAttribute("onchange", changeStr);
	}

	if (subType === TYPE_DATETIMEEDIT) {
		input.setAttribute("onclick", "showEcmaDatePicker(this,event);" + clickStr);
	}

}

function drawBorder(b, hComp, isSubform) {
//    return;
	var edge = findChildren(b, T_EDGE);
	var edgeLen = edge.length;
	if (b.presence === "visible" && edgeLen > 0) {
		if (edgeLen > 1 && !isSubform) {
			for (var i = 0; i < edgeLen; i++) {
				var e = edge[i];
				var thick = XPTS(e.thickness);
				thick = thick > 0 ? Math.max(thick, 0.8) : thick;
				var c = "black";
				var ec = findChild(e, T_COLOR);
				if (ec !== null) {
					c = toRGB(ec.value);
				}

				var stroke = XFA_BORDER_STYLE[e.stroke];
				if (e.presence === "visible") {
					switch (i) {
						case 0:
							hComp.style.borderTop = thick + "pt " + stroke + " " + c;
							break;
						case 1:
							hComp.style.borderRight = thick + "pt " + stroke + " " + c;
							break;
						case 2:
							hComp.style.borderBottom = thick + "pt " + stroke + " " + c;
							break;
						case 3:
							hComp.style.borderLeft = thick + "pt " + stroke + " " + c;
							break;
					}
				}
			}
		} else {
			var e = edge[0];
			if (e.presence === "visible") {
				var thick = XPTS(e.thickness);
				var stroke = XFA_BORDER_STYLE[e.stroke];
				thick = thick > 0 ? Math.max(thick, 0.8) : thick;
				var c = "black";
				var ec = findChild(e, T_COLOR);
				if (ec !== null) {
					c = toRGB(ec.value);
				}
				hComp.style.outline = thick + "pt " + stroke + " " + c;
			}
		}
	}

	var bf = findChild(b, T_FILL);
	if (bf !== null) {
		drawFill(bf, hComp);
	}
}

function drawFont(nn, hComp) {
	hComp.style.fontSize = nn.size;
	hComp.style.fontFamily = nn.typeface;
	hComp.style.fontWeight = nn.weight;
	hComp.style.fontStyle = nn.posture;
	var ff = findChild(nn, T_FILL);
	if (ff !== null) {
		var cc = findChild(ff, T_COLOR);
		if (cc != null) {
			hComp.style.color = toRGB(cc.value);
		}
	}
}

function drawFill(bf, hComp) {

	var firstColor = "rgb(255,255,255)";
	var secondColor = "rgb(0,0,0)";

	var bc = findChild(bf, T_COLOR);
	if (bc !== null) {
		firstColor = toRGB(bc.value);
	}

	var bfList = bf.nodes;
	var len = bfList.length;
	var csFound = false;
	for (var i = 0; i < len; i++) {
		var sh = bfList.item(i);
		var cn = sh.className;
		if (cn === "linear") {
			var bc2 = findChild(sh, T_COLOR);
			if (bc2 !== null) {
				secondColor = toRGB(bc2.value);
			}
			updateLinearCSS(hComp, sh.type.substring(2), secondColor, firstColor);
			csFound = true;
		} else if (cn === "radial") {
			var bc2 = findChild(sh, T_COLOR);
			if (bc2 !== null) {
				secondColor = toRGB(bc2.value);
			}
			if (sh.type === "toEdge") {
				updateRadialCSS(hComp, firstColor, secondColor);
			} else {
				updateRadialCSS(hComp, secondColor, firstColor);
			}
			csFound = true;
		} else if (cn === "pattern") {
			csFound = true;
		}

	}
	if (!csFound) {
		hComp.style.backgroundColor = firstColor;
	}
}

function updateRadialCSS(hEle, c1, c2) {
	var rVal = '';
	var prefixes = ['-o-', '-ms-', '-moz-', '-webkit-'];
	var dom = document.createElement('div');
	for (var i = 0, ii = prefixes.length; i < ii; i++) {
		dom.style.background = prefixes[i] + 'linear-gradient(#000000, #ffffff)';
		if (dom.style.background) {
			rVal = prefixes[i];
		}
	}
	delete dom;
	hEle.style.background = rVal + 'radial-gradient(circle,' + c1 + ', ' + c2 + ')';
}

function updateLinearCSS(hEle, dir, c1, c2) {
	var rVal = '';
	var prefixes = ['-o-', '-ms-', '-moz-', '-webkit-'];
	var dom = document.createElement('div');
	for (var i = 0, ii = prefixes.length; i < ii; i++) {
		dom.style.background = prefixes[i] + 'linear-gradient(#000000, #ffffff)';
		if (dom.style.background) {
			rVal = prefixes[i];
		}
	}
	delete dom;
	hEle.style.background = rVal + 'linear-gradient(' + dir + ', ' + c1 + ', ' + c2 + ')';
}

function findContainerHeight(nn) {
	if (!nn.isContainer) {
		return 0;
	} else if (nn.className === "area") {
		var hh = 0;
		var list = nn.nodes;
		var len = list.length;
		for (var i = 0; i < len; i++) {
			var temp = list.item(i);
			if (temp.isContainer) {
				var y = temp._y;
				var tempH = findContainerHeight(temp);
				hh = Math.max(y + tempH, hh);
			}
		}
		return hh;
	} else if (nn.isWidget) {
		return getDomHeight(nn);
	}
	var h = nn._h;
	var minH = nn._minH;
	var hh = Math.max(h, minH);
	if ((hh === 0 && nn.layout === "position") || nn.layout === "row") {
		var list = nn.nodes;
		var len = list.length;
		for (var i = 0; i < len; i++) {
			var temp = list.item(i);
			if (temp.isContainer) {
				var y = temp._y;
				var tempH = findContainerHeight(temp);
				hh = Math.max(y + tempH, hh);
			}
		}
		var mm = findChild(nn, T_MARGIN);
		if (mm !== null) {
			hh += XPTS(mm.topInset);
			hh += XPTS(mm.bottomInset);
		}
	}
	return hh;
}

function findContainerWidth(nn) {
	if (!nn.isContainer) {
		return 0;
	}
	var w = nn._w;
	var minW = nn._minW;
	var maxW = nn._maxW;
	var ww = Math.max(w, minW);
	if (ww > 0) {
		return w;
	}
	if (nn.isWidget) {
		return Math.max(ww, maxW);
	}

	//no width found do the calculation
	var list = nn.nodes;
	var len = list.length;
	var totalW = 0;
	if (nn.layout === "position" || nn.layout === "tb") {
		for (var i = 0; i < len; i++) {
			var temp = list.item(i);
			if (temp.isContainer) {
				var x = temp._x;
				var tempW = findContainerWidth(temp);
				totalW = Math.max(x + tempW, totalW);
			}
		}
	} else if (nn.layout === "lr-tb" || nn.layout === "rl-tb") {
		for (var i = 0; i < len; i++) {
			var temp = list.item(i);
			if (temp.isContainer) {
				var tempW = findContainerWidth(temp);
				totalW += tempW;
			}
		}
	} else if (nn.layout === "table") {
		var cw = nn.columnWidths;
		if (cw.length > 0) {
			var wSplits = cw.split(" ");
			for (var i = 0, ii = wSplits.length; i < ii; i++) {
				var val = XPTS(wSplits[i]);
				if (val !== "-1") {
					totalW += val;
				}
			}
		}
	} else if (nn.layout === "row" || nn.layout === "rl-row") {
		totalW = findContainerWidth(nn.parent);
	}
	return totalW;
}

function isNodeVisibe(nn) {
	var par = nn;
	while (par !== null) {
		if (par.isContainer && ("presence" in par) && par.presence === "invisible") {
			return false;
		}
		par = par.parent;
	}
	return true;
}

function getPageAreaWidth(pa) {
	var w = 0;
	var med = findChild(pa, T_MEDIUM);
	if (med !== null) {
		if (med.orientation === "portrait") {
			w = XPTS(med.short);
		} else {
			w = XPTS(med.long);
		}
	}
	if (w === 0) {
		var contA = findChild(pa, T_CONTENTAREA);
		if (contA !== null) {
			w = contA._x + contA._w;
		}
	}
	return w;
}

function getPageAreaHeight(pa) {
	var h = 0;
	var med = findChild(pa, T_MEDIUM);
	if (med !== null) {
		if (med.orientation === "portrait") {
			h = XPTS(med.long);
		} else {
			h = XPTS(med.short);
		}
	}
	if (h === 0) {
		var contA = findChild(pa, T_CONTENTAREA);
		if (contA !== null) {
			h = contA._y + contA._h;
		}
	}
	return h;
}

function getDomHeight(nn) {
	if (nn.className === "field" && nn.getSubType() === TYPE_TEXTEDIT
			&& nn.multiLine === "1" && nn.rawValue) {
		var sh = XPTS(nn.domElement.getElementsByTagName("textarea")[0].scrollHeight + "px");
		var ch = XPTS(nn.domElement.clientHeight + "px");
		var capt = findChild(nn, T_CAPTION);
		if (capt !== null && (capt.placement === "top" || capt.placement === "bottom") && capt.reserve.length > 0) {
			sh += XPTS(capt.reserve);
		}
		return Math.max(sh, ch);
	}
	var h = nn._h;
	var minH = nn._minH;
	return Math.max(h, minH);
}

function reDraw(focusEle) {
	var previous = 0;
	if (isIE) {
		previous = document.documentElement.scrollTop;
	} else {
		previous = window.scrollY;
	}

	ALL_PAGEAREAS = new Array();
	findDescendants(xfa.form, T_PAGEAREA, ALL_PAGEAREAS);

	for (var i = 0, ii = ALL_PAGEAREAS.length; i < ii; i++) {
		ALL_PAGEAREAS[i].occurCount = 0;
	}

	var mainXFAForm = document.getElementById("mainXFAForm");
	mainXFAForm.innerHTML = "";

	var sample = document.getElementById("preRenderer");
	var arr = new Array();
	findDescendants(xfa.form, T_DRAW, arr);
	for (var z = 0, zz = arr.length; z < zz; z++) {
		var ff = arr[z];
		var res;
		if (!isIE && focusEle) {
			res = ff.domElement;
		} else {
			res = getDraw(ff);
		}
		sample.appendChild(res);
	}
	arr = new Array();
	findDescendants(xfa.form, T_FIELD, arr);
	for (var z = 0, zz = arr.length; z < zz; z++) {
		var ff = arr[z];
		var res;
		if (!isIE && focusEle) {
			res = ff.domElement;
		} else {
			res = getField(ff);
		}
		sample.appendChild(res);
	}

	if (!isIE && focusEle) {
		expandTextArea(focusEle);
	} else {
		var tarr = sample.getElementsByTagName("textarea");
		for (var i = 0, ii = tarr.length; i < ii; i++) {
			expandTextArea(tarr[i]);
		}
	}

	processingPageArea = ALL_PAGEAREAS[0];
	processingContentArea = findChild(processingPageArea, T_CONTENTAREA);

	var cAreaDom = drawPageArea(processingPageArea, processingContentArea);
	processingPageArea.occurCount++;

	var indentArr = new Array();
	indentArr.push(cAreaDom.parentNode);
	indentArr.push(cAreaDom);

	layoutContainers(xfa.form, indentArr);

	window.scrollTo(0, previous);
	sample.textContent = "";
}

function updateRelevantPageArea(bNode) {
	var occ = processingPageArea.getOccur();

	if (bNode.className === "breakBefore") {
		if (occ.max !== -1 && bNode.targetType === "pageArea") {
			if (bNode.target.length > 0) {
				if (bNode.target.charAt(0) === '#') {
					var targetID = bNode.target.substring(1);
					for (var i = 0, ii = ALL_PAGEAREAS.length; i < ii; i++) {
						if (ALL_PAGEAREAS[i].id === targetID) {
							processingPageArea = ALL_PAGEAREAS[i];
							processingContentArea = findChild(processingPageArea, T_CONTENTAREA);
							break;
						}
					}
				} else {
					var targetName = bNode.target;
					for (var i = 0, ii = ALL_PAGEAREAS.length; i < ii; i++) {
						if (ALL_PAGEAREAS[i].name === targetName) {
							processingPageArea = ALL_PAGEAREAS[i];
							processingContentArea = findChild(processingPageArea, T_CONTENTAREA);
							break;
						}
					}
				}

			} else if (parseInt(occ.max) === processingPageArea.occurCount) {

				var index = -1;
				for (var i = 0, ii = ALL_PAGEAREAS.length; i < ii; i++) {
					if (ALL_PAGEAREAS[i] === processingPageArea) {
						index = i;
					}
				}
				if (index !== -1 && ALL_PAGEAREAS[index + 1]) {
					processingPageArea = ALL_PAGEAREAS[index + 1];
					processingContentArea = findChild(processingPageArea, T_CONTENTAREA);
				}
			}
			processingPageArea.occurCount++;
		}


	} else if (bNode.className === "breakAfter") {


	} else if (bNode.className === "Break") {

		if (occ.max !== -1 && bNode.before === "pageArea") {
			if (bNode.beforeTarget.length > 0) {
				if (bNode.beforeTarget.charAt(0) === '#') {
					var targetID = bNode.beforeTarget.substring(1);
					for (var i = 0, ii = ALL_PAGEAREAS.length; i < ii; i++) {
						if (ALL_PAGEAREAS[i].id === targetID) {
							processingPageArea = ALL_PAGEAREAS[i];
							processingContentArea = findChild(processingPageArea, T_CONTENTAREA);
							break;
						}
					}
				}

			} else if (parseInt(occ.max) === processingPageArea.occurCount) {

				var index = -1;
				for (var i = 0, ii = ALL_PAGEAREAS.length; i < ii; i++) {
					if (ALL_PAGEAREAS[i] === processingPageArea) {
						index = i;
					}
				}
				if (index !== -1 && ALL_PAGEAREAS[index + 1]) {
					processingPageArea = ALL_PAGEAREAS[index + 1];
					processingContentArea = findChild(processingPageArea, T_CONTENTAREA);
				}
			}
			processingPageArea.occurCount++;
		}

	}
}

function layoutPageArea(curObj, indentArr) {
	var cLay = curObj.layout;
	var list = curObj.nodes;
	var len = list.length;

	var pos = indentArr.length;

	var rowIndex = 0;
	for (var i = 0; i < len; i++) {
		var temp = list.item(i);
		var toPut = indentArr[pos - 1];
		var cAreaDom = indentArr[1];
		if (temp.isContainer && temp.presence !== "hidden" && temp.className != "contentArea") {

			var y = temp._y;
			var x = temp._x;

			var sd = null;
			if (temp.isWidget) {
				sd = temp.domElement.cloneNode(true);
				sd.somElement = temp;
				var ui = sd.getElementsByClassName("ui")[0];
				if (ui) {
					ui.somElement = temp;
				}
			} else {
				sd = getSubform(temp);
				indentArr.push(sd);
				sd.style.zIndex = indentArr.length;
			}
			if (curObj.className === "pageArea") {
				sd.style.position = "absolute";
				sd.style.margin = y + "pt" + " 0pt 0pt " + x + "pt";
			}

			if (cLay === "position") {
				sd.style.position = "absolute";
				sd.style.marginLeft = x + "pt";
				sd.style.marginTop = y + "pt";
			} else if (cLay === "lr-tb") {
				sd.style.float = "left";
			} else if (cLay === "rl-tb") {
				sd.style.float = "right";
			} else if (cLay === "row") {
				var tableObj = curObj.parent;
				var cwStr = tableObj.columnWidths;
				var curSpan = parseInt(temp.colSpan);
				var widthArray = new Array();
				if (cwStr.length > 0) {
					var wSplits = cwStr.split(" ");
					var wLen = wSplits.length;
					for (var s = 0; s < wLen; s++) {
						var val = XPTS(wSplits[s]);
						widthArray.push(val);
					}
				}
				var cw = widthArray[rowIndex];
				for (var s = 1; s < curSpan; s++) {
					cw += widthArray[rowIndex + s];
				}
				sd.style.width = cw + "pt";
				if (!isFireFox && !isChrome) {
					sd.style.float = "left";
				}
				rowIndex++;
			}
			toPut.appendChild(sd);

			if (!temp.isWidget) {
				layoutPageArea(temp, indentArr);
			}

			if (cLay === "tb") {
				var br = DCE("div");
				toPut.appendChild(br);
			}
		}
	}
	indentArr.pop();
}

function layoutContainers(curObj, indentArr) {
	var cLay = curObj.layout;
	var list = curObj.nodes;
	var len = list.length;
	var maxHH = Math.max(curObj._h, curObj._minH);

	var cAreaDivsLen = indentArr[1].getElementsByTagName("div").length;
	var totalFields = indentArr[1].getElementsByClassName("field");
	var totalDraws = indentArr[1].getElementsByTagName("draw");
	var inCAREA = totalFields.length > 0 || totalDraws.length > 0;

	var brkBefore = findChild(curObj, T_BREAKBEFORE);
	var brk = findChild(curObj, T_BREAK);

	if (brkBefore !== null && cAreaDivsLen > 2) {
		if ((brkBefore.targetType === "contentArea" && brkBefore.startNew === "1") || brkBefore.targetType === "pageArea") {
			updateRelevantPageArea(brkBefore);
			getAncestorSubformsDom(curObj, indentArr);
			var cd = getSubform(curObj);
			indentArr[indentArr.length - 1].appendChild(cd);
			indentArr.push(cd);
		}
	} else if (brk !== null && brk.before.length > 0 && cAreaDivsLen > 2) {
		updateRelevantPageArea(brk);
		getAncestorSubformsDom(curObj, indentArr);
		var cd = getSubform(curObj);
		indentArr[indentArr.length - 1].appendChild(cd);
		indentArr.push(cd);
	} else if (curObj.isContainer && curObj.classID !== T_EXCLGROUP && (curObj.layout === "position" || curObj.layout === "row") && inCAREA) {
		var cAreaDom = indentArr[1];
		if (cAreaDom.scrollHeight > (cAreaDom.clientHeight + 2)) {
			var toPut = getAncestorSubformsDom(curObj, indentArr);
			toPut.appendChild(curObj.domElement);
			indentArr.push(curObj.domElement);
		}
	}

	var pos = indentArr.length;

	var rowIndex = 0;
	for (var i = 0; i < len; i++) {
		var temp = list.item(i);
		var toPut = indentArr[pos - 1];
		var cAreaDom = indentArr[1];
		if (temp.isContainer && temp.classID !== T_PAGESET && temp.presence !== "hidden") {

			var y = temp._y;
			var x = temp._x;

			var sd = null;
			if (temp.isWidget) {
				sd = temp.domElement;
			} else {
				sd = getSubform(temp);
				indentArr.push(sd);
				sd.style.zIndex = indentArr.length + 2;
			}

			if (curObj.classID === T_AREA) {
				sd.style.position = "absolute";
				sd.style.marginLeft = x + "pt";
				sd.style.marginTop = y + "pt";
			} else if (cLay === "position") {
				sd.style.position = "absolute";
				sd.style.marginLeft = x + "pt";
				sd.style.marginTop = y + "pt";
			} else if (cLay === "lr-tb") {
				sd.style.float = "left";
			} else if (cLay === "rl-tb") {
				sd.style.float = "right";
			} else if (cLay === "row") {
				var tableObj = curObj.parent;
				var cwStr = tableObj.columnWidths;
				var curSpan = parseInt(temp.colSpan);
				var widthArray = new Array();
				if (cwStr.length > 0) {
					var wSplits = cwStr.split(" ");
					var wLen = wSplits.length;
					for (var s = 0; s < wLen; s++) {
						var val = XPTS(wSplits[s]);
						widthArray.push(val);
					}
				}
				var cw = widthArray[rowIndex];
				for (var s = 1; s < curSpan; s++) {
					cw += widthArray[rowIndex + s];
				}
				if (temp.colSpan === "-1") {
					sd.style.width = "100%";
				} else {
					sd.style.width = cw + "pt";
				}
				if (!isFireFox && !isChrome) {
					sd.style.float = "left";
				}
				rowIndex += curSpan;
			}

//          
			toPut.appendChild(sd);
			if (temp.isWidget) {
				var tl = temp.parent.layout;
				//just add 1 incase something goes wrong                 
				if (cAreaDom.scrollHeight > (cAreaDom.clientHeight + 2)) {
					var canParse = false;
					if (tl === "position") {
						canParse = maxHH === 0 && curObj.classID !== T_EXCLGROUP;
					} else if (tl !== "row") {
						canParse = true;
					}
					if (canParse) {
						var toPut2 = getAncestorSubformsDom(temp, indentArr);
						toPut.removeChild(sd);
						toPut = toPut2;
						toPut.appendChild(sd);
					}
				}
			} else {
				layoutContainers(temp, indentArr);
			}

			if (cLay === "tb") {
				var ddd = DCE("div");
				toPut.appendChild(ddd);
			}
		}
	}

	if (curObj.className.indexOf("subform") === 0 && cLay === "position") {
		var maxHH = Math.max(curObj._h);
		if (maxHH === 0) {
			var dom = curObj.domElement;
			for (var i = 0, ii = dom.children.length; i < ii; i++) {
				var cStyle = window.getComputedStyle(dom.children[i], null);
				maxHH = Math.max(maxHH, (XPTS(cStyle.height + "px") + XPTS(cStyle.marginTop + "px")));
			}
			var margin = findChild(curObj, T_MARGIN);
			if (margin !== null) {
				maxHH += XPTS(margin.topInset);
				maxHH += XPTS(margin.bottomInset);
			}
			dom.style.height = maxHH + "pt";
		}
	}

	if (indentArr.length > 3) {
		indentArr.pop();
	}
}

function getAncestorSubformsDom(nn, indentArr) {
	indentArr.splice(0, indentArr.length);
	var cAreaDom = drawPageArea(processingPageArea, processingContentArea);
	indentArr[0] = cAreaDom.parentNode;
	indentArr[1] = cAreaDom;

	var par = nn.parent;
	var divs = new Array();

	while (par !== null) {
		if (par.isContainer) {
			var sd = getSubform(par);
			divs.push(sd);
		}
		par = par.parent;
	}
	var dd = divs[0];
	for (var i = 0, ii = divs.length; i < ii; i++) {
		indentArr[2 + i] = divs[ii - 1 - i];
		indentArr[1 + i].appendChild(indentArr[2 + i]);
	}
	return dd;
}

function getSubform(temp) {
	var sd = DCE("div");

	temp.domElement = sd;
	sd.setAttribute("class", temp.className);

	if (temp.name.length > 0) {
		sd.setAttribute("name", temp.name);
	}

	var sdCSS = sd.style;

	if (temp.presence === "hidden") {
		sdCSS.display = "none";
	} else if (temp.presence === "invisible") {
		sdCSS.visibility = "hidden";
	}

	if (temp.layout === "table") {
		sdCSS.display = "table";
		sdCSS.borderSpacing = "0pt";
	} else if (temp.layout === "tb") {
		sdCSS.display = "block";
	} else if (temp.layout === "row") {
		if (isFireFox || isChrome) {
			sdCSS.display = "flex";
		} else {
			sdCSS.display = "inline-block";
		}
	}

//    var isFlowed = temp.layout === "tb" || temp.layout === "lr-tb" || temp.layout === "rl-tb";

	var y = temp._y;
	var x = temp._x;
	var h = temp._h;
	var minH = temp._minH;

	if (temp.className === "area") {
		sdCSS.width = (findContainerWidth(temp.parent) - temp._x) + "pt";
		sdCSS.height = (findContainerHeight(temp)) + "pt";
		return sd;
	}

	var cw = Math.max(temp._w, temp._minW);
	if (cw > 0) {
		sdCSS.width = cw + "pt";
	}

	if (temp.layout === "table" || temp.layout === "row") {
		sdCSS.width = findContainerWidth(temp) + "pt";
	}

	if (h > 0) {
		sdCSS.height = h + "pt";
	} else if (minH > 0 && temp.layout !== "tb" && temp.layout !== "position") {
		sdCSS.minHeight = minH + "pt";
	}

	var mm = findChild(temp, T_MARGIN);
	if (mm !== null) {
		sdCSS.paddingTop = XPTS(mm.topInset) + "pt";
		sdCSS.paddingLeft = XPTS(mm.leftInset) + "pt";
		sdCSS.paddingBottom = XPTS(mm.bottomInset) + "pt";
	}

	var bb = findChild(temp, T_BORDER);
	if (bb !== null) {
		drawBorder(bb, sd, true);
	}

	return sd;
}

function drawPageArea(pArea, cArea) {
	var pd = DCE("div");
	var pw = getPageAreaWidth(pArea);
	var ph = getPageAreaHeight(pArea);
	pd.style.width = pw + "pt";
	pd.style.height = ph + "pt";
	pd.setAttribute("class", "pageArea");

	var indArr = new Array();
	indArr.push(pd);
	layoutPageArea(pArea, indArr);
	var cd = DCE("div");
	cd.setAttribute("class", "contentArea");
	cd.style.marginLeft = cArea._x + "pt";
	cd.style.marginTop = cArea._y + "pt";
	cd.style.width = cArea._w + "pt";
	cd.style.height = cArea._h + "pt";
	pd.appendChild(cd);

	document.getElementById("mainXFAForm").appendChild(pd);
	return cd;
}

function processXFAData() {

	xfa = new xfa();

	updateFDFXFA();

	templateStr = document.getElementById("templateDump").textContent;
	templateStr = EcmaPDF.decode64(templateStr);

	var dataStr = document.getElementById("dataDump").textContent;
	var dataStr = EcmaPDF.decode64(dataStr);

	var f = new form();
	f.model = f;
	xfa.nodes.append(f);
	xfa.form = f;

	var t = new template();
	t.model = template;
	xfa.nodes.append(t);
	xfa.template = t;

	var e = new eventPseudoModel();
	xfa.nodes.append(e);
	xfa.event = e;

	xfa.dataset = new dataModel();
	xfa.layout = new layoutPseudoModel();
	xfa.host = new hostPseudoModel();

	var domParser = parseXFAXML(templateStr);
	generateTemplateModel(domParser.getElementsByTagName("template")[0], t);
	updateScriptXFA(t);
	for (av in AvailableVariables) {
		try {
			eval(av + " = AvailableVariables." + av);
		} catch (err) {
			console.log(err);
		}
	}

	for (av in AvailableVariableScripts) {
		var varScript = AvailableVariableScripts[av];
		varScript = formCalcToJavascript(varScript, false);
		try {
			eval(av + " = new function()" + varScript + ";");
		} catch (err) {
			console.log(err);
		}
	}

	generateSomModel(t, f);

	xfa.dataset.data = new node();
	xfa.data = xfa.dataset.data;
	xfa.data.className = "data";
	xfa.data.name = "xfa:data";
	xfa.data.model = xfa.dataset;

	if (dataStr.length > 5) {
		var dataParser = parseXFAXML(dataStr);
		var elems = dataParser.getElementsByTagName("*");
		generateDataModel(elems[1], xfa.data);
		initialMergeDomToSom(xfa.data, xfa.form);
		mergeGlobalDomToSom(xfa.data, xfa.form);
		xfa.data.nodes.clear();
	}

	iterateAndFireEvent(xfa.form, "docReady", new Array());
	iterateAndFireEvent(xfa.form, "ready", new Array());
	iterateAndFireEvent(xfa.form, "initialize", new Array());
	iterateAndFireCalculate(xfa.form, new Array());

	xfa.layout.relayout();
	initializeCalendar();
	document.getElementById("FDFXFA_PageCount").textContent = "/ "+document.getElementsByClassName("pageArea").length;
}

function initializeCalendar() {

	var div = DCE("div");
	div.setAttribute("id", "ecmaDatePicker");
	document.getElementsByTagName("body")[0].appendChild(div);

	var weeks = ["Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"];

	var leftLink = DCE("a");
	leftLink.textContent = "<";
	leftLink.setAttribute("href", "javascript:showEcmaPrevYear()");
	div.appendChild(leftLink);

	var yearLabel = DCE("label");
	div.appendChild(yearLabel);

	var rightLink = DCE("a");
	rightLink.textContent = ">";
	rightLink.setAttribute("href", "javascript:showEcmaNextYear()");
	div.appendChild(rightLink);

	leftLink = DCE("a");
	leftLink.textContent = "<";
	leftLink.setAttribute("href", "javascript:showEcmaPrevMonth()");
	div.appendChild(leftLink);

	var monthLabel = DCE("label");
	div.appendChild(monthLabel);

	rightLink = DCE("a");
	rightLink.textContent = ">";
	rightLink.setAttribute("href", "javascript:showEcmaNextMonth()");
	div.appendChild(rightLink);

	var table = DCE("table");
	div.appendChild(table);
	var tr = DCE("tr");
	for (var i = 0; i < 7; i++) {
		var th = DCE("th");
		th.textContent = weeks[i];
		tr.appendChild(th);
	}
	table.appendChild(tr);
	for (var i = 0; i < 6; i++) {
		tr = DCE("tr");
		for (var j = 0; j < 7; j++) {
			var td = DCE("td");
			tr.appendChild(td);
		}
		table.appendChild(tr);
	}
	updateDatePicker();
}

function showEcmaNextYear() {
	var dp = document.getElementById("ecmaDatePicker");
	dp.curYear++;
	updateDatePicker();
}

function showEcmaPrevYear() {
	var dp = document.getElementById("ecmaDatePicker");
	dp.curYear--;
	updateDatePicker();
}

function showEcmaNextMonth() {
	var dp = document.getElementById("ecmaDatePicker");
	if (dp.curMonth < 11) {
		dp.curMonth++;
		updateDatePicker();
	}
}

function showEcmaPrevMonth() {
	var dp = document.getElementById("ecmaDatePicker");
	if (dp.curMonth > 0) {
		dp.curMonth--;
		updateDatePicker();
	}
}

function showEcmaDatePicker(dateDiv, event) {
	if (dateDiv) {
		processingDateEle = dateDiv;
		dp = document.getElementById("ecmaDatePicker");
		processingDateEle.parentNode.appendChild(dp);
		var raw = dateDiv.somElement.rawValue;
		var dd = new Date();
		if (raw && raw.length > 0) {
			var mm = Date.parse(raw);
			dd = new Date(mm);
		}
		dp.curYear = dd.getFullYear();
		dp.curMonth = dd.getMonth();
		updateDatePicker();
		dp.style.display = "inherit";
		dp.style.zIndex = "5000";
	}
}

function hideEcmaDatePicker() {
	var dp = document.getElementById("ecmaDatePicker");
	dp.style.display = "none";
	document.getElementsByTagName("body")[0].appendChild(dp);
	dp.style.zIndex = "0";
}

function updateDatePicker() {
	var dp = document.getElementById("ecmaDatePicker");
	var year = dp.curYear;
	var month = dp.curMonth;

	var monthNames = ["January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December"];

	var labels = dp.getElementsByTagName("label");
	labels[0].textContent = year;
	labels[1].textContent = monthNames[month];

	var tDate = new Date();
	tDate.setFullYear(year);
	tDate.setMonth(month);
	tDate.setDate(1);

	var lastDayOfMonth = new Date(year, month + 1, 0).getDate();

	var weekNo = tDate.getDay();
	var tds = dp.getElementsByTagName("td");
	for (var i = 0, ii = tds.length; i < ii; i++) {
		tds[i].textContent = "";
		tds[i].removeAttribute("class");
		tds[i].removeAttribute("onclick");
	}
	var c = 1;
	for (var i = weekNo, ii = (lastDayOfMonth + weekNo); i < ii; i++) {
		var dateStr = year + "-" + padPrefix(month + 1) + "-" + padPrefix(c);
		tds[i].textContent = c++;
		tds[i].setAttribute("class", "date");
		tds[i].setAttribute("onclick", "selectEcmaDate('" + dateStr + "')");
	}

	function padPrefix(n) {
		return n < 10 ? '0' + n : n;
	}
}

function selectEcmaDate(str) {
	if (processingDateEle) {
		processingDateEle.value = str;
		processingDateEle.somElement.rawValue = str;
		hideEcmaDatePicker();
	}
}


var myriadOTF = "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";