/**
 * ConnectionPool
 *
 * Copyright (c) 1995 - 2003 Macromedia, Inc. All rights reserved.
 * DO NOT REDISTRIBUTE THIS SOFTWARE IN ANY WAY WITHOUT THE EXPRESSED
 * WRITTEN PERMISSION OF MACROMEDIA.
 *
 * Created on Mar 27, 2003
 * Author: tomj
 */
package coldfusion.mail;

import javax.mail.Transport;
import javax.mail.MessagingException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;

/**
 * This class manages SMTP mail server connections
 */
public class ConnectionPool
{
    /**
     * A connection to a server and the last time it was accessed.
     */
    protected class Server
    {
        protected Transport connection;
        protected long last;

        public Server(Transport t, long last)
        {
            this.connection = t;
            this.last = last;
        }
    }

    // The pool - String to LinkedList (of Servers) map
    private HashMap pool = new HashMap();

    /**
     * Create a new SMTP Mail server connection pool
     */
    public ConnectionPool()
    {
    }

    /**
     * Get a mail server connection from the pool
     *
     * @param key the combination of [user:password@]server:port
     * @return A JavaMail Transport object, or null if not found
     */
    public synchronized Transport checkout(String key)
    {
        Transport tran = null;

        // Look up the key
        LinkedList connectionList = (LinkedList) pool.get(key);
        if (connectionList == null)
            return null;

        // Find a connection
        if (!connectionList.isEmpty())
        {
            Server server = (Server) connectionList.removeFirst();
            tran = server.connection;
            /*
            // This code will trigger the sendmail "checksmtpattack()" function
            // which notices if we send too many NOOP commands (>10).
            // So we can't verify the connection this way
            if (!tran.isConnected())
            {
            }
            */
        }

        // Return connection if found or null if not
        return tran;
    }

    /**
     * Add a SMTP Mail server connection to the pool
     *
     * @param key the combination of [user:password@]server:port
     * @param connection the connection
     */
    public synchronized void checkin(String key, Transport connection)
    {
        // Create the entry to put in the list
        Server svr = new Server(connection, System.currentTimeMillis());

        // Look up the key
        LinkedList connectionList = (LinkedList) pool.get(key);

       // If we didn't find a list, create one
        if (connectionList == null)
        {
            LinkedList l = new LinkedList();
            l.add(svr);
            pool.put(key, l);
        }
        else   // Add this to the existing list
        {
            connectionList.add(svr);
        }

        return;
    }

    /**
     * Close any outstanding connections that haven't been used for
     * over a minute.
     */
    public synchronized void cleanup()
    {
        final int ONE_MINUTE = 60*1000;
        long now = System.currentTimeMillis();

        // Iterate over each entry in the map
        Iterator itPool = pool.keySet().iterator();
        while (itPool.hasNext())
        {
            String key = (String) itPool.next();
            LinkedList connectionList = (LinkedList) pool.get(key);
            if (connectionList != null)
            {
                // Close all connections on the list unused for 60 seconds
                for (int i = 0; i < connectionList.size(); i++)
                {
                    Server server = (Server) connectionList.get(i);
                    if ((now - server.last) > ONE_MINUTE)
                    {
                        // Remove this server from the list
                        connectionList.remove(i);

                        // Close the connection
                        try
                        {
                            server.connection.close();
                        }
                        catch (MessagingException e)
                        { // ignore
                        }
                    }
                } // connection loop
            }

            // We could remove the entry from the Map, but odds are
            // good that we will need the list again in the future
            // so keep the key pointing to an empty list.

        } // pool loop
    }

}

