/**
 * Copyright 2007 Adobe Systems Incorporated
 * All Rights Reserved.
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

package coldfusion.print;

import java.awt.print.PageFormat;
import java.awt.print.Paper;
import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.HashMap;
import java.util.Map;

import javax.print.DocFlavor;
import javax.print.DocPrintJob;
import javax.print.PrintService;
import javax.print.SimpleDoc;
import javax.print.attribute.PrintRequestAttributeSet;
import javax.print.attribute.standard.OrientationRequested;
import javax.print.attribute.standard.PageRanges;
import javax.servlet.jsp.PageContext;

import org.jpedal.PdfDecoder;
import org.jpedal.constants.JPedalSettings;
import org.jpedal.constants.PDFflags;
import org.jpedal.objects.raw.PdfDictionary;
import org.jpedal.utils.PdfBook;

import static coldfusion.print.core.PrintExceptions.*;
import coldfusion.document.JPedalFontRegistry;
import coldfusion.print.core.CFPrintException;
import coldfusion.server.ServiceFactory;
import coldfusion.tagext.io.FileUtils;
import coldfusion.vfs.VFSFileFactory;

/**
 * PrintPDF
 * <p/>
 * Created on Jan 18, 2007
 *
 * @author cframpto
 */
public class PrintPDF extends PrintBase
{
    private PrintPDFAttributes pdfAttributes;
    private String media = null;

    public PrintPDF(PageContext pageContext, Object source, String password)
    {
        super(pageContext, "pdf", source, password);
    }

    // JPedal caches to "java.io.tmpdir"/jpedal which is most likely outside of any sandbox.  If using sandbox,
    // execute JPedal code as privledged.  This assumes the source has been already been checked for read access and
    // the destination has already been checked for write access.
    public final void print(final String printer, final PrintRequestAttributeSet attributeSet, final String media,
                            final PrintPDFAttributes pdfAttributes)
            throws CFPrintException
    {
        if (System.getSecurityManager() == null)
        {
            _print(printer, attributeSet, media, pdfAttributes);
        }
        else
        {
            try
            {
                AccessController.doPrivileged(new PrivilegedExceptionAction()
                {
                    public Object run() throws CFPrintException
                    {
                        _print(printer, attributeSet, media, pdfAttributes);
                        return null;
                    }
                });
            }
            catch (PrivilegedActionException e)
            {
                throw (CFPrintException) e.getException();
            }
        }
    }

    private void _print(String printer, PrintRequestAttributeSet attributeSet, String media,
                        PrintPDFAttributes pdfAttributes)
            throws CFPrintException
    {
        this.printer = printer;
        this.attributeSet = attributeSet;
        this.media = media;
        this.pdfAttributes = pdfAttributes;
        setFidelity();

        PdfDecoder decode_pdf = null;

        try
        {
            decode_pdf = new PdfDecoder(true);

            // Set up any aliases that are needed to map the os font names to the Postscript font names.
            JPedalFontRegistry.init(decode_pdf);

            
			// If a PDFDocWrapper comes in, write the pdf to a byte[] so JPedal can use it.
            if (ServiceFactory.getPDFService(false) != null && ServiceFactory.getPDFService().isPDFCoreObject(source))
            {
            	source = ServiceFactory.getPDFService().writeDocument(this.password, source, false,false);
                
            }

            if (source instanceof byte[])
            {
                decode_pdf.openPdfArray((byte[]) source);
            }
            else if (source instanceof String)
            {
                if(VFSFileFactory.checkIfVFile((String)source))
                {
                    decode_pdf.openPdfArray(FileUtils.readBinaryFile((String)source));
                }
                else
                {
                    // Open the file (and read metadata including pages in  file).
                    decode_pdf.openPdfFile((String) source);
                }
            }
            else
            {
                // should never get here
                return;
            }

            // Set the pdf password if there is one.  This can't be done until after the
            // file has been opened!!!
            if (this.password != null)
                decode_pdf.setEncryptionPassword(this.password);

            // Check security and permissions for AllowPrintHigh.
            if (!isAllowPrintHigh(decode_pdf))
                throw new CFPrintSecurityAuthorizationException("AllowPrintHigh");

            // Print it!
            printPdf(decode_pdf);

        }
        catch (CFPrintException e)
        {
            // just rethrow - don't wrap
            throw e;
        }
        catch (Exception e)
        {
            throw new CFPrintPDFException(e);
        }
        finally
        {
            if (decode_pdf != null)
                decode_pdf.closePdfFile();
        }
    }

    private boolean isAllowPrintHigh(PdfDecoder decode_pdf)
    {
        // if the document isn't secured with a password, anyone can print it
        if (!decode_pdf.isEncrypted())
            return true;

        // the document is secured but we got the owner password - anything is allowed
        int pdfFlag = decode_pdf.getIO().getObjectReader().getPDFflag(PDFflags.VALID_PASSWORD_SUPPLIED);
        if (pdfFlag == PDFflags.VALID_OWNER_PASSWORD)
            return true;

        // The document is secured but we got the user password.  Check the permission.
        if (pdfFlag == PDFflags.VALID_USER_PASSWORD)
        {
            // high printing only - [(4 + 2048) for both high and low, 4 for just low]
            int pdfPermissionFlag = decode_pdf.getIO().getObjectReader().getPDFflag(PDFflags.USER_ACCESS_PERMISSIONS);
            if (pdfPermissionFlag != -1 && ((pdfPermissionFlag & 2048) == 2048))
                return true;
        }

        return false;
    }

    private void printPdf(PdfDecoder decode_pdf)
            throws java.awt.print.PrinterException, javax.print.PrintException,
            org.jpedal.exception.PdfException, CFPrintRequestAttributeException
    {
        // Get the print service for the named or default printer.
        PrintService printService = getPrintService();

        // Now that we have a real printer, verify that the media, if any, is valid for the printer and 
        // attributeSet requested.
        setMedia(printService, this.media);

        // If fidelity is true, check right now if there are any attributes which aren't supported.  There
        // is no need to proceed. 
        checkAttributesForFidelity(printService);

        // Make sure the job name is set if the user didn't specify it.
        String jobName = setJobNameAttribute(printService);

        DocPrintJob docPrintJob = printService.createPrintJob();

        // For CF8, we won't print annotations.  It's too late to add this attribute to the command.
        //decode_pdf.showAnnotations = false;
        Map map=new HashMap();
        //this would stop all subtype popups
        map.put(JPedalSettings.IGNORE_FORMS_ON_PRINT, new int[]{PdfDictionary.Popup, PdfDictionary.Text});
        PdfDecoder.modifyJPedalParameters(map);

        // Setup the paper and associate it with the PageFormat which the Pageable interface will use 
        // for each page.   Tell JPedal the format of the page to use for it's pageable interface.
        PageFormat myPageFormat = setPageFormat(printService);
        decode_pdf.setPageFormat(myPageFormat);

        // Tell JPedal what page range to print.
        PageRanges pageRange = getPageRange(decode_pdf.getPageCount(), pdfAttributes);
        decode_pdf.setPagePrintRange(pageRange);

        // Tell JPedal about pdf specific print attributes.
        // Auto-rotate and center, page scaling, page subset, reverse page attributes and use pdf paper size.
        decode_pdf.setPrintAutoRotateAndCenter(pdfAttributes.isAutoRotateAndCenter());
        decode_pdf.setPrintPageScalingMode(pdfAttributes.getPageScaling());
        decode_pdf.setPrintPageMode(pdfAttributes.getPageMode());
        decode_pdf.setUsePDFPaperSize(pdfAttributes.isUsePdfPageSize());

        // If debug enabled with command line switch enable JPedal debugging as well.
        //These 2 methods have been removed in latest jpedal build
       /* if (isDebug)
        {
            PdfDecoder.setDebugPrint(true);
            decode_pdf.showImageableArea();
        }*/

        // Monitor print job events   
        
        //Start-kumar
        //We use a PrintJob Listener not Adapter so not sure about this code
        PrintJobWatcher pjDone = new PrintJobWatcher(docPrintJob, jobName, decode_pdf.getNumberOfPages());
        //end-kumar
        
        // Print it
        this.attributeSet.add(pageRange);
        SimpleDoc doc;
        try
        {
            final PdfBook pdfBook = new PdfBook(decode_pdf, printService, this.attributeSet);
            pdfBook.setChooseSourceByPdfPageSize(pdfAttributes.isUsePdfPageSize());
            doc = new SimpleDoc(pdfBook, DocFlavor.SERVICE_FORMATTED.PAGEABLE, null);
        }
        catch (Throwable e)
        {
            doc = new SimpleDoc(decode_pdf, DocFlavor.SERVICE_FORMATTED.PAGEABLE, null);
        }

        //Start-kumar
        //Second parameter can also be null
        docPrintJob.print(doc, this.attributeSet);
        //end-kumar
        
        // Wait for the print job to be done
        pjDone.waitForDone();
    }

    private PageFormat setPageFormat(PrintService printService)
            throws java.awt.print.PrinterException
    {
        // Set up the paper based on the default for the printer and optional user request attributes.
        Paper myPaper = getPaper(printService);

        PageFormat myPageFormat = new PageFormat();
        myPageFormat.setPaper(myPaper);

        // Optionally set the orientation.  
        OrientationRequested orientation = (OrientationRequested) attributeSet.get(OrientationRequested.class);
        if (orientation != null)
        {
            // CF ignores the orientation if autoRotateAndCenter is specified because JPedal does.
            if (!pdfAttributes.isAutoRotateAndCenter())
            {
                // JPedal only supports PORTRAIT. Use autoRotateAndCenter for landscape. 
                if (orientation == OrientationRequested.PORTRAIT)
                {
                    myPageFormat.setOrientation(PageFormat.PORTRAIT);
                }
                else if (orientation == OrientationRequested.LANDSCAPE ||
                        orientation == OrientationRequested.REVERSE_LANDSCAPE ||
                        orientation == OrientationRequested.REVERSE_LANDSCAPE)
                {
                    //myPageFormat.setOrientation(PageFormat.LANDSCAPE);
                    //myPageFormat.setOrientation(PageFormat.REVERSE_LANDSCAPE);
                    //myPageFormat.setOrientation(PageFormat.REVERSE_LANDSCAPE);
                    throw new CFPrintPDFLandscapeException(orientation.toString());
                }
            }
        }

        return myPageFormat;
    }

    private PageRanges getPageRange(int totalPages, PrintPDFAttributes pdfAttributes)
    {
        boolean evenOnly = pdfAttributes.isEvenPagesOnly();
        boolean oddOnly = pdfAttributes.isOddPagesOnly();

        // See if PageRanges was specified.
        PageRanges pageRange = (PageRanges) attributeSet.get(PageRanges.class);
        if (pageRange == null)
        {
            pageRange = new PageRanges(1, totalPages);
            // Print all pages.
            if (!evenOnly && !oddOnly)
                return pageRange;
        }

        // We'll handle odd/even since JPedal can't get it right.
        pdfAttributes.setPageSubset(PrintPDFAttributes.ALL_PAGES);

        StringBuffer pages = new StringBuffer("");

        int i = -1;
        while ((i = pageRange.next(i)) != -1 && i <= totalPages)
        {
            if (pageRange.contains(i))
            {
                boolean addPage = true;

                if (evenOnly && i % 2 != 0)
                {
                    addPage = false;
                }
                else if (oddOnly && i % 2 == 0)
                {
                    addPage = false;
                }

                if (addPage)
                {
                    if (pages.length() > 0)
                        pages.append(',');
                    pages.append(i);
                }
            }
        }

        // Rebuild the page range.  There has to be at least one page in the range. 
        PageRanges newPageRange = null;
        try
        {
            newPageRange = new PageRanges(pages.toString());
            pageRange = newPageRange;
        }
        catch (Exception e)
        {
            // Probably an empty page range after cutting range off at totalPages
            throw new CFPrintInvalidPageNumberException(pageRange.toString(), String.valueOf(totalPages));
        }

        // This is really important!!  The pageable interface determines what pages are to be
        // printed.  If this is in the attribute set it can cause real problems, at least with
        // the printer "Adobe PDF" if page 2 is specified, and only 1 page (which is page 2) is sent to 
        // the distiller since it seems to be waiting for two pages.
        attributeSet.remove(PageRanges.class);

        return pageRange;
    }

//    /**
//     * Orientation error for cfprint tag.
//     * Error: Can not use orientation attribute unless the autoRotateAndCenter attribute is set to false.
//     */
//    public class CFPrintPDFOrientationException extends CFPrintException
//    {
//        CFPrintPDFOrientationException()
//        {
//        }
//    }
//    
    
}
