/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.jcs.auxiliary.disk.indexed;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.io.Serializable;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import org.apache.commons.jcs.auxiliary.disk.indexed.IndexedDiskElementDescriptor;
import org.apache.commons.jcs.engine.behavior.IElementSerializer;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

class IndexedDisk {
    public static final byte HEADER_SIZE_BYTES = 4;
    private final IElementSerializer elementSerializer;
    private static final Log log = LogFactory.getLog(IndexedDisk.class);
    private final String filepath;
    private final FileChannel fc;

    public IndexedDisk(File file, IElementSerializer elementSerializer) throws FileNotFoundException {
        this.filepath = file.getAbsolutePath();
        this.elementSerializer = elementSerializer;
        RandomAccessFile raf = new RandomAccessFile(this.filepath, "rw");
        this.fc = raf.getChannel();
    }

    protected <T extends Serializable> T readObject(IndexedDiskElementDescriptor ded) throws IOException, ClassNotFoundException {
        String message = null;
        boolean corrupted = false;
        long fileLength = this.fc.size();
        if (ded.pos > fileLength) {
            corrupted = true;
            message = "Record " + ded + " starts past EOF.";
        } else {
            ByteBuffer datalength = ByteBuffer.allocate(4);
            this.fc.read(datalength, ded.pos);
            datalength.flip();
            int datalen = datalength.getInt();
            if (ded.len != datalen) {
                corrupted = true;
                message = "Record " + ded + " does not match data length on disk (" + datalen + ")";
            } else if (ded.pos + (long)ded.len > fileLength) {
                corrupted = true;
                message = "Record " + ded + " exceeds file length.";
            }
        }
        if (corrupted) {
            log.warn((Object)("\n The file is corrupt: \n " + message));
            throw new IOException("The File Is Corrupt, need to reset");
        }
        ByteBuffer data = ByteBuffer.allocate(ded.len);
        this.fc.read(data, ded.pos + 4L);
        data.flip();
        return (T)((Serializable)this.elementSerializer.deSerialize(data.array(), null));
    }

    protected void move(IndexedDiskElementDescriptor ded, long newPosition) throws IOException {
        int chunkSize;
        ByteBuffer datalength = ByteBuffer.allocate(4);
        this.fc.read(datalength, ded.pos);
        datalength.flip();
        int length = datalength.getInt();
        if (length != ded.len) {
            throw new IOException("Mismatched memory and disk length (" + length + ") for " + ded);
        }
        long readPos = ded.pos;
        long writePos = newPosition;
        ByteBuffer buffer = ByteBuffer.allocate(16384);
        for (int remaining = 4 + length; remaining > 0; remaining -= chunkSize) {
            chunkSize = Math.min(remaining, buffer.capacity());
            buffer.limit(chunkSize);
            this.fc.read(buffer, readPos);
            buffer.flip();
            this.fc.write(buffer, writePos);
            buffer.clear();
            writePos += (long)chunkSize;
            readPos += (long)chunkSize;
        }
        ded.pos = newPosition;
    }

    protected boolean write(IndexedDiskElementDescriptor ded, byte[] data) throws IOException {
        long pos = ded.pos;
        if (log.isTraceEnabled()) {
            log.trace((Object)("write> pos=" + pos));
            log.trace((Object)(this.fc + " -- data.length = " + data.length));
        }
        if (data.length != ded.len) {
            throw new IOException("Mismatched descriptor and data lengths");
        }
        ByteBuffer buffer = ByteBuffer.allocate(4 + data.length);
        buffer.putInt(data.length);
        buffer.put(data);
        buffer.flip();
        int written = this.fc.write(buffer, pos);
        return written == data.length;
    }

    protected boolean writeObject(Serializable obj, long pos) throws IOException {
        byte[] data = this.elementSerializer.serialize(obj);
        this.write(new IndexedDiskElementDescriptor(pos, data.length), data);
        return true;
    }

    protected long length() throws IOException {
        return this.fc.size();
    }

    protected void close() throws IOException {
        this.fc.close();
    }

    protected synchronized void reset() throws IOException {
        if (log.isDebugEnabled()) {
            log.debug((Object)("Resetting Indexed File [" + this.filepath + "]"));
        }
        this.fc.truncate(0L);
        this.fc.force(true);
    }

    protected void truncate(long length) throws IOException {
        if (log.isInfoEnabled()) {
            log.info((Object)("Truncating file [" + this.filepath + "] to " + length));
        }
        this.fc.truncate(length);
    }

    protected String getFilePath() {
        return this.filepath;
    }
}

